/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2006-2008  John Schneiderman <JohnMS@member.fsf.org>                    *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#include "war.h"
#include "warinterface.h"
#include "userprofiledatabase.h"
#include "card.h"
#include "kardsgterror.h"
#include "computerplayer.h"
#include "player.h"

#include <algorithm>
#include <QString>
#include <QFile>
#include <QTextStream>
#include <QTextCodec>

War::War(WarInterface *gameInterface, UserProfileDatabase &profileDatabase): GameBase(profileDatabase), m_rules(), m_warCards(), m_deck()
{
    m_inWar = false;
    m_pInterface = gameInterface;
    m_firstToPlay = 0;
}

War::~War()
{}

void War::startPlay()
{
    time_t seconds;
    int rnd;

    if ((m_players.size() > m_rules.maximumNumberOfPlayers()) || (m_players.size() < m_rules.minimumNumberOfPlayers()))
        throw KardsGTError("War", "startPlay", "Wrong number of players were added to the game.");
    m_pInterface->show();
    // Temporarily create a random dealer.
    time(&seconds);
    srand(seconds);
    rnd=(rand() % 100) + 1;
    if ( (rnd > 50) )
        m_dealerIndex=1;
    else
        m_dealerIndex=0;
    deal();
}

bool War::save(const QString &filename)
{
    QFile file(filename);
    QTextStream out(&file);

    if (! file.open(QIODevice::WriteOnly))
        return false;
    out.setCodec(QTextCodec::codecForName("UTF-8"));
    out << "Game: " << "war" << endl;
    // Begin GameBase Save
    saveGameBase(out);
    // End GameBase Save
    out << "Deck: " << m_deck << endl;
    out << "InWar: " << m_inWar << endl;
    out << "WarCards: " << m_warCards << endl;
    out << "FirstToPlay: " << m_firstToPlay << endl;
    if (file.error() != QFile::NoError)
    {
        file.close();
        return false;
    }
    else
    {
        file.close();
        return true;
    }
}

bool War::load(const QString &filename)
{
    QFile file(filename);
    QTextStream in(&file);
    QString label;
    int intToBool;

    if (! file.open(QIODevice::ReadOnly))
    {
        file.close();
        return false;
    }
    in.setCodec(QTextCodec::codecForName("UTF-8"));
    in >> label >> label;
    if (label != "war")
        return false;
    // Begin GameBase Load
    loadGameBase(in);
    // End GameBase Load
    in >> label >> m_deck;
    in >> label >> intToBool;
    if (intToBool == 1)
        m_inWar = true;
    else
        m_inWar = false;
    in >> label >> m_warCards;
    in >> label >> m_firstToPlay;
    file.close();
    m_pInterface->updateTable();
    return true;
}

int War::minimumPlayers() const
{
    return m_rules.minimumNumberOfPlayers();
}

int War::maximumPlayers() const
{
    return m_rules.maximumNumberOfPlayers();
}

bool War::cardPlayed(const Card &card)
{

    m_playSequence.addCard(card);
    m_players.currentPlayer().hand().removeCard(card);
    m_pInterface->updateTable();
    // Check for winner or pair
    if ((m_playSequence.size() % 2) == 0)
    {
        int size = m_playSequence.size();

        if (m_playSequence[size - 1].rank() == m_playSequence[size - 2].rank())
            handleWar();
        else if (m_playSequence[size - 1].rank() > m_playSequence[size - 2].rank())
            winCards(m_players.currentPlayer());
        else
            winCards(m_players.leftPlayer());
    }
    handleGameStatus();
    setupNextPlayer();
    m_pInterface->updateTable();
    return false;
}

void War::setupNextPlayer()
{
    if (! m_rules.isGameOver(m_players))
    {
        m_players.currentPlayer().setTurn(false);
        m_players.nextNonEmptyPlayer();
        m_players.currentPlayer().setTurn(true);
    }
    else
        for (int i=0; i < m_players.size(); ++i)
            m_players[i].setTurn(false);

}

void War::winCards(Player &player)
{
    for (int i = 0; i < m_players.size(); ++i)
        m_players[i].setTurn(false);
    if (m_inWar)
    {
        player.incrementGamesWon();
        if ( m_warCards.size() > 6 )
            player.incrementGamesWon();
        if ( m_warCards.size() > 9 )
            player.incrementGamesWon();
        m_inWar = false;
        m_pInterface->displayMessage(player, QString("I won the war."));
        m_pInterface->showWarCards(player, m_warCards);
    }
    random_shuffle(m_playSequence.begin(), m_playSequence.end());
    for (int i = 0; i < m_playSequence.size(); ++i)
        player.hand().addCard(m_playSequence[i]);
    random_shuffle(m_warCards.begin(), m_warCards.end());
    for (int i = 0; i < m_warCards.size(); ++i)
        player.hand().addCard(m_warCards[i]);
    m_warCards.clear();
    m_playSequence.clear();
}

int War::handleGameStatus()
{
    if (m_rules.isGameOver(m_players))
    {
        for (int i = 0; i < m_players.size(); ++i)
            m_players[i].setTurn(false);
        m_pInterface->updateTable();
        m_pInterface->gameWon();
    }
    return -1;
}

void War::deal()
{
    for (int i = 0; i < m_players.size(); ++i)
        m_players[i].hand().clear();
    nextDealer();
    m_deck.shuffle();
    for (int i = 0; i < m_rules.numberOfCardsToDeal(); ++i)
        for (int j = 0; j < m_players.size(); ++j)
            m_players.nextPlayer().hand().addCard(m_deck.dealCard());
    handleGameStatus();
    setupNextPlayer();
    if (m_players.currentPlayer() == m_players[0])
        m_firstToPlay = 0;
    else
        m_firstToPlay = 1;
    m_pInterface->updateTable();
}

void War::handleWar()
{
    m_players.currentPlayer().setTurn(false);
    m_inWar = true;
    m_pInterface->promptMessage("War!", "War has been declared!");
    discardToWarPile(m_players.currentPlayer());
    discardToWarPile(m_players.rightPlayer());
}

void War::discardToWarPile(Player &player)
{
    for (int i = 0; i < WarRules::CARDS_TO_WAR_PILE; ++i)
    {
        m_warCards.addCard(player.hand().front());
        player.hand().removeCard(player.hand().front());
    }
}
