/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2006-2008  John Schneiderman <JohnMS@member.fsf.org>                    *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#include "warinterface.h"
#include "userprofiledatabase.h"
#include "kardpile.h"
#include "kardplayer.h"
#include "kard.h"
#include "warplayer.h"
#include "kardsgterror.h"
#include "kardsequence.h"
#include "kardplaysequence.h"
#include "kardmessagedisplay.h"

#include <QMessageBox>
#include <QLCDNumber>
#include <QLabel>
#include <QTimerEvent>
#include <QColor>
#include <QPalette>

WarInterface::WarInterface(UserProfileDatabase &profileDatabase, QWidget *parent): QWidget(parent), War(this, profileDatabase)
{
    Ui::WarInterface::setupUi(this);
    m_pUserProfile = &profileDatabase;

    //Set the colours for our game
    setBackgroundRole(QPalette::Background);
    setAutoFillBackground (true);
    QPalette pal(this->palette());
    pal.setColor(QPalette::Background, Qt::darkGreen);
    this->setPalette(pal);

    // Start the timer check for the non-human players
    m_computerTimerId=startTimer(COMPUTER_PLAYER_TIME);
    m_clearingDelayId=-1; // Don't want to waste time clearing when we've just started.
    connect(player1Pile, SIGNAL(kardSelected(Kard &)), this, SLOT(player1CardPlayed(Kard &)));
    connect(player2Pile, SIGNAL(kardSelected(Kard &)), this, SLOT(player2CardPlayed(Kard &)));
}

WarInterface::~WarInterface()
{}

void WarInterface::updateTable()
{
    // Set the card back images
    player1Pile->setCardBackImage(m_pUserProfile->cardBackImageFilename());
    player2Pile->setCardBackImage(m_pUserProfile->cardBackImageFilename());
    warPile->setCardBackImage(m_pUserProfile->cardBackImageFilename());
    playingField->setCardBackImage(m_pUserProfile->cardBackImageFilename());

    // Set the card front path
    player1Pile->setCardFrontImagePath(m_pUserProfile->pathToCardFrontImages());
    player2Pile->setCardFrontImagePath(m_pUserProfile->pathToCardFrontImages());
    warPile->setCardFrontImagePath(m_pUserProfile->pathToCardFrontImages());
    playingField->setCardFrontImagePath(m_pUserProfile->pathToCardFrontImages());

    // Update player hands
    player1Pile->setCardPile(m_players[0].hand());
    player2Pile->setCardPile(m_players[1].hand());
    warPile->setCardPile(m_warCards);
    if (m_playSequence.isEmpty()) // The play sequence must have just been cleared, that means we need a delay in the clearing to allow the human player the chance to see what was played.
    {
        if (m_clearingDelayId == 0) // Make sure we only set off the timer once.
            m_clearingDelayId=startTimer(CLEARING_DELAY_TIME);
    }
    else
        playingField->setCardSequence(m_playSequence, m_firstToPlay);

    // Update table info
    // Player 1
    player1->setPlayerImage(m_players[0].name().toLower() + ".png");
    player1->setDealer(m_players[0].isDealer());
    player1->setTurn(m_players[0].isTurn());
    player1->setSkillLevel(m_players[0].skillLevel());
    player1Wins->display(m_players[0].gamesWon());
    player1Soldiers->display(m_players[0].hand().size());
    // Player 2
    player2->setPlayerImage(m_pUserProfile->userMoodImage("NeutralMood"));
    player2->setDealer(m_players[1].isDealer());
    player2->setTurn(m_players[1].isTurn());
    player2->setSkillLevel(m_players[1].skillLevel());
    player2Wins->display(m_players[1].gamesWon());
    player2Soldiers->display(m_players[1].hand().size());
}

void WarInterface::gameWon()
{
    if ((m_players[0].gamesWon() == m_rules.winningGameScore()) || m_players[1].hand().isEmpty())
    {
        promptMessage("Game Over", QString(m_players[0].name() + " won.\n" + "Final Scores: " + m_players[0].name() + " %1, " + m_players[1].name() + " %2").arg(m_players[0].gamesWon()).arg(m_players[1].gamesWon()));
        m_pProfileDatabase->setGameStatistics("war", false);
    }
    else if ((m_players[1].gamesWon() == m_rules.winningGameScore()) || m_players[0].hand().isEmpty())
    {
        promptMessage("Game Over", QString(m_players[1].name() + " won.\n" + "Final Scores: " + m_players[0].name() + " %1, " + m_players[1].name() + " %2").arg(m_players[0].gamesWon()).arg(m_players[1].gamesWon()));
        m_pProfileDatabase->setGameStatistics("war", true);
    }
    else
        throw KardsGTError("WarInterface", "gameWon", "No one won?");
}

void WarInterface::displayMessage(const Player &player, const QString &message)
{
    if (m_players[0].name() == player.name())
        player1->setCaption(message);
    else if (m_players[1].name() == player.name())
        player2->setCaption(message);
    updateTable();
}

void WarInterface::promptMessage(const QString &caption, const QString &message)
{
    QMessageBox::information(this, caption, message);
    updateTable();
}

void WarInterface::showWarCards(const Player &player,  const CardSequence &cards)
{
    vector<CardSequence> warCards;
    vector<QString> message;

    warCards.push_back(cards);
    message.push_back(QString(player.name() + "'s soldiers won."));
    KardMessageDisplay display("War over!", warCards, message, m_pUserProfile->pathToCardFrontImages(), m_pUserProfile->cardBackImageFilename(), this);
    display.exec();
}

void WarInterface::player1CardPlayed(Kard &kard)
{
    // Make sure it's their turn.
    if (! m_players[0].isTurn())
    {
        promptMessage("Warning", m_players[0].name() + ", it is not your turn.");
        kard.setSelected(false);
        return;
    }

    // Try to play the card.
    if (kard.isSelected())
        if (m_rules.isLegalPlay(m_playSequence, kard.card(), m_players[0]))
        {
            cardPlayed(kard.card());
            updateTable();
        }
}

void WarInterface::player2CardPlayed(Kard &kard)
{
    // Make sure it's their turn.
    if (! m_players[1].isTurn())
    {
        promptMessage("Warning", m_players[1].name() + ", it is not your turn.");
        kard.setSelected(false);
        return;
    }

    // Try to play the card.
    if (kard.isSelected())
        if (m_rules.isLegalPlay(m_playSequence, kard.card(), m_players[1]))
        {
            cardPlayed(kard.card());
            updateTable();
        }
}

void WarInterface::timerEvent(QTimerEvent *event)
{
    if (m_players.isEmpty()) //Prevent any player timers from going off when we have no players
    {
        QWidget::timerEvent(event);
        return;
    }
    if (m_players[0].isTurn() && (event->timerId() == m_computerTimerId))
        computerPlay();
    else if (event->timerId() == m_clearingDelayId)
    {
        killTimer(m_clearingDelayId); // Stop our clearing timer delay
        m_clearingDelayId = 0; // Reset our id
        updateTable();
    }
    else
        QWidget::timerEvent(event);
}

void WarInterface::computerPlay()
{
    Card card;
    WarPlayer *compPlayer=static_cast<WarPlayer *>(&m_players[0]);

    card=compPlayer->playCard(m_playSequence);
    if (compPlayer->hand().front() == card)
    {
        player1Pile->selectTopCard();
        return;
    }
    else
        throw KardsGTError("CribbageInterface", "computerPlay", "Failed to find the card the computer selected!");
}
