/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2006-2008  John Schneiderman <JohnMS@member.fsf.org>                    *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#ifndef AIBASE_H
#define AIBASE_H

#include "player.h"

class QString;
class CardSequence;
class RuleBase;

/**
 * This is the abstract interface for the way a computer player would play a card in a game.
 * This can be thought of as the "brain" of a player, it holds all the knowledge of each game the AI knows how to play.
 *
 * @author John Schneiderman
 */
class AIBase
{
public:
    /**
     * Default constructor.
     */
    AIBase();
    /**
     * Default destructor.
     */
    virtual ~AIBase();
    /**
     * An accessor to the name of our AI player.
     * @return the name of our AI.
     */
    virtual QString name() const=0;
    /**
     * This selects a number of cards based on the game being played and the strategy the AI uses.
     * @param game is the name of the game or game phase the AI player is currently in. This information should be used to determine what the player needs to do.
     * @param playSequence is the current play sequence of cards in the game.
     * @param rules is the rules for the current game.
     * @param hand is the cards in the AI players hand.
     * @param specialInformation is the game state information normally known to a human player.
     * @return the cards the player selects.
     */
    virtual CardSequence whichCards(const QString &game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand, const GameInformation &specialInformation) const=0;
    /**
     * This gives a decision on the game being played and the strategy the AI uses.
     * @param game is the name of the game or game phase the AI player is currently in. This information should be used to determine what the player needs to do.
     * @param playSequence is the current play sequence of cards in the game.
     * @param rules is the rules for the current game.
     * @param hand is the cards in the AI players hand.
     * @param specialInformation is the game state information normally known to a human player.
     * @param score is the players current score.
     * @return the legal decision the players wishes to make, usually specified in the rules of the game.
     */
    virtual int decision(const QString &game, const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand, const GameInformation &specialInformation, int score) const=0;
    /**
     * This gives the skill level for the AI Player for a particular game.
     * @param game is the game we want to know how well the AI player can play.
     * @return is the skill level of the player based on the SkillLevel in @ref Player .
     */
    virtual Player::SkillLevel skillLevel(const QString &game) const=0;
};
#endif
