/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2005-2008  John Schneiderman <JohnMS@member.fsf.org>                    *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#ifndef CARDSEQUENCE_H
#define CARDSEQUENCE_H

#include "card.h"

#include <deque>
using std::deque;
#include <iterator>
using std::iterator;
#include <iostream>
using std::iostream;

class QTextStream;
class QDebug;

/**
 * This is a sequence of cards. This is generally used to hold a player's hand or the sequence of cards played.
 *
 * @author John Schneiderman
 */
class CardSequence
{
public:
    /// @param iterator is an alias to deque<Card>::iterator
    typedef deque<Card>::iterator iterator;
    /// @param const_iterator is an alias to deque<Card>::const_iterator
    typedef deque<Card>::const_iterator const_iterator;

    /**
     * This is the default constructor.
     */
    CardSequence();
    /**
     * This is a copy constructor for a single card sequence.
     */
    CardSequence(const Card &card);
    /**
     * This determines if the current sequence has no cards in it.
     * @return true if the sequence is empty or has only invalid cards, false elsewise.
     */
    bool isEmpty() const;
    /**
     * This determines if the sequence has a specific card in it.
     * @param card is the card to check to see if it's in the sequence.
     * @return true if card is in the sequence, false elsewise.
     */
    bool hasCard(const Card &card) const;
    /**
     * This determines if the sequence has a specific suit in it.
     * @param suit is the suit to search for.
     * @return true if the suit is found in the sequence, false elsewise.
     */
    bool hasSuit(const Card::Suit &suit) const;
    /**
     * This determines if the sequence has a specific rank in it.
     * @param rank is the rank to search for.
     * @return true if the rank is found in the sequence, false elsewise.
     */
    bool hasRank(const Card::Rank &rank) const;
    /**
     * This is used to add a card to the end of a players sequence.
     * @param card is the card to add to a players sequence.
     */
    void addCard(const Card &card);
    /**
     * This is used to add a card to the beginning of a players sequence.
     * @param card is the card to add to a players sequence.
     */
    void addCardToFront(const Card &card);
    /**
     * This is used to remove the first instance of a card from a players sequence.
     * @param card is the card to remove from a players sequence.
     * @return true if the card was successful in its attempt to remove the card, false elsewise.
     */
    bool removeCard(const Card &card);
    /**
     * Gives the current number of cards in the sequence.
     * @return the number of cards.
     */
    int size() const;
    /**
     * Random accessor for the card sequence.
     * @param index is the card to get from the sequence.
     * @throw KardsGTError if index is out of range.
     * @return the card requested.
     */
    const Card& operator[](int index) const;
    /**
     * Random accessor for the sequence.
     * @param index is the card to get from the sequence.
     * @throw KardsGTError if index is out of range.
     * @return a reference to the card requested.
     */
    Card& operator[](int index);
    /**
     * Equivalent operator.
     * @note The card sequence order does matter in determining equivalence.
     * @param compare is the card sequence to compare.
     * @return true if the sequence of cards in *this is the same as the sequence of cards in compare.
     */
    bool operator==(const CardSequence &compare) const;
    /**
     * Less than operator.
     * @note We only compare the sizes of the card sequences.
     * @param compare is the card sequence to compare.
     * @return true if the sequence of cards in *this is less than the sequence of cards in compare.
     */
    bool operator<(const CardSequence &compare) const;
    /**
     * This creates a combined sequence based on *this and a passed in sequence.
     * @param rhs is the card sequence to combine with *this.
     * @return *this combined with rhs having rhs at the end.
     */
    CardSequence operator+(const CardSequence &rhs) const;
    /**
     * This appends a card sequence to the end of *this.
     * @param rhs is the card sequence to add to the end of *this.
     * @return *this.
     */
    CardSequence & operator+=(const CardSequence &rhs);
    /**
     * This removes a sequence from *this.
     * @param rhs is the sequence to remove from *this.
     * @return the result of remove rhs from *this.
     */
    CardSequence operator-(const CardSequence &rhs) const;
    /**
     * This removes and assigns the sequence from *this.
     * @param rhs is the sequence to remove from *this.
     * @return *this.
     */
    CardSequence & operator-=(const CardSequence &rhs);
    /**
     * Sorts all the cards first by suit then by rank.
     */
    void sortBySuit();
    /**
     * Removes all of the cards from the sequence.
     */
    void clear();
    /**
     * Gives the first card in the sequence.
     * @return the first card.
     */
    const Card& front() const;
    /**
     * Gives the first card in the sequence.
     * @return the first card.
     */
    Card& front();
    /**
     * Gives the last card in the sequence.
     * @return the last card.
     */
    const Card& back() const;
    /**
     * Gives the last card in the sequence.
     * @return the last card.
     */
    Card& back();
    /**
     * Gives an iterator to the first card in the sequence.
     * @return gives an iterator to the first card.
     */
    iterator begin();
    /**
     * Gives a constant iterator to the first card in the sequence.
     * @return gives an const_iterator to the first card.
     */
    const_iterator begin() const;
    /**
     * Gives an iterator to the end of the cards in the sequence.
     * @return gives an iterator to the end of the cards.
     */
    iterator end();
    /**
     * Gives a constant iterator to the end of the cards in the sequence.
     * @return gives an const_iterator to the end of the cards.
     */
    const_iterator end() const;
    /**
     * A stream insertion operator that writes out a card sequence.
     * @param out this is the buffer to write the card sequence to.
     * @param sequence is the card sequence to write out.
     * @return a reference to out.
     * @note Each of the cards are separated by white space.
     */
    friend ostream& operator<<(ostream &out, const CardSequence &sequence);
    /**
     * A stream extraction operator that will read in a card sequence.
     * @param in this is the buffer to read the card sequence from.
     * @param sequence is the card sequence to read into.
     * @return a reference to in.
     * @note Each card to read in is assumed to be separated by white space.
     */
    friend istream& operator>>(istream &in, CardSequence &sequence);
    /**
     * A stream insertion operator that will writes out a card sequence.
     * @param out this is the buffer to write the card sequence to.
     * @param sequence is the card sequence to write out.
     * @return a reference to out.
     * @note Each of the cards are separated by white space, and the last card is followed by an empty card.
     */
    friend QTextStream& operator<<(QTextStream &out, const CardSequence &sequence);
    /**
     * A stream insertion operator that will writes out a card sequence.
     * @param out this is the buffer to write the card sequence to.
     * @param sequence is the card sequence to write out.
     * @return a reference to out.
     * @note Each of the cards are separated by white space, and the last card is followed by an empty card.
     */
    friend QDebug& operator<<(QDebug &out, const CardSequence &sequence);
    /**
     * A stream extraction operator that will read in a card sequence.
     * @param in this is the buffer to read the card sequence from.
     * @param sequence is the card sequence to read into.
     * @return  a reference to in.
     * @note Each card to read in is assumed to be separated by white space, and continues to read until an empty card is read in.
     */
    friend QTextStream& operator>>(QTextStream &in, CardSequence &sequence);

private:
    /// @param m_sequence is the sequence of cards.
    deque<Card> m_sequence;
};
#endif
