/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2006-2008  John Schneiderman <JohnMS@member.fsf.org>                    *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#ifndef KARDPILE_H_
#define KARDPILE_H_

#include <QWidget>
#include <QDragEnterEvent>
#include <QDropEvent>
#include <QPaintEvent>
#include <vector>
using std::vector;

class CardPile;
class QPainter;
class Kard;
class Card;
class QStackedLayout;

/**
 * This displays a pile of cards.
 *
 * @author John Schneiderman
 */
class KardPile: public QWidget
{
    Q_OBJECT
    Q_PROPERTY(bool m_topCardUp READ isTopCardUp WRITE setTopCardUp)

public:
    /**
     * General constructor for our widget.
     * @param parent is parent of the KardPile widget.
     */
    KardPile(QWidget *parent=0);
    /**
     * General constructor for our widget.
     * @param pile is the pile to display
     * @param parent is parent of the KardPile widget.
     */
    KardPile(CardPile pile, QWidget *parent=0);
    /**
     * General destructor for our widget.
     */
    ~KardPile();
    /**
     * Provides the size hint for our widget.
     * @return the preferred size of our widget.
     */
    QSize sizeHint() const;
    /**
     * Tells us if our widget is displaying the top card face up.
     * @return true if we are displaying the top card face up, false if we are not.
     */
    bool isTopCardUp() const;
    /**
     * This is an accessor for our card pile.
     * @return our card pile.
     */
    const CardPile& cardPile() const;
    /**
     * Sets our widget to display the top card face up or not.
     * @param topUp true if we want to have our top card face up, false elsewise.
     */
    void setTopCardUp(bool topUp);
    /**
     * This will set our displaying card pile.
     * @param cardPile is the pile of cards to set our widget to.
     */
    void setCardPile(const CardPile &cardPile);
    /**
     * This clears the kard pile.
     */
    void clear();
    /**
     * This selects the top card.
     */
    void selectTopCard();
    /**
     * Adds a card to the bottom of the card pile.
     * @param card is the card to be inserted into the bottom of the pile.
     */
    void addCardToBottom(const Card &card);
    /**
     * This will select a specific card in our kard pile.
     * @param card is the card we want to select.
     * @throw KardsGTError if card is not the top card in the pile.
     */
    void setSelected(const Card &card);
    /**
     * Sets the image to use for the back of the cards.
     * @param backImage is the file-name for the image on the back of the kards.
     */
    void setCardBackImage(const QString &backImage);
    /**
     * Sets the path to the directory that contains the images for the front of our kards.
     * @param imagePath is the path to the images.
     */
    void setCardFrontImagePath(const QString &imagePath);
    /**
     * This will remove the top card from the card pile.
     */
    void removeCard();

signals:
    /**
     * This is a signal that notifies a slot that a card has been selected.
     * @param kard is the kard that was selected.
     */
    void kardSelected(Kard &kard);

protected:
    /**
     * This draws our widget.
     * @param event is the event that caused our widget to be drawn.
     */
    virtual void paintEvent(QPaintEvent *event);
    /**
     * These are the types of objects we can accept for dropping.
     * @param event is the generating event.
     */
    virtual void dragEnterEvent(QDragEnterEvent *event);
    /**
     * This handles when an object is dropped into us.
     * @param event is the generating event.
     */
    virtual void dropEvent(QDropEvent *event);

private:
    /**
     * The preferred size of the card.
     * @param PREFERRED_WIDTH is the preferred width of the card.
     * @param PREFERRED_HEIGHT is the preferred height of the card.
     */
    enum PreferredSize
    {
        PREFERRED_WIDTH=70,
        PREFERRED_HEIGHT=90
    };
    /// @param m_topCardUp is true if we are displaying the front of the top card and false if ware displaying the back of the card.
    bool m_topCardUp;
    /// @param m_pPile is the pile of cards we're representing.
    CardPile *m_pCardPile;
    /// @param m_kards used in the layout of the kard pile.
    vector<Kard *> m_kards;
    /// @param m_pLayout is the layout to display the Kards in.
    QStackedLayout *m_pLayout;
    /// @param m_cardImagePath is path to the directory that contains the images for the front of our kards.
    QString m_cardImagePath;
    /// @param m_cardBackImage is the file-name for the back of our kards.
    QString m_cardBackImage;

    /**
     * This draws the Kards in a pseudo-3-D way.
     * @param painter is where we're going to draw the cards to.
     */
    void draw(QPainter &painter);
    /**
     * This will add a Kard to the bottom of our card pile.
     * @param kard is the Kard to add to our card pile.
     */
    void addKard(Kard &kard);
    /**
     * This will remove a Kard from our card pile.
     * @param kard is the kard to remove from the top of the card pile.
     * @throw KardsGTError if kard is not the top card in the pile.
     */
    void removeKard(Kard &kard);
};
#endif
