/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2005-2008  John Schneiderman <JohnMS@member.fsf.org>                    *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#include "kardsequence.h"
#include "kard.h"
#include "kardsgterror.h"
#include "card.h"
#include "kardpile.h"

#include <algorithm>
#include <vector>
using std::vector;
using std::iterator;

#include <QLayout>
#include <QPainter>
#include <QEvent>
#include <QHBoxLayout>
#include <QDropEvent>
#include <QPaintEvent>
#include <QPixmap>
#include <QMouseEvent>
#include <QVBoxLayout>
#include <QDragEnterEvent>
#include <QCursor>
#include <QMenu>
#include <QAction>

KardSequence::KardSequence(QWidget *parent,  bool vertical):QWidget(parent), m_cardSequence(), m_kards()
{
    time_t seconds;

    time(&seconds);
    srand(seconds);
    setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    m_vertical=vertical;
    m_pLayout = m_pRightLayout = m_pLeftLayout = 0;
    setupLayout();
    m_faceUp = false;
    m_sortPopup = true;
    m_cardImagePath = "";
    m_cardBackImage = "";
    setAcceptDrops(true);
}

KardSequence::KardSequence(const CardSequence &sequence, QWidget *parent,  bool vertical):QWidget(parent), m_cardSequence(), m_kards()
{
    time_t seconds;

    time(&seconds);
    srand(seconds);
    setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    m_vertical=vertical;
    m_pLayout = m_pRightLayout = m_pLeftLayout = 0;
    setupLayout();
    m_faceUp = false;
    m_sortPopup = true;
    m_cardImagePath = "";
    m_cardBackImage = "";
    setCardSequence(sequence);
    setAcceptDrops(true);
}

KardSequence::~KardSequence()
{
    int size=static_cast<int>(m_kards.size());

    for (int i=0; i < size; ++i)
        delete m_kards[i];
    delete m_pLayout;
}

QSize KardSequence::sizeHint()
{
    return QSize(PREFERRED_WIDTH, PREFERRED_HEIGHT);
}

const CardSequence& KardSequence::cardSequence() const
{
    return m_cardSequence;
}

bool KardSequence::isFaceUp() const
{
    return m_faceUp;
}

bool KardSequence::isVertical() const
{
    return m_vertical;
}

int KardSequence::size() const
{
    return m_cardSequence.size();
}

void KardSequence::setFaceUp(bool faceUp)
{
    m_faceUp = faceUp;
    for (int index=0; index < m_cardSequence.size(); ++index)
        m_kards[index]->setFaceUp(m_faceUp);
    update();
    updateGeometry();
}

void KardSequence::setVertical(bool vertical)
{
    CardSequence temp=m_cardSequence;
    m_vertical = vertical;
    setupLayout();
    clear();
    setCardSequence(temp);
    update();
    updateGeometry();
}

void KardSequence::clear()
{
    int size=m_kards.size();

    for (int i=0; i < size; ++i)
        removeKard(*m_kards[0]);
    m_kards.clear();
    m_cardSequence.clear();
    update();
    updateGeometry();
}

void KardSequence::setCardSequence(const CardSequence &sequence)
{
    // Don't bother if we're already the same.
    if (m_cardSequence == sequence)
        return;

    //Set up the new cards
    clear();
    for (int i=0; i < sequence.size(); ++i)
        addCard(sequence[i]);
    update();
    updateGeometry();
}

void KardSequence::addKard(Kard &kard)
{
    addCard(kard.card());
}

void KardSequence::addCard(const Card &card)
{
    Kard *kard = new Kard(card, m_faceUp, this);

    m_cardSequence.addCard(card);
    if (m_vertical)
        if (m_kards.size() < VERTICAL_SPLIT)
            m_pLeftLayout->addWidget(kard);
        else
            m_pRightLayout->addWidget(kard);
    else
        if (m_kards.size() < HORIZONTAL_SPLIT)
            m_pLeftLayout->addWidget(kard);
        else
            m_pRightLayout->addWidget(kard);
    kard->setCardBack(m_cardBackImage);
    kard->setCardPath(m_cardImagePath);
    m_kards.push_back(kard);
    update();
    updateGeometry();
    connect(kard, SIGNAL(cardSelected(Kard &)), this, SIGNAL(kardSelected(Kard &)));
}

void KardSequence::removeKard(Kard &kard)
{
    removeCard(kard.card());
}

void KardSequence::removeCard(const Card &card)
{
    vector<Kard *>::iterator iter=m_kards.begin();
    int i=0;

    while (iter != m_kards.end())
    {
        if ((*iter)->card() == card)
        {
            (*iter)->hide();
            if (m_vertical)
                if (m_kards.size() < VERTICAL_SPLIT)
                    m_pLeftLayout->removeWidget(*iter);
                else
                {
                    m_pRightLayout->removeWidget(*iter);
                    m_pLeftLayout->removeWidget(*iter);
                }
            else
                if (m_kards.size() < HORIZONTAL_SPLIT)
                    m_pLeftLayout->removeWidget(*iter);
                else
                {
                    m_pRightLayout->removeWidget(*iter);
                    m_pLeftLayout->removeWidget(*iter);
                }
            m_cardSequence.removeCard(card);
            m_kards.erase(iter);
            update();
            updateGeometry();
            return;
        }
        i++, iter++;
    }
    throw KardsGTError(QString("KardSequence"), QString("removeCard"), "Failed to remove " + card.toString() + ".");
}

void KardSequence::paintEvent(QPaintEvent *)
{
    QPainter painter(this);

    draw(painter);
}

void KardSequence::draw(QPainter &painter)
{
    painter.drawRect(painter.window());
    for (int i=0; i < m_cardSequence.size(); ++i)
        m_kards[i]->show();
}

void KardSequence::setSelected(const Card &card)
{
    vector<Kard *>::iterator iter=m_kards.begin();

    while (iter != m_kards.end())
    {
        if ((*iter)->card() == card)
        {
            (*iter)->setSelected(true);
            break;
        }
        iter++;
    }
}

void KardSequence::dragEnterEvent(QDragEnterEvent *event)
{
    if (event && event->mimeData())
    {
        const QMimeData *mimeData = event->mimeData();
        if (mimeData->hasFormat("application/x-kardsgt.text.kard"))
            event->acceptProposedAction();
    }
}

void KardSequence::dropEvent(QDropEvent *event)
{
    Card card;

    if (event && event->mimeData())
    {
        QWidget *fromWidget = event->source();
        const QMimeData *mimeData = event->mimeData();

        if (fromWidget && (fromWidget != this) && fromWidget->inherits("KardSequence"))
        {
            card.setCard(QString(mimeData->data("application/x-kardsgt.text.kard")));
            static_cast<KardSequence *>(fromWidget)->setSelected(card);
        }
        else if (fromWidget && (fromWidget == this) && fromWidget->inherits("KardSequence"))
        {
            card.setCard(QString(mimeData->data("application/x-kardsgt.text.kard")));
            Kard kard(card, false);
            moveCard(card, event->pos());
            emit static_cast<KardSequence *>(fromWidget)->kardMoved(kard);
        }
        else if (fromWidget && (fromWidget!= this) && fromWidget->inherits("KardPile"))
        {
            card.setCard(QString(mimeData->data("application/x-kardsgt.text.kard")));
            static_cast<KardPile *>(fromWidget)->setSelected(card);
        }
    }
}

void KardSequence::moveCard(const Card &card, QPoint position)
{
    int size = m_kards.size();
    CardSequence sequence;
    bool inserted = false;

    for (int index=0; index < size; ++index)
    {
        // Find the index of the card to place the new card in front of.
        if ((! inserted) && (position.x() < m_kards[index]->pos().x()))
        {
            if (m_kards[index]->card() == card)
                continue;
            sequence.addCard(card);
            inserted = true;
            sequence.addCard(m_kards[index]->card());
        }
        else if (m_kards[index]->card() == card)
            continue;
        else
            sequence.addCard(m_kards[index]->card());
    }
    if (! inserted)
        sequence.addCard(card);
    setCardSequence(sequence);
}

void KardSequence::mousePressEvent(QMouseEvent *event)
{
    if ((m_kards.size() == 0) || (! m_sortPopup))
        return;

    // Arrange our kards
    if (event->button() == Qt::RightButton)
    {
        QMenu *sortMenu = new  QMenu(this);

        QAction *rankAction = sortMenu->addAction(tr("Sort by &rank"));
        QAction *suitAction = sortMenu->addAction(tr("Sort by &suit"));
        QAction *randAction = sortMenu->addAction(tr("Sort by r&andom"));
        connect(rankAction, SIGNAL(triggered()), this, SLOT(sortByRank()));
        connect(suitAction, SIGNAL(triggered()), this, SLOT(sortBySuit()));
        connect(randAction, SIGNAL(triggered()), this, SLOT(sortByRandom()));
        sortMenu->exec(QCursor::pos());
        delete sortMenu;
    }
}

void KardSequence::sortByRank()
{
    CardSequence sequence = m_cardSequence;

    sort(sequence.begin(), sequence.end());
    setCardSequence(sequence);
    emit kardMoved();
}

void KardSequence::sortBySuit()
{
    CardSequence sequence = m_cardSequence;

    sequence.sortBySuit();
    setCardSequence(sequence);
    emit kardMoved();
}

void KardSequence::sortByRandom()
{
    CardSequence sequence = m_cardSequence;

    random_shuffle(sequence.begin(), sequence.end());
    setCardSequence(sequence);
    emit kardMoved();
}

bool KardSequence::isSortPopup() const
{
    return m_sortPopup;
}

void KardSequence::setSortPopup(bool sortPopup)
{
    m_sortPopup = sortPopup;
}

void KardSequence::setCardBackImage(const QString &backImage)
{
    m_cardBackImage = backImage;
    for (int index=0, size=m_kards.size(); index < size; ++index)
        m_kards[index]->setCardBack(backImage);
    update();
    updateGeometry();
}

void KardSequence::setCardFrontImagePath(const QString &imagePath)
{
    m_cardImagePath = imagePath;
    for (int index=0, size=m_kards.size(); index < size; ++index)
        m_kards[index]->setCardPath(imagePath);
    update();
    updateGeometry();
}

void KardSequence::setupLayout()
{
    if (m_pLayout)
        delete m_pLayout;
    if (m_vertical)
    {
        m_pLayout=new QHBoxLayout(this);
        m_pLeftLayout=new QVBoxLayout();
        m_pRightLayout=new QVBoxLayout();
    }
    else
    {
        m_pLayout=new QVBoxLayout(this);
        m_pLeftLayout=new QHBoxLayout();
        m_pRightLayout=new QHBoxLayout();
    }
    m_pLayout->addLayout(m_pLeftLayout);
    m_pLayout->addLayout(m_pRightLayout);
}
