/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2005-2008  John Schneiderman <JohnMS@member.fsf.org>                    *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#include "cribbageinterface.h"
#include "cribbage.h"
#include "kardselection.h"
#include "cardsequence.h"
#include "cribbageplayer.h"
#include "kardmessagedisplay.h"
#include "cribbageboard.h"
#include "card.h"
#include "kard.h"
#include "kardsgterror.h"
#include "kardplayer.h"
#include "kardpile.h"
#include "kardsequence.h"
#include "kardplaysequence.h"
#include "userprofiledatabase.h"

#include <QLabel>
#include <QLCDNumber>
#include <QPixmap>
#include <QString>
#include <QTextEdit>
#include <QMessageBox>
#include <QTimerEvent>
#include <QColor>
#include <QPalette>

CribbageInterface::CribbageInterface(UserProfileDatabase &profileDatabase, QWidget* parent): QWidget(parent), Cribbage(this, profileDatabase)
{
    Ui_CribbageInterface::setupUi(this);
    m_pUserProfile = &profileDatabase;
    player2Cards->setFaceUp(true);
    playerDiscards->setFaceUp(true);
    playerDiscards->setNumberOfPlayers(2);
    cardDeck->setTopCardUp(true);

    //Set the colours for our game
    setBackgroundRole(QPalette::Background);
    setAutoFillBackground (true);
    QPalette pal(this->palette());
    pal.setColor(QPalette::Background, Qt::darkGreen);
    this->setPalette(pal);
    crib->setPalette(pal);
    lcdTxt->setPalette(pal);
    countDisplay->setPalette(pal);
    textLabelCrib->setPalette(pal);

    // Start the timer check for the non-human players
    m_computerTimerId=startTimer(COMPUTER_PLAYER_TIME);
    m_clearingDelayId=-1; // Don't want to waste time clearing when we've just started.

    // Connect the player's cards to the selected slots
    connect(player1Cards, SIGNAL(kardSelected(Kard &)), this, SLOT(player1CardPlayed(Kard &)));
    connect(player2Cards, SIGNAL(kardSelected(Kard &)), this, SLOT(player2CardPlayed(Kard &)));
    // Connect the player's cards to the moving slots
    connect(player1Cards, SIGNAL(kardMoved(Kard &)), this, SLOT(player1CardMoved()));
    connect(player2Cards, SIGNAL(kardMoved(Kard &)), this, SLOT(player2CardMoved()));
}

CribbageInterface::~CribbageInterface()
{}

void CribbageInterface::updateTable()
{
    // Set the card back images
    player1Cards->setCardBackImage(m_pUserProfile->cardBackImageFilename());
    player2Cards->setCardBackImage(m_pUserProfile->cardBackImageFilename());
    crib->setCardBackImage(m_pUserProfile->cardBackImageFilename());
    cardDeck->setCardBackImage(m_pUserProfile->cardBackImageFilename());
    playerDiscards->setCardBackImage(m_pUserProfile->cardBackImageFilename());

    // Set the card front path
    player1Cards->setCardFrontImagePath(m_pUserProfile->pathToCardFrontImages());
    player2Cards->setCardFrontImagePath(m_pUserProfile->pathToCardFrontImages());
    crib->setCardFrontImagePath(m_pUserProfile->pathToCardFrontImages());
    cardDeck->setCardFrontImagePath(m_pUserProfile->pathToCardFrontImages());
    playerDiscards->setCardFrontImagePath(m_pUserProfile->pathToCardFrontImages());

    //Update the scores.
    cribbageBoard->pegScore(Qt::red, m_players[FIRST_PLAYER].score());
    cribbageBoard->pegScore(Qt::blue, m_players[SECOND_PLAYER].score());
    countDisplay->display(m_roundPoints);

    // Update player hands
    player1Cards->setCardSequence(m_players[FIRST_PLAYER].hand());
    player2Cards->setCardSequence(m_players[SECOND_PLAYER].hand());
    if (m_playSequence.isEmpty()) // The play sequence must have just been cleared, that means we need a delay in the clearing to allow the human player the chance to see what was played.
    {
        if (m_clearingDelayId == 0) // Make sure we only set off the timer once.
            m_clearingDelayId=startTimer(CLEARING_DELAY_TIME);
    }
    else
        playerDiscards->setCardSequence(m_playSequence, m_playOrder);

    // Update table info
    crib->setCardSequence(m_crib);
    cardDeck->setCardPile(m_deck);
    kardPlayer1->setPlayerImage(m_players[FIRST_PLAYER].name().toLower() + ".png");
    kardPlayer2->setPlayerImage(m_pUserProfile->userMoodImage("NeutralMood"));
    kardPlayer1->setDealer(m_players[FIRST_PLAYER].isDealer());
    kardPlayer2->setDealer(m_players[SECOND_PLAYER].isDealer());
    kardPlayer1->setSkillLevel(m_players[FIRST_PLAYER].skillLevel());
    kardPlayer2->setSkillLevel(m_players[SECOND_PLAYER].skillLevel());
    kardPlayer1->setTurn(m_players[FIRST_PLAYER].isTurn());
    kardPlayer2->setTurn(m_players[SECOND_PLAYER].isTurn());
    cribbageBoard->setName(Qt::red, m_players[FIRST_PLAYER].name());
    cribbageBoard->setName(Qt::blue, m_players[SECOND_PLAYER].name());

    update();
    updateGeometry();
}

void CribbageInterface::player1CardPlayed(Kard &kard)
{
    // Make sure it's their turn.
    if (! m_players[FIRST_PLAYER].isTurn())
    {
        promptMessage("Warning", m_players[FIRST_PLAYER].name() + ", it is not your turn.");
        kard.setSelected(false);
        return;
    }

    // Try to play the card.
    if (! m_players[FIRST_PLAYER].hand().isEmpty() && (m_rules.isLegalPlay(m_playSequence, kard.card(), m_players[FIRST_PLAYER])))
    {
        cardPlayed(kard.card());
        updateTable();
    }
}

void CribbageInterface::player2CardPlayed(Kard &kard)
{
    // Make sure it's their turn.
    if (! m_players[SECOND_PLAYER].isTurn())
    {
        promptMessage("Warning", m_players[SECOND_PLAYER].name() + ", it is not your turn.");
        kard.setSelected(false);
        return;
    }

    // Try to play the card.
    if (! m_players[SECOND_PLAYER].hand().isEmpty() && (m_rules.isLegalPlay(m_playSequence, kard.card(), m_players[SECOND_PLAYER])))
    {
        cardPlayed(kard.card());
        updateTable();
    }
}

void CribbageInterface::player1CardMoved()
{
    m_players[FIRST_PLAYER].hand().clear();
    for (int i = 0, size = player1Cards->cardSequence().size(); i < size; ++i)
        m_players[FIRST_PLAYER].hand().addCard(player1Cards->cardSequence()[i]);
}

void CribbageInterface::player2CardMoved()
{
    m_players[SECOND_PLAYER].hand().clear();
    for (int i = 0, size = player2Cards->cardSequence().size(); i < size; ++i)
        m_players[SECOND_PLAYER].hand().addCard(player2Cards->cardSequence()[i]);
}

CardSequence CribbageInterface::requestCards(const QString &prompt, const Player &player, int numberOfCards)
{
    KardSelection *selectionInterface=new KardSelection(prompt, numberOfCards, player.hand(), m_pUserProfile->pathToCardFrontImages(), m_pUserProfile->cardBackImageFilename(), this);

    if (selectionInterface->exec())
    {
        CardSequence selection=selectionInterface->selection();
        delete selectionInterface;
        return selection;
    }
    else
        return requestCards(prompt, player, numberOfCards);
}

void CribbageInterface::displayMessage(const Player &player, const QString &message)
{
    if (m_players[FIRST_PLAYER].name() == player.name())
        kardPlayer1->setCaption(message);
    else if (m_players[SECOND_PLAYER].name() == player.name())
        kardPlayer2->setCaption(message);
    updateTable();
}

void CribbageInterface::promptMessage(const QString &caption, const QString &message)
{
    QMessageBox::information(this, caption, message);
    updateTable();
}

pair<int, int> CribbageInterface::previousScores() const
{
    pair<int, int> scores;

    scores.first=cribbageBoard->scores(Qt::red).second;
    scores.second=cribbageBoard->scores(Qt::blue).second;
    return scores;
}

void CribbageInterface::setPreviousScores(int redScore, int blueScore)
{
    cribbageBoard->setScores(Qt::red, pair<int, int>(m_players[FIRST_PLAYER].score(), redScore));
    cribbageBoard->setScores(Qt::blue, pair<int, int>(m_players[SECOND_PLAYER].score(), blueScore));
    updateTable();
}

void CribbageInterface::showStarterCard(bool show)
{
    cardDeck->setTopCardUp(show);
    update();
    updateGeometry();
}

void CribbageInterface::computerPlay()
{
    CribbagePlayer *compPlayer=static_cast<CribbagePlayer *>(&m_players[FIRST_PLAYER]);

    player1Cards->setSelected(compPlayer->playCard(m_playSequence));
}

void CribbageInterface::timerEvent(QTimerEvent *event)
{
    if (m_players.isEmpty()) //Prevent any player timers from going off when we have no players
    {
        QWidget::timerEvent(event);
        return;
    }
    if (m_players[FIRST_PLAYER].isTurn() && (event->timerId() == m_computerTimerId))
        computerPlay();
    else if (event->timerId() == m_clearingDelayId)
    {
        killTimer(m_clearingDelayId); // Stop our clearing timer delay
        m_clearingDelayId = 0; // Reset our id
        updateTable();
    }
    else
        QWidget::timerEvent(event);
}

void CribbageInterface::displayRoundSummary(const QString &caption, const vector<CardSequence> &displayHands, const vector<QString> &messages)
{
    KardMessageDisplay summary(caption, displayHands, messages, m_pUserProfile->pathToCardFrontImages(), m_pUserProfile->cardBackImageFilename(), this);

    summary.exec();
    updateTable();
}

void CribbageInterface::resetGame()
{
    cardDeck->clear();
    player1Cards->clear();
    player2Cards->clear();
    playerDiscards->clear();
    crib->clear();
    updateTable();
}

void CribbageInterface::updateStakes(int roundWinnerIndex)
{
    resetGame();
    cribbageBoard->reset();
    cribbageBoard->pegStakes(Qt::red, m_players[FIRST_PLAYER].gamesWon());
    cribbageBoard->pegStakes(Qt::blue, m_players[SECOND_PLAYER].gamesWon());
    // Give the player some skill points for winning the round
    if (roundWinnerIndex == 1)
        m_pProfileDatabase->setGameStatistics("cribbage", true);
    else if (roundWinnerIndex == 0)
        m_pProfileDatabase->setGameStatistics("cribbage", false);
    updateTable();
}

void CribbageInterface::gameWon()
{
    if (m_players[FIRST_PLAYER].gamesWon() == m_rules.winningGameScore())
    {
        promptMessage("Game Over", QString(m_players[FIRST_PLAYER].name() + " won.\n" + "Final Scores: " + m_players[FIRST_PLAYER].name() + " %1, " + m_players[SECOND_PLAYER].name() + " %2").arg(m_players[FIRST_PLAYER].gamesWon()).arg(m_players[SECOND_PLAYER].gamesWon()));
        m_pProfileDatabase->setGameStatistics("cribbage", false);
    }
    else if (m_players[SECOND_PLAYER].gamesWon() == m_rules.winningGameScore())
    {
        promptMessage("Game Over", QString(m_players[SECOND_PLAYER].name() + " won.\n" + "Final Scores: " + m_players[FIRST_PLAYER].name() + " %1, " + m_players[SECOND_PLAYER].name() + " %2").arg(m_players[FIRST_PLAYER].gamesWon()).arg(m_players[SECOND_PLAYER].gamesWon()));
        m_pProfileDatabase->setGameStatistics("cribbage", true);
    }
    else
        throw KardsGTError("CribbageInterface", "gameWon", "No one won?");
}
