/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2006-2008  John Schneiderman <JohnMS@member.fsf.org>                    *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#ifndef CRIBBAGEPEG_H
#define CRIBBAGEPEG_H

#include <QWidget>
#include <QColor>

/**
 * This is a widget to draw the pegs in a cribbage board.
 *
 * @author John Schneiderman
 */
class CribbagePeg: public QWidget
{
public:
    /**
     * This is the default constructor.
     * @param colour is the colour to make the peg.
     * @param position is the position to that the peg starts at.
     * @param leadPeg is to be true if this peg is a leading peg.
     * @param parent is the parent of this widget.
     */
    CribbagePeg(const QColor &colour, const QPoint &position, bool leadPeg, QWidget *parent=0);
    /**
     * Gives the current x-coordinate.
     * @note The x-position is referenced as a rectangle.
     * @return current upper left x-coordinate.
     */
    int x() const;
    /**
     * Gives the current y-coordinate.
     * @note The y-position is referenced as a rectangle.
     * @return current upper left y-coordinate.
     */
    int y() const;
    /**
     * Gives the current position
     * @note The position is referenced as a rectangle.
     * @return the upper left coordinates.
     */
    QPoint position() const;
    /**
     * Gives the original position as specified during the peg's creation.
     * @note The x-position is referenced as a rectangle.
     * @return the upper left coordinate of the original position.
     */
    QPoint startPosition() const;
    /**
     * Gives the colour of the pegs.
     * @return the colour of the pegs.
     */
    const QColor& colour() const;
    /**
     * Gives information about concerning if the peg is the current leading peg.
     * @return true if the peg is currently the leading peg, false elsewise.
     */
    bool isFront() const;
    /**
     * Gives the current score that has been pegged.
     * @return the current score.
     */
    int score() const;
    /**
     * Sets whether the peg is the current leading peg.
     * @param lead should be true if the peg is the leading peg, false elsewise.
     */
    void setLead(bool lead);
    /**
     * Sets the current pegged score.
     * @param score is the new score to peg.
     */
    void setScore(int score);
    /**
     * Moves the peg to a new position.
     * @param x is the upper left x-coordinate.
     * @param y is the upper left y-coordinate.
     */
    void setPosition(int x, int y);
    /**
     * Moves the peg to a new position.
     * @param point is new upper left coordinate.
     */
    void setPosition(const QPoint &point);
    /**
     * Draws the pegs.
     * @param painter is where to draw the pegs.
     */
    void draw(QPainter *painter);
    /**
     * Places the peg back at it's starting position.
     */
    void reset();

private:
    /**
     * These are the dimentions to use when drawing the pegs.
     * @param PEG_WIDTH is how wide the top of the peg is.
     * @param PEG_HEIGHT is how high the peg is.
     * @param PEN_WIDTH is the thickness of the pen used to draw the peg.
     */
    enum PegDimensions
    {
        PEG_WIDTH=2,
        PEG_HEIGHT=1,
        PEN_WIDTH=1
    };
    /// @param m_positionX is the current x-coordinate position.
    int m_positionX;
    /// @param m_positionY is the current y-coordinate position.
    int m_positionY;
    /// @param m_score is the current score the peg is representing.
    int m_score;
    /// @param m_startX is the starting x-coordinate position.
    int m_startX;
    /// @param m_startY is the starting y-coordinate position.
    int m_startY;
    /// @param m_colour is the colour of the peg.
    QColor m_colour;
    /// @param m_lead  is whether the peg is the leading peg or not.
    bool m_lead;
    /// @param m_startLead is the whether the peg is the leading peg or not when it started.
    bool m_startLead;
};
#endif
