/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2008  John Schneiderman <JohnMS@member.fsf.org>                         *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#include "sallycrazyeightsai.h"
#include "basicgamestrategies.h"
#include "kardsgterror.h"
#include "../../games/crazyeights/crazyeightsrules.h"

#include <QtDebug>

SallyCrazyEightsAI::SallyCrazyEightsAI(const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand, const GameInformation &specialInformation)
{
    m_playSequence = playSequence;
    m_pRules = &rules;
    m_hand = hand;
    m_specialInformation = specialInformation;
}

SallyCrazyEightsAI::~SallyCrazyEightsAI()
{
}

CardSequence SallyCrazyEightsAI::selectCard() const
{
    CardSequence selection;
    Card::Rank leadRank = m_playSequence.front().rank();
    Card::Suit leadSuit = m_playSequence.front().suit();
    BasicGameStrategies basicStrategy(*m_pRules);

    // Determine if we have the lead suit
    if (m_hand.hasSuit(leadSuit))
    {
        // Find the highest card to play
        selection.addCard(basicStrategy.highestCardOfSuit(m_hand, leadSuit));
        if (selection.front().rank() == Card::EIGHT)
            if (basicStrategy.numberOfCardsOfSuit(m_hand, leadSuit) > 1)
            {
                qDebug() << "     SallyCrazyEightsAI, Not playing our Eight right away.";
                selection.clear();
                selection.addCard(basicStrategy.nextHighestCardOfSuit(m_hand, Card(Card::EIGHT, leadSuit), leadSuit));
            }
        /*
            If the player has a card in the lead suit, but the suit only has a few cards in it
            and the player has a card of the same rank in a different suit that has more
            cards in it, we should play that instead. IE. 2S 9S TS JC KC, with a lead of JS,
            the player should play the JC instead of the 9S.
        */
        if ((basicStrategy.numberOfCardsOfSuit(m_hand, leadSuit) <= 3) && m_hand.hasRank(leadRank))
        {
            Card test;

            // Find the card with the corresponding rank.
            for (int index = 0, size = m_hand.size(); index < size; ++index)
                if (m_hand[index].rank() == leadRank)
                {
                    test = m_hand[index];
                    break;
                }
            // Make sure it's worth the switch
            if (basicStrategy.numberOfCardsOfSuit(m_hand, test.suit()) > 3)
            {
                selection.clear();
                selection.addCard(test);
                qDebug() << "     SallyCrazyEightsAI, Switching suits with a rank card: " << selection;
            }
            else
                qDebug() << "     SallyCrazyEightsAI, Playing the highest card: " << selection;
        }
        else
            qDebug() << "     SallyCrazyEightsAI, Playing the highest card: " << selection;
    }
    else
    {
        // See if we have a card of the same rank to play
        if (m_hand.hasRank(leadRank))
        {
            for (int index = 0, size = m_hand.size(); index < size; ++index)
                if (m_hand[index].rank() == leadRank)
                {
                    selection.addCard(m_hand[index]);
                    break;
                }
            qDebug() << "     SallyCrazyEightsAI, Playing the lead rank: " << selection;
        }
        else if (m_hand.hasRank(Card::EIGHT)) // Play an eight
        {
            for (int index = 0, size = m_hand.size(); index < size; ++index)
                if (m_hand[index].rank() == Card::EIGHT)
                {
                    selection.addCard(m_hand[index]);
                    break;
                }
            qDebug() << "     SallyCrazyEightsAI, Playing the eight: " << selection;
        }
        else // Return a blank card to indicate we need to draw a card or pass
        {
            selection.addCard(Card());
            qDebug() << "     SallyCrazyEightsAI, Pass or drawing: " << selection;
        }
    }
    return selection;
}

Card::Suit SallyCrazyEightsAI::selectSuit(int ) const
{
    Card::Suit suit = Card::SUIT_ERR;
    BasicGameStrategies basicStrategy(*m_pRules);
    int numberOfCardsInHighSuit = -1;
    int numberOfCards = -1;

    for (int index = 0; index < Card::NUMBER_OF_SUITS; ++index)
    {
        numberOfCards = basicStrategy.numberOfCardsOfSuit(m_hand, static_cast<Card::Suit>(index));
        if (numberOfCards > numberOfCardsInHighSuit)
        {
            numberOfCardsInHighSuit = numberOfCards;
            suit = static_cast<Card::Suit>(index);
        }
    }
    qDebug() << "     SallyCrazyEightsAI, Changing suit to: " << suit;
    return suit;
}
