/***************************************************************************************************
*****    This file is part of KardsGT.                                                         *****
*****                                                                                          *****
*****    Copyright (C) 2007-2008  John Schneiderman <JohnMS@member.fsf.org>                    *****
*****                                                                                          *****
*****    This program is free software: you can redistribute it and/or modify it under the     *****
*****    terms of the GNU General Public License as published by the Free Software Foundation, *****
*****    either version 3 of the License, or (at your option) any later version.               *****
*****                                                                                          *****
*****    This program is distributed in the hope that it will be useful, but WITHOUT ANY       *****
*****    WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A       *****
*****    PARTICULAR PURPOSE.  See the GNU General Public License for more details.             *****
*****                                                                                          *****
*****    You should have received a copy of the GNU General Public License along with this     *****
*****    program. If not, see <http://www.gnu.org/licenses/>.                                  *****
***************************************************************************************************/
#ifndef WALLYEUCHREAI_H_
#define WALLYEUCHREAI_H_

#include "cardsequence.h"
#include "card.h"
#include "basicgamestrategies.h"

class RuleBase;

/**
 * This is the strategy used by Wally to play Euchre.
 *
 * @author John Schneiderman
 */
class WallyEuchreAI
{
public:
    /**
     * Constructor
     * @param playSequence is the current play sequence.
     * @param rules is the current rules.
     * @param hand is the current hand.
     */
    WallyEuchreAI(const CardSequence &playSequence, const RuleBase &rules, const CardSequence &hand);
    /**
     * Destructor
     */
    ~WallyEuchreAI();
    /**
     * Determines if the he wants the turn-up card.
     * @param turnUpCard is the turn-up card.
     * @param score is his team score.
     * @param opponentScore is his opponent's team score.
     * @return WANT_TURN_UP if he wants the turn-up card, DOES_NOT_DECIDE elsewise.
     */
    int turnUp(const Card &turnUpCard, int score, int opponentScore) const;
    /**
     * Determines the suit he might want.
     * @param turnUpCard is the turn-up card.
     * @param score is his team score.
     * @param opponentScore is his opponent's team score.
     * @return the suit he would like, SUIT_ERR elsewise.
     */
    int selectTrump(const Card &turnUpCard, int score, int opponentScore) const;
    /**
     * Determines if he wishes to play alone.
     * @param score is his team score.
     * @param opponentScore is his opponent's team score.
     * @return PLAY_ALONE if he wants to play alone, DOES_NOT_DECIDE elsewise.
     */
    int playAlone(int score, int opponentScore) const;
    /**
     * Determines if he wishes to defend alone.
     * @return DEFEND_ALONE if he wants to defend alone, DOES_NOT_DECIDE elsewise.
     */
    int defendAlone() const;
    /**
     * Determines if he wants the turn-up card.
     * @return WANT_TURN_UP_CARD if he wants the turn-up card, DOES_NOT_DECIDE elsewise.
     */
    int wantTurnUp() const;
    /**
     * Selects a card to play from his hand.
     * @param partnersCard is his partner's card.
     * @return the card he's selected.
     */
    CardSequence selectCards(const Card &partnersCard) const;
    /**
     * Selects a card to discard from his hand.
     * @return the card he's selected.
     */
    CardSequence discard() const;

private:
    /// @param m_playSequence is the play sequence we're working with.
    CardSequence m_playSequence;
    /// @param m_hand is the hand of cards we're working with.
    CardSequence m_hand;
    /// @param m_pRules are the rules for the game.
    const RuleBase *m_pRules;
    /// @param basicStrategies is the general basic strategies.
    BasicGameStrategies basicStrategies;
};
#endif
