/* -*- c++ -*-
 *
 * mldonkeyappletgui.cpp
 *
 * Copyright (C) 2003 Petter E. Stokke <gibreel@gibreel.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */


#include <klocale.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qtooltip.h>
#include <kpushbutton.h>
#include <kiconloader.h>
#include <kglobalsettings.h>

#include "mldonkeyappletgui.h"
#include "mldonkeyapplet.h"

MLDonkeyAppletGUI::MLDonkeyAppletGUI( QWidget* parent, const char* name, WFlags fl )
    : QWidget( parent, name, fl )
{
    if ( !name )
	setName( "MLDonkeyAppletGUI" );

    KIconLoader icons("mldonkeyapplet");

    QIconSet muteIcon(icons.loadIcon("mld-mutedonkey", KIcon::User));
    muteIcon.setPixmap(icons.loadIcon("mld-unmutedonkey", KIcon::User), QIconSet::Small, QIconSet::Normal, QIconSet::On);

    LaunchButton = new KPushButton( this, "LaunchButton" );
    LaunchButton->setSizePolicy( QSizePolicy( QSizePolicy::Fixed, QSizePolicy::Fixed, 0, 0, LaunchButton->sizePolicy().hasHeightForWidth() ) );
    LaunchButton->setMaximumSize( QSize( 20, 20 ) );
    LaunchButton->setFocusPolicy( QPushButton::NoFocus );
    LaunchButton->setIconSet( icons.loadIconSet("mld-launchgui", KIcon::User) );
    LaunchButton->setToggleButton(true);
    LaunchButton->setFlat(true);
    QToolTip::add(LaunchButton, i18n("Show/hide the KMLDonkey interface"));


    MuteButton = new KPushButton( this, "MuteButton" );
    MuteButton->setSizePolicy( QSizePolicy( QSizePolicy::Fixed, QSizePolicy::Fixed, 0, 0, MuteButton->sizePolicy().hasHeightForWidth() ) );
    MuteButton->setMaximumSize( QSize( 20, 20 ) );
    MuteButton->setFocusPolicy( QPushButton::NoFocus );
    MuteButton->setIconSet(muteIcon);
    MuteButton->setToggleButton(true);
    MuteButton->setFlat(true);
    QToolTip::add(MuteButton, i18n("Toggle mute mode"));

    FirstLabel = new QLabel( this, "FirstLabel" );
    FirstLabel->setSizePolicy( QSizePolicy( QSizePolicy::Minimum, QSizePolicy::Fixed, 0, 0, FirstLabel->sizePolicy().hasHeightForWidth() ) );
    FirstLabel->setMaximumSize( QSize( 32767, 20 ) );
    FirstLabel->setMargin( 2 );
    FirstLabel->setAlignment( int( QLabel::AlignVCenter | QLabel::AlignRight ) );

    SecondLabel = new QLabel( this, "SecondLabel" );
    SecondLabel->setSizePolicy( QSizePolicy( QSizePolicy::Minimum, QSizePolicy::Fixed, 0, 0, SecondLabel->sizePolicy().hasHeightForWidth() ) );
    SecondLabel->setMaximumSize( QSize( 32767, 20 ) );
    SecondLabel->setMargin( 2 );
    SecondLabel->setAlignment( int( QLabel::AlignVCenter | QLabel::AlignRight ) );

    SecondStatus = new QLabel( this, "SecondStatus" );
    SecondStatus->setSizePolicy( QSizePolicy( QSizePolicy::Minimum, QSizePolicy::Fixed, 0, 0, SecondStatus->sizePolicy().hasHeightForWidth() ) );
    SecondStatus->setMaximumSize( QSize( 32767, 20 ) );
    SecondStatus->setFrameShape( QLabel::Panel );
    SecondStatus->setFrameShadow( QLabel::Sunken );
    SecondStatus->setPaletteBackgroundColor( KGlobalSettings::baseColor() );

    FirstStatus = new QLabel( this, "FirstStatus" );
    FirstStatus->setSizePolicy( QSizePolicy( QSizePolicy::Minimum, QSizePolicy::Fixed, 0, 0, FirstStatus->sizePolicy().hasHeightForWidth() ) );
    FirstStatus->setMaximumSize( QSize( 32767, 20 ) );
    FirstStatus->setFrameShape( QLabel::Panel );
    FirstStatus->setFrameShadow( QLabel::Sunken );
    FirstStatus->setPaletteBackgroundColor( KGlobalSettings::baseColor() );

    SecondStatus->setText( "..." );
    FirstStatus->setText( "..." );

    connect(LaunchButton, SIGNAL(toggled(bool)), this, SLOT(toggleLaunch(bool)));
    connect(MuteButton, SIGNAL(toggled(bool)), this, SLOT(toggleMute(bool)));

    setSizePolicy( QSizePolicy( QSizePolicy::Fixed, QSizePolicy::Fixed, 0, 0, sizePolicy().hasHeightForWidth() ) );

    mysize = -1;
    horiz = NULL; grid = NULL;
    relayoutWidgets(0);
}

/*
 *  Destroys the object and frees any allocated resources
 */
MLDonkeyAppletGUI::~MLDonkeyAppletGUI()
{
    // no need to delete child widgets, Qt does it all for us
}

void MLDonkeyAppletGUI::updateLayout()
{
    hide();

    if (horiz) delete horiz;
    if (grid) delete grid;
    horiz = NULL; grid = NULL;

    MLDonkeyApplet* app = (MLDonkeyApplet*)parent();
    int count = app->activeDisplays.count(), pos = 0;

    FirstLabel->hide();
    SecondLabel->hide();
    FirstStatus->hide();
    SecondStatus->hide();

    if (!mysize) {
	MuteButton->show();

	grid = new QGridLayout(this, 2, 1, 0, 0);

	if (count > 0) {

	    if (app->showLabels) {
		FirstLabel->show();
		if (count > 1) {
		    SecondLabel->show();
		    grid->addWidget(FirstLabel, 0, pos, Qt::AlignAuto);
		    grid->addWidget(SecondLabel, 1, pos, Qt::AlignAuto);
		} else {
		    grid->addMultiCellWidget(FirstLabel, 0, 1, pos, pos, Qt::AlignAuto);
		}
		pos++;
	    }

	    FirstStatus->show();
	    if (count > 1) {
		SecondStatus->show();
		grid->addWidget(FirstStatus, 0, pos, Qt::AlignAuto);
		grid->addWidget(SecondStatus, 1, pos, Qt::AlignAuto);
	    } else {
		grid->addMultiCellWidget(FirstStatus, 0, 1, pos, pos, Qt::AlignAuto);
	    }
	    pos++;

	}

	grid->addWidget( LaunchButton, 0, pos, Qt::AlignAuto );
	grid->addWidget( MuteButton, 1, pos, Qt::AlignAuto );

    } else {
	horiz = new QHBoxLayout(this);
	FirstLabel->hide();
	SecondLabel->hide();
	FirstStatus->hide();
	SecondStatus->hide();

	if (count) {
	    if (app->showLabels && !app->showDouble) {
		FirstLabel->show();
		horiz->addWidget(FirstLabel, Qt::AlignAuto);
	    }
	    FirstStatus->show();
	    horiz->addWidget( FirstStatus, Qt::AlignAuto );

	    if (count > 1) {
		if (app->showLabels && !app->showDouble) {
		    SecondLabel->show();
		    horiz->addWidget(SecondLabel, Qt::AlignAuto);
		}
		SecondStatus->show();
		horiz->addWidget( SecondStatus, Qt::AlignAuto );
	    }
	}

	horiz->addWidget( LaunchButton, Qt::AlignAuto );
	if (app->showMute) {
	    MuteButton->show();
	    horiz->addWidget( MuteButton, Qt::AlignAuto );
	}
	else MuteButton->hide();
    }

    adjustSize();
    show();
}

void MLDonkeyAppletGUI::relayoutWidgets(int small)
{
    if (mysize == small) return;
    mysize = small;
    updateLayout();
}

void MLDonkeyAppletGUI::updateLabels(const QString& first, const QString& second)
{
    FirstLabel->setText(first);
    SecondLabel->setText(second);
}

void MLDonkeyAppletGUI::updateStatus(const QString& first, const QString& second)
{
    FirstStatus->setText(first);
    SecondStatus->setText(second);
}

void MLDonkeyAppletGUI::updateTooltips(const QString& first, const QString& second)
{
    QToolTip::remove(FirstStatus);
    QToolTip::remove(SecondStatus);
    QToolTip::add(FirstStatus, first);
    QToolTip::add(SecondStatus, second);
}

void MLDonkeyAppletGUI::donkeyDisconnected()
{
    SecondStatus->setText("...");
    FirstStatus->setText("...");
}

void MLDonkeyAppletGUI::setLaunchState(bool state)
{
    blockSignals(true);
    LaunchButton->setOn(state);
    blockSignals(false);
}

void MLDonkeyAppletGUI::toggleLaunch(bool t)
{
    emit toggledLaunch(t);
}

void MLDonkeyAppletGUI::toggleMute(bool t)
{
    emit toggledMute(t);
}

#include "mldonkeyappletgui.moc"
