/* -*- c++ -*-
 *
 * infodialog.cpp
 *
 * Copyright (C) 2003 Petter E. Stokke <gibreel@gibreel.net>
 * Copyright (C) 2003 Sebastian Sauer <mail@dipe.org> 
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#include "infodialog.h"
#include "clientlist.h"
#include "kmldonkey.h"
#include "webservices.h"
#include "preview.h"

#include <kglobal.h>
#include <kconfig.h>
#include <klocale.h>
#include <kdebug.h>
#include <qlayout.h>
#include <qlabel.h>

#include <fileinfo.h>

static QLabel* makeLabel(const QString& label, QWidget* parent, bool alignTop = false)
{
    QLabel* l = new QLabel(label, parent);
    if (alignTop) l->setAlignment(Qt::AlignTop | Qt::AlignRight);
    else l->setAlignment(Qt::AlignVCenter | Qt::AlignRight);
    l->setSizePolicy(QSizePolicy::Minimum, QSizePolicy::Fixed);
    return l;
}

FileInfoDialog::FileInfoDialog(int file)
    : KDialogBase(Tabbed, i18n("File Information"),
                  Close, Default, 0, 0, false)
{
    fileno = file;
    FileInfo* fi = KMLDonkey::App->donkey->findDownloadFileNo(fileno);
    if (!fi) return;

    QFrame *frame = addPage(QString("File"));
    QGridLayout* layout = new QGridLayout(frame, 7, 2, 4);

    layout->addWidget(makeLabel(i18n("Name:"), frame), 0, 0);
    layout->addWidget(makeLabel(i18n("Complete:"), frame), 1, 0);
    layout->addWidget(makeLabel(i18n("Downloaded:"), frame), 2, 0);
    layout->addWidget(makeLabel(i18n("Speed:"), frame), 3, 0);
    layout->addWidget(makeLabel(i18n("Hash:"), frame), 4, 0);
    layout->addWidget(makeLabel(i18n("Format:"), frame), 5, 0);
    layout->addWidget(makeLabel(i18n("Comment:"), frame), 6, 0);
    layout->addWidget(makeLabel(i18n("Alt. names:"), frame, true), 7, 0);

    nameView = new KLineEdit(fi->fileName(), frame, "nameView");
    nameView->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);
    nameView->setReadOnly(true);
    layout->addWidget(nameView, 0, 1);

    availView = new AvailabilityWidget(fileno, frame, "availView");
    layout->addWidget(availView, 1, 1);

    QHBoxLayout* l2 = new QHBoxLayout();

    sizeView = new KLineEdit(humanReadableSize(fi->fileDownloaded()), frame, "sizeView");
    sizeView->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);
    sizeView->setReadOnly(true);
    l2->addWidget(sizeView);

    l2->addWidget(makeLabel(i18n("Size:"), frame));
    totalSizeView = new KLineEdit(humanReadableSize(fi->fileSize()), frame, "totalSizeView");
    totalSizeView->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);
    totalSizeView->setReadOnly(true);
    l2->addWidget(totalSizeView);

    l2->addWidget(makeLabel(i18n("Network:"), frame));
    networkView = new KLineEdit(KMLDonkey::App->donkey->findNetworkNo(fi->fileNetwork())->networkName(), frame, "networkView");
    networkView->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);
    networkView->setReadOnly(true);
    l2->addWidget(networkView);

    layout->addLayout(l2, 2, 1);

    QHBoxLayout* l3 = new QHBoxLayout();

    speedView = new KLineEdit(humanReadableSize((int64)fi->fileSpeed()) + "b/s", frame, "speedView");
    speedView->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);
    speedView->setReadOnly(true);
    l3->addWidget(speedView);

    l3->addWidget(makeLabel(i18n("ETA:"), frame));
    etaView = new KLineEdit(calculateETA(fi), frame, "etaView");
    etaView->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);
    etaView->setReadOnly(true);
    l3->addWidget(etaView);

    layout->addLayout(l3, 3, 1);

    hashView = new KLineEdit(FileInfo::md4ToString(fi->fileMD4()), frame, "hashView");
    hashView->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);
    hashView->setReadOnly(true);
    layout->addWidget(hashView, 4, 1);

    formatView = new KLineEdit(fi->fileFormatInfo(), frame, "formatView");
    formatView->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);
    formatView->setReadOnly(true);
    layout->addWidget(formatView, 5, 1);

    commentView = new KLineEdit(fi->fileComment(), frame, "commentView");
    commentView->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);
    commentView->setReadOnly(true);
    layout->addWidget(commentView, 6, 1);

    namesView = new KTextBrowser(frame, "namesView");
    namesView->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::MinimumExpanding);
    namesView->setTextFormat(QTextBrowser::PlainText);
    namesView->setWordWrap(QTextBrowser::NoWrap);
    namesView->setText(fi->fileNames().join("\n"));
    layout->addWidget(namesView, 7, 1);

    layout->setColStretch(0, 1);
    layout->setColStretch(1, 100);

    QFrame *srcframe = addPage(i18n("Sources"));
    QBoxLayout* srclayout = new QVBoxLayout(srcframe);
    clientList = new ClientList(fileno, srcframe);
    srclayout->addWidget(clientList);

    clientList->restoreLayout(KGlobal::config(), "DownloadSourceList");

    if (KMLDonkey::App->hostManager->isLocalHost(KMLDonkey::App->lastHost)) {
	DonkeyHost* host = (DonkeyHost*)KMLDonkey::App->hostManager->hostProperties(KMLDonkey::App->lastHost);
	KURL url(host->tempDirectory());
	url.addPath(FileInfo::md4ToString(fi->fileMD4()));

	previewframe = addPage(i18n("Preview"));
	(new QVBoxLayout(previewframe))->addWidget(preview = new PreviewWidget(url, fi->fileName(), previewframe));
    } else {
	preview = 0;
	previewframe = 0;
    }

    if (KMLDonkey::App->donkey->findNetworkNo(fi->fileNetwork())->networkName() == "Donkey") {
	QFrame* jigleframe = addPage(i18n("Jigle"));
	(new QVBoxLayout(jigleframe))->addWidget(new JigleFileInfo(fileno, jigleframe));
	QFrame* bitziframe = addPage(i18n("Bitzi"));
	(new QVBoxLayout(bitziframe))->addWidget(new BitziFileInfo(fileno, bitziframe));
    }

    updateCaption(fi);

    setInitialSize(configDialogSize("FileInfoDialog"));
    connect(this, SIGNAL(finished()), this, SLOT(saveSize()));
    connect(KMLDonkey::App->donkey, SIGNAL(fileUpdated(int)), this, SLOT(updateFile(int)));
}

void FileInfoDialog::updateCaption(FileInfo* fi)
{
    QString caption;
    caption = i18n("Dialog window caption: (percent complete) filename",
		   "(%1%) %2").arg(KGlobal::locale()->formatNumber((fi->fileDownloaded() * 100.0) / fi->fileSize(), 2)).arg(fi->fileName());
    setCaption(caption);
}

void FileInfoDialog::saveSize()
{
    saveDialogSize("FileInfoDialog");
    clientList->saveLayout(KGlobal::config(), "DownloadSourceList");
    deleteLater();
}

void FileInfoDialog::updateFile(int file)
{
    if (file != fileno) return;
    FileInfo* fi = KMLDonkey::App->donkey->findDownloadFileNo(fileno);
    if (!fi) {
        deleteLater();
        return;
    }
    nameView->setText(fi->fileName());
    availView->update();
    sizeView->setText(humanReadableSize(fi->fileDownloaded()));
    totalSizeView->setText(humanReadableSize(fi->fileSize()));
    speedView->setText(i18n("download speed in bytes per second", "%1b/s").arg(humanReadableSize((int64)fi->fileSpeed())));
    etaView->setText(calculateETA(fi));
    formatView->setText(fi->fileFormatInfo());
    commentView->setText(fi->fileComment());
    updateCaption(fi);
}

#include "infodialog.moc"
