/* -*- c++ -*-
 *
 * infolist.cpp
 *
 * Copyright (C) 2003 Petter E. Stokke <gibreel@gibreel.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#include <kglobal.h>
#include <klocale.h>
#include <kdebug.h>
#include <klistview.h>
#include <qvariant.h>
#include <qpainter.h>
#include <sys/types.h>
#include <time.h>

#include "infolist.h"
#include "infolist.moc"
#include "kmldonkey.h"

// InfoList

InfoList::InfoList(QWidget* parent, const char* name)
    : KListView(parent, name)
{
    setSelectionModeExt( KListView::Extended );
    setAllColumnsShowFocus( TRUE );
    setShowSortIndicator( TRUE );
    setResizeMode( KListView::NoColumn );
    setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    setMinimumSize(1,1);
}

InfoList::~InfoList()
{
}

void InfoList::initialise(KConfig* config, const QString& group)
{
    int i;
    for (i=0; i<m_columnCount; i++)
        buildColumn(i);
    restoreLayout(config, group);
}

void InfoList::finalise(KConfig* config, const QString& group)
{
    saveLayout(config, group);
}

int InfoList::addColumn(const QString& label, int width, QListView::WidthMode widthMode)
{
    m_columns.insert(m_columnCount, label);
    m_columnWidth.insert(m_columnCount, width);
    m_columnWidthMode.insert(m_columnCount, widthMode);
    return m_columnCount++;
}

int InfoList::visibleToReal(int c) const
{
    return m_visibleReal[c];
}

int InfoList::realToVisible(int c) const
{
    return m_realVisible[c];
}

QValueList<int> InfoList::visibleColumns() const
{
    QValueList<int> foo = m_realVisible.keys();
    qHeapSort(foo);
    return foo;
}

int InfoList::realColumnCount() const
{
    return m_columnCount;
}

int InfoList::buildColumn(int c)
{
    int r = KListView::addColumn(m_columns[c], m_columnWidth[c]);
    setColumnWidthMode(r, m_columnWidthMode[c]);
    m_visibleReal.replace(r, c);
    m_realVisible.replace(c, r);
    return r;
}

void InfoList::focusInEvent(QFocusEvent* event)
{
    KListView::focusInEvent(event);
    emit gotFocus();
    emit focusChanged();
}

void InfoList::focusOutEvent(QFocusEvent* event)
{
    KListView::focusInEvent(event);
    emit lostFocus();
    emit focusChanged();
}


// InfoItemCacheEntry

void InfoItemCacheEntry::update(const QString& str)
{
    m_str = str;
    m_isNum = false;
}

void InfoItemCacheEntry::update(const QString& str, double num)
{
    m_str = str;
    m_num = num;
    m_isNum = true;
}

int InfoItemCacheEntry::compare(const InfoItemCacheEntry* i) const
{
    // We assume compares are never made against an entry with a different type.
    if (m_isNum)
        return (m_num < i->m_num) ? -1 : (m_num > i->m_num) ? 1 : 0;
    else
        return m_str.lower().localeAwareCompare(i->m_str.lower());
}

// InfoItem (base class)

InfoItem::InfoItem( KListView *parent, int file )
    : KListViewItem( parent )
{
    fileno = file;
    cache.setAutoDelete(true);
}

InfoItem::~InfoItem()
{
}

void InfoItem::refresh(bool alsoRepaint)
{
    if (!listView()) return;
    int c, t = listView()->columns();
    InfoItemCacheEntry* e = cache.first();
    for (c=0; c < t; c++) {
        if (!e) cache.append(e = new InfoItemCacheEntry());
        if (isNumeric(c))
            e->update(xtext(c), numeric(c));
        else
            e->update(xtext(c));
        e = cache.next();
    }
    if (alsoRepaint)
        repaint();
}

QString InfoItem::text(int column) const
{
    const InfoItemCacheEntry* e = cacheEntry(column);
    if (!e) return QString::null;
    return e->text();
}

void InfoItem::setFileNo( int file )
{
    fileno = file;
    refresh();
}

int InfoItem::fileNo()
{
    return fileno;
}

double InfoItem::numeric(int) const
{
    return 0.0;
}

bool InfoItem::isNumeric(int) const
{
    return false;
}

const InfoItemCacheEntry* InfoItem::cacheEntry(int column) const
{
    return ((QPtrList<InfoItemCacheEntry>)cache).at(column);
}

int InfoItem::compare( QListViewItem* i, int col, bool ) const
{
    return cacheEntry(col)->compare(static_cast<InfoItem*>(i)->cacheEntry(col));
}


// Utility functions

QString humanReadableSize(int64 rsz)
{
    QString foo;
    double sz = (double)rsz;

    if (KMLDonkey::App->humanReadableSizes) {
        if (sz >= (double)(1024 * 1024 * 1024)) {
            sz = sz / (1024 * 1024 * 1024);
            foo = i18n("gigabyte suffix", "%1G").arg(KGlobal::locale()->formatNumber(sz, 1));
        } else if (sz >= (double)(1024 * 1024)) {
            sz = sz / (1024 * 1024);
            foo = i18n("megabyte suffix", "%1M").arg(KGlobal::locale()->formatNumber(sz, 1));
        } else if (sz >= (double)1024) {
            sz = sz / 1024;
            foo = i18n("kilobyte suffix", "%1K").arg(KGlobal::locale()->formatNumber(sz, 1));
        } else foo = KGlobal::locale()->formatNumber(sz, 0);
    }
    else {
        foo = KGlobal::locale()->formatNumber(sz, 0);
    }

    return foo;
}

QString humanReadableSpeed(double sp)
{
    if (!sp) return i18n("signifies absence of data in list columns", "-");
    else return KGlobal::locale()->formatNumber(sp / 1024.0, 1);
}

QString humanReadableTime(time_t t, bool shortFormat)
{
    if (!t) return i18n("zero seconds", "0s");
    if (t < 0) return i18n("signifies absence of data in list columns", "-");
    QString foo;
    int f = 0;
    if (t > 86400) {
        foo += i18n("number of days", "%1d ").arg(KGlobal::locale()->formatNumber(t/86400, 0));
        t %= 86400;
        f = 1;
        if (shortFormat) return foo.simplifyWhiteSpace();
    }
    if (t > 3600) {
        foo += i18n("number of hours", "%1h ").arg(KGlobal::locale()->formatNumber(t/3600, 0));
        t %= 3600;
        if (shortFormat) return foo.simplifyWhiteSpace();
    }
    if (t > 60) {
        foo += i18n("number of minutes", "%1m ").arg(KGlobal::locale()->formatNumber(t/60, 0));
        t %= 60;
        if (shortFormat) return foo.simplifyWhiteSpace();
    }
    if (t && !f) {
        foo += i18n("number of seconds", "%1s").arg(KGlobal::locale()->formatNumber(t, 0));
    }
    return foo.simplifyWhiteSpace();
}

QString calculateETA(FileInfo* fi)
{
    if (fi->fileSize() < fi->fileDownloaded()) {
        return i18n("file should have completed already", "Overdue");
    }
    if (fi->fileSize() == fi->fileDownloaded()) {
        return i18n("file is just about to complete", "Imminent");
    }
    if (!fi->fileSpeed()) {
        return i18n("signifies absence of data in list columns", "-");
    }
    return humanReadableTime((time_t)((double)(fi->fileSize() - fi->fileDownloaded()) / fi->fileSpeed()), false);
}

QString humanReadablePriority(int pri)
{
    if (pri > 0)
        return pri > 10 ? i18n("very high priority", "Very high") : i18n("high priority", "High");
    if (pri < 0)
        return pri < -10 ? i18n("very low priority", "Very low") : i18n("low priority", "Low");
    return i18n("normal priority", "Normal");
}


// DownloadFile

DownloadFile::DownloadFile( KListView *parent, int file )
    : InfoItem(parent,file)
      , AvailabilityRenderer(file)
{
    setRenameEnabled(0, true);
    refresh();
}

void DownloadFile::setText(int col, const QString& name)
{
    InfoItem::setText(col, name);
    if (col) return;

    QString foo = "rename " + QString::number(fileno) + " \"" + name + "\"";
    KMLDonkey::App->sendConsoleMessage(foo);
    KMLDonkey::App->donkey->refreshFileInfo(fileno);
}

QString DownloadFile::xtext( int column ) const
{
    QString foo;
    FileInfo* it = KMLDonkey::App->donkey->findDownloadFileNo(fileno);
    if (!it) {
        if (!column) return i18n("Unknown file");
        return i18n("signifies absence of data in list columns", "-");
    }
    switch (column) {
    case 0: // name
        return it->fileName();
    case 1: // Priority
        return humanReadablePriority(it->filePriority());
    case 4: // size
        return humanReadableSize(it->fileSize());
    case 5: // downloaded
        return humanReadableSize(it->fileDownloaded());
    case 6: // remaining
        return i18n("percentage", "%1%").arg(KGlobal::locale()->formatNumber(it->fileSize() ? (it->fileDownloaded() * 100.0) / it->fileSize() : 0.0, 1));
    case 2: // status
        switch (it->fileState()) {
        case FileInfo::Downloading:
            if (!it->fileSpeed()) return i18n("looking for file sources", "Looking");
            return i18n("file is being downloaded", "Downloading");
        case FileInfo::Paused: return i18n("file is paused", "Paused");
        case FileInfo::Complete: return i18n("file is complete", "Complete");
        case FileInfo::Shared: return i18n("file is being shared", "Shared");
        case FileInfo::Cancelled: return i18n("file has been canceled", "Canceled");
        case FileInfo::New: return i18n("new file added to list", "New");
        case FileInfo::Aborted: return i18n("file has been aborted with a reason", "Aborted: %1").arg(it->fileAbortedMsg());
        case FileInfo::Queued: return i18n("file has been queued", "Queued");
        }
        return i18n("unknown file state", "Unknown");
    case 3: // speed
        return humanReadableSpeed(it->fileSpeed());
    case 7: // eta
        return calculateETA(it);
    case 8: // network
        return KMLDonkey::App->donkey->findNetworkNo(it->fileNetwork())->networkName();
    case 9: // availability
        return QString::null;
    case 10: // hash
        return FileInfo::md4ToString(it->fileMD4());
    case 11: // last seen
        return humanReadableTime((time_t)it->fileLastSeen(), true);
    case 12: // age
    {
        time_t age = time(0) - it->fileAge();
        return humanReadableTime(age, true);
    }
    default:
        return "ERROR!";
    }
}

double DownloadFile::numeric( int col ) const
{
    FileInfo* it = KMLDonkey::App->donkey->findDownloadFileNo(fileno);
    if (!it) return 0.0;

    switch (col) {
    case 1:
        return (double)it->filePriority();
    case 4:
        return (double)it->fileSize();
    case 5:
        return (double)it->fileDownloaded();
    case 6:
        return it->fileSize() ? ((double)it->fileDownloaded() * 100.0) / (double)it->fileSize() : 0.0;
    case 3:
        return it->fileSpeed();
    case 9:
        return (double)it->fileSources().size();
    case 11:
        return (time_t)it->fileLastSeen();
    case 12:
        return it->fileAge();
    default:
        return 0.0;
    }
}

bool DownloadFile::isNumeric(int col) const
{
    switch (col) {
    case 1:
    case 3:
    case 4:
    case 5:
    case 6:
    case 9:
    case 11:
    case 12:
        return true;
    default:
        return false;
    }
}

void DownloadFile::paintCell(QPainter* p, const QColorGroup& cg, int col, int w, int align)
{
    if (col == 9) {
        QRect foo(0, 0, w, height());
        p->setBrush(Qt::red);
        p->setPen(Qt::green);
        p->drawRect(foo);
        paintAvailability(*p, foo);
        return;
    }

    QColorGroup colgrp(cg);
    if (KMLDonkey::App->coloredViews) {
        FileInfo* it = KMLDonkey::App->donkey->findDownloadFileNo(fileno);
        if (it) {
            switch (it->fileState()) {
            case FileInfo::Downloading:
                if (it->fileSpeed() > 0)
                    colgrp.setColor(QColorGroup::Text, KMLDonkey::App->colorDownloadDownloading);
                else
                    colgrp.setColor(QColorGroup::Text, KMLDonkey::App->colorDownloadLooking);
                break;
            case FileInfo::Paused:
                colgrp.setColor(QColorGroup::Text, KMLDonkey::App->colorDownloadPaused);
                break;
            case FileInfo::Queued:
                colgrp.setColor(QColorGroup::Text, KMLDonkey::App->colorDownloadQueued);
                break;
            default:
                break;
            }
        }
    }

    p->save();
    InfoItem::paintCell(p, colgrp, col, w, align);
    p->restore();
}

int DownloadFile::width(const QFontMetrics& fm, const QListView* lv, int c) const
{
    if (c == 9) return (chunkNo > 100) ? 100 : chunkNo;
    return InfoItem::width(fm, lv, c);
}


// DownloadedFile

QString DownloadedFile::xtext( int column ) const
{
    FileInfo* it = KMLDonkey::App->donkey->findDownloadedFileNo(fileno);
    if (!it) it = KMLDonkey::App->donkey->findDownloadFileNo(fileno);
    if (!it) {
        if (!column) return i18n("Unknown file");
        return i18n("signifies absence of data in list columns", "-");
    }
    switch (column) {
    case 0: // name
        return it->fileName();
    case 1: // size
        return humanReadableSize(it->fileSize());
    case 2: // format
        return it->fileFormatInfo();
    case 3: // network
        return KMLDonkey::App->donkey->findNetworkNo(it->fileNetwork())->networkName();
    case 4: // hash
        return FileInfo::md4ToString(it->fileMD4());
    default:
        return "ERROR!";
    }
}

double DownloadedFile::numeric( int col ) const
{
    FileInfo* it = KMLDonkey::App->donkey->findDownloadedFileNo(fileno);
    if (!it) it = KMLDonkey::App->donkey->findDownloadFileNo(fileno);
    if (!it) return 0.0;
    switch (col) {
    case 1:
        return (double)it->fileSize();
    default:
        return 0.0;
    }
}

bool DownloadedFile::isNumeric(int col) const
{
    switch (col) {
    case 1:
        return true;
    default:
        return false;
    }
}


// ServerInfoItem

QString ServerInfoItem::xtext( int column ) const
{
    ServerInfo* it = KMLDonkey::App->donkey->findServerNo(fileno);
    if (!it) {
        if (!column) return i18n("Unknown server");
        return i18n("signifies absence of data in list columns", "-");
    }
    switch (column) {
    case 0: // name
        return it->serverName();
    case 1: // network
        return KMLDonkey::App->donkey->findNetworkNo(it->serverNetwork())->networkName();
    case 2: // status
        switch (it->serverState()) {
        case ServerInfo::NotConnected:
        case ServerInfo::NotConnected2: return i18n("not connected to host", "Not connected");
        case ServerInfo::Connecting: return i18n("connecting to host", "Connecting");
        case ServerInfo::Initiating: return i18n("initiating connection to host", "Initiating");
        case ServerInfo::Downloading: return i18n("downloading from host", "Downloading");
        case ServerInfo::Connected:
        case ServerInfo::Connected2: return i18n("connected to host", "Connected");
        case ServerInfo::NewHost: return i18n("new host added to list", "New host");
        case ServerInfo::Removed: return i18n("host has been removed", "Removed");
        case ServerInfo::Blacklisted: return i18n("host is blacklisted", "Blacklisted");
        default: return i18n("unknown host state", "Unknown");
        }
    case 3: // users
        return KGlobal::locale()->formatNumber(it->serverNUsers(), 0);
    case 4: // files
        return KGlobal::locale()->formatNumber(it->serverNFiles(), 0);
    case 5: // description
        return it->serverDescription();
    case 6: // address
        return i18n("hostname:port", "%1:%2").arg(it->serverAddress()).arg(it->serverPort());
    default:
        return "ERROR!";
    }
}

double ServerInfoItem::numeric( int col ) const
{
    ServerInfo* it = KMLDonkey::App->donkey->findServerNo(fileno);
    if (!it) return 0.0;
    switch (col) {
    case 3:
        return (double)it->serverNUsers();
    case 4:
        return (double)it->serverNFiles();
    default:
        return 0.0;
    }
}

bool ServerInfoItem::isNumeric(int col) const
{
    switch (col) {
    case 3:
    case 4:
        return true;
    default:
        return false;
    }
}

void ServerInfoItem::paintCell(QPainter* p, const QColorGroup& cg, int col, int w, int align)
{
    QColorGroup colgrp(cg);
    if (KMLDonkey::App->coloredViews) {
        ServerInfo* it = KMLDonkey::App->donkey->findServerNo(fileno);
        if (it) {
            switch (it->serverState()) {
            case ServerInfo::NotConnected:
                colgrp.setColor(QColorGroup::Text, KMLDonkey::App->colorServerNotConnected);
                break;
            case ServerInfo::Blacklisted:
                colgrp.setColor(QColorGroup::Text, KMLDonkey::App->colorServerBlacklisted);
                break;
            case ServerInfo::Connecting:
            case ServerInfo::Initiating:
                colgrp.setColor(QColorGroup::Text, KMLDonkey::App->colorServerConnecting);
                break;
            case ServerInfo::Connected:
            case ServerInfo::Connected2:
                colgrp.setColor(QColorGroup::Text, KMLDonkey::App->colorServerConnected);
                break;
            default:
                break;
            }
        }
    }

    p->save();
    InfoItem::paintCell(p, colgrp, col, w, align);
    p->restore();
}

// SharedFile

QString SharedFile::xtext( int column ) const
{
    ShareInfo* it = KMLDonkey::App->donkey->findShareNo(fileno);
    if (!it) {
        if (!column) return i18n("Unknown share");
        return i18n("signifies absence of data in list columns", "-");
    }
    switch (column) {
    case 0: // name
        return it->shareName();
    case 1: // network
        return KMLDonkey::App->donkey->findNetworkNo(it->shareNetwork())->networkName();
    case 2: // requests
        return KGlobal::locale()->formatNumber(it->shareRequests(), 0);
    case 3: // bytes
        return humanReadableSize(it->shareUploaded());
    default:
        return "ERROR!";
    }
}

double SharedFile::numeric( int col ) const
{
    ShareInfo* it = KMLDonkey::App->donkey->findShareNo(fileno);
    if (!it) return 0.0;
    switch (col) {
    case 2:
        return (double)it->shareRequests();
    case 3:
        return (double)it->shareUploaded();
    default:
        return 0.0;
    }
}

bool SharedFile::isNumeric(int col) const
{
    switch (col) {
    case 2:
    case 3:
        return true;
    default:
        return false;
    }
}

// UploadersFile

int UploadersFile::compare(QListViewItem* item, int col, bool ascending) const
{
    switch (col) {
    case 0:
    case 5:
    case 6: {
        bool ok;
        int i = text(col).toInt(&ok);
        if (ok) {
            int j = item->text(col).toInt(&ok);
            if (ok) return (i < j) ? -1 : (i > j) ? 1 : 0;
        }
    } break;
    default:
        break;
    }

    return KListViewItem::compare(item, col, ascending);
}

// ClientItem

void ClientItem::setFilesListed(bool listed)
{
    filesListed = listed; refresh();
}

QString ClientItem::xtext( int column ) const
{
    ClientInfo* it = KMLDonkey::App->donkey->findClientNo(fileno);
    if (!it) {
        if (!column) return i18n("Unknown client");
        return i18n("signifies absence of data in list columns", "-");
    }
    switch (column) {
    case 0: // name
        return it->clientName();
    case 1: // network
        return KMLDonkey::App->donkey->findNetworkNo(it->clientNetwork())->networkName();
    case 2: // type
        switch (it->clientType()) {
        case ClientInfo::NormalClient:
            return i18n("a normal client", "Normal");
        case ClientInfo::FriendClient:
            return i18n("a client on the friend list", "Friend");
        case ClientInfo::ContactClient:
            return i18n("a client on the contact list", "Contact");
        default:
            return i18n("client type is unknown", "Unknown");
        }
    case 3: // kind
        return it->clientKind();
    case 4: // state
        switch (it->clientState()) {
        case ClientInfo::NotConnected:
        case ClientInfo::NotConnected2: return i18n("not connected to host", "Not connected");
        case ClientInfo::Connecting: return i18n("connecting to host", "Connecting");
        case ClientInfo::Initiating: return i18n("initiating connection to host", "Initiating");
        case ClientInfo::Downloading: return filesListed ? i18n("this friend's file list is present", "Files listed")
                                          : i18n("downloading from host", "Downloading");
        case ClientInfo::Connected: return filesListed ? i18n("this friend's file list is present", "Files listed")
                                        : i18n("connected to host", "Connected");
        case ClientInfo::Connected2: return filesListed ? i18n("this friend's file list is present", "Files listed")
                                         : i18n("we're at this position in client's queue", "Queued: %1")
                                            .arg(KGlobal::locale()->formatNumber(it->clientQueuePosition()), 0);
        case ClientInfo::NewHost: return i18n("new host added to list", "New host");
        case ClientInfo::Removed: return i18n("host has been removed", "Removed");
        case ClientInfo::Blacklisted: return i18n("host is blacklisted", "Blacklisted");
        default: return i18n("unknown host state", "Unknown");
        }
    default:
        return "ERROR!";
    }
}

void ClientItem::paintCell(QPainter* p, const QColorGroup& cg, int col, int w, int align)
{
    QColorGroup colgrp(cg);
    if (KMLDonkey::App->coloredViews) {
        ClientInfo* cl = KMLDonkey::App->donkey->findClientNo(fileno);
        if (cl) {
            switch (cl->clientState()) {
            case ClientInfo::NotConnected:
            case ClientInfo::NotConnected2:
                colgrp.setColor(QColorGroup::Text, KMLDonkey::App->colorSourceNotConnected);
                break;
            case ClientInfo::Connecting:
            case ClientInfo::Initiating:
                colgrp.setColor(QColorGroup::Text, KMLDonkey::App->colorSourceConnecting);
                break;
            case ClientInfo::Downloading:
                colgrp.setColor(QColorGroup::Text, KMLDonkey::App->colorSourceDownloading);
                break;
            case ClientInfo::Connected:
            case ClientInfo::Connected2:
                colgrp.setColor(QColorGroup::Text, KMLDonkey::App->colorSourceQueued);
                break;
            case ClientInfo::Blacklisted:
                colgrp.setColor(QColorGroup::Text, KMLDonkey::App->colorSourceBlacklisted);
                break;
            default:
                break;
            }
        }
    }

    p->save();
    InfoItem::paintCell(p, colgrp, col, w, align);
    p->restore();
}

// ClientFile

QString ClientFile::xtext( int column ) const
{
    const ResultInfo* it = KMLDonkey::App->donkey->findClientFile(fileno);
    if (!it) {
        if (!column) return i18n("Unknown file");
        return i18n("signifies absence of data in list columns", "-");
    }
    switch (column) {
    case 0: // name
        return it->resultName();
    case 1: // network
        return KMLDonkey::App->donkey->findNetworkNo(it->resultNetwork())->networkName();
    case 2: // size
        return humanReadableSize(it->resultSize());
    case 3: // format
        return it->resultFormat();
    case 4: // comment
        return it->resultComment();
    case 5: // hash
        return FileInfo::md4ToString(it->resultMD4());
    default:
        return "ERROR!";
    }
}

double ClientFile::numeric( int col ) const
{
    const ResultInfo* it = KMLDonkey::App->donkey->findClientFile(fileno);
    if (!it) return 0.0;
    switch (col) {
    case 2:
        return (double)it->resultSize();
    default:
        return 0.0;
    }
}

bool ClientFile::isNumeric(int col) const
{
    switch (col) {
    case 2:
        return true;
    default:
        return false;
    }
}
