/* -*- c++ -*-
 *
 * search.cpp
 *
 * Copyright (C) 2003 Petter E. Stokke <gibreel@gibreel.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#include <qpainter.h>
#include <qlabel.h>
#include <qvbox.h>
#include <qgroupbox.h>
#include <qscrollview.h>
#include <qsplitter.h>
#include <qcheckbox.h>
#include <qlayout.h>
#include <qpopupmenu.h>
#include <qclipboard.h>
#include <qheader.h>
#include <qtimer.h>
#include <qregexp.h>

#include <kdebug.h>
#include <klocale.h>
#include <kconfig.h>
#include <klineedit.h>
#include <kpushbutton.h>
#include <kmessagebox.h>
#include <klistview.h>
#include <kcombobox.h>
#include <kaction.h>
#include <kmimetype.h>
#include <kiconloader.h>

#include "kmldonkey.h"
#include "search.h"
#include "searchquery.h"
#include "prefs.h"

#include <kdeversion.h>

#if KDE_IS_VERSION(3,1,90)
#include <ktabwidget.h>
#include <kinputdialog.h>
#else
#include "closabletab.h"
#include <klineeditdlg.h>
#endif

long int filesizeStr2Int(QString filesize)
{
    QString s = filesize.lower().stripWhiteSpace();
    if (s.isEmpty()) return 0;
    long int result;
    bool ok;

    if (s.endsWith("mb") || s.endsWith("kb")) s.remove(s.length()-1, 1);

    if (s.endsWith("m")) { // megabytes
        result = s.left(s.length()-1).stripWhiteSpace().toLong() * 1048576;
    }
    else if (s.endsWith("k")) { // kilobytes
        result = s.left(filesize.length()-1).stripWhiteSpace().toLong() * 1024;
    }
    else if (s.endsWith("b")) { // bytes
        result = s.left(filesize.length()-1).stripWhiteSpace().toLong();
    }
    else { // only numbers means bytes as well
        result = s.toLong(&ok);
        if (! ok) result = 0;
    }

    return (result < 0 ? 0 : result);
}

/*** SearchResultItem ***/

SearchResultItem::SearchResultItem(KListView *parent, int num, const ResultInfo *searchinfo) : InfoItem(parent,num)
{
    number = num;
    name = searchinfo->resultName();
    names = searchinfo->resultNames();
    filesize = searchinfo->resultSize();

    availability = 0;
    const QMap<QString,QVariant>& tags = searchinfo->resultTags();
    if(tags.contains("availability"))
        availability = tags["availability"].toInt();

    alreadydone = searchinfo->resultAlreadyDone();
    Network* nw = KMLDonkey::App->donkey->findNetworkNo( searchinfo->resultNetwork() );
    nwName = (nw) ? nw->networkName() : QString::number(searchinfo->resultNetwork());
    format = searchinfo->resultFormat();
    comment = searchinfo->resultComment();
    md4 = FileInfo::md4ToString(searchinfo->resultMD4());
    refresh();
}

SearchResultItem::~SearchResultItem()
{
}

QString SearchResultItem::xtext(int col) const
{
    switch (col) {
    case 0:
	return name;
    case 1:
	return humanReadableSize(filesize);
    case 2:
	return KGlobal::locale()->formatNumber(availability, 0);
    case 3:
	return nwName;
    case 4:
	return KMimeType::findByURL( "file:/" + name, 0, false, true)->comment();
    case 5:
	return format;
    case 6:
	return comment;
    case 7:
	return md4;
    default:
	return "ERROR";
    }
}

double SearchResultItem::numeric(int col) const
{
    switch (col) {
        case 1:  return (double)filesize;
        case 2:  return (double)availability;
        default: return 0.0;
    }
}

bool SearchResultItem::isNumeric(int col) const
{
    switch (col) {
        case 1:
        case 2: return true;
        default: return false;
    }
}

void SearchResultItem::paintCell(QPainter* p, const QColorGroup& cg, int col, int w, int align)
{
    QColorGroup colgrp(cg);
    if (KMLDonkey::App->coloredViews) {
        if (alreadydone) {
            colgrp.setColor(QColorGroup::Text, KMLDonkey::App->colorSearchAlreadyDone);
        }
        else if (availability < (int32)KMLDonkey::App->searchThreshold) {
            colgrp.setColor(QColorGroup::Text, KMLDonkey::App->colorSearchFewSources);
        }
        else {
            colgrp.setColor(QColorGroup::Text, KMLDonkey::App->colorSearchManySources);
        }
    }

    p->save();
    InfoItem::paintCell(p, colgrp, col, w, align);
    p->restore();
}

inline int SearchResultItem::getNum() { return number; }
inline QStringList SearchResultItem::getNames() { return names; }
inline int64 SearchResultItem::getSize() { return filesize; }

/*** SearchResultView ***/

SearchResultView::SearchResultView(QWidget *parent, const char *name)
    : InfoList(parent, name),
      QToolTip(viewport())
{
    addColumn( i18n("File Name"), 400);

    addColumn( i18n("Size"), -1, QListView::Maximum);

    addColumn( i18n("Availability") );
    addColumn( i18n("Network") );

    addColumn( i18n("File Type"), -1, QListView::Maximum);

    addColumn( i18n("Format") );
    addColumn( i18n("Comment") );

    addColumn( i18n("Hash"), -1, QListView::Maximum);
}

void SearchResultView::maybeTip(const QPoint& p)
{
    SearchResultItem* item = (SearchResultItem*)itemAt(p);
    if (! item) return;

    QRect r(itemRect(item));
    if (! r.isValid()) return;

    QHeader* h = header();
    if (h && h->mapToLogical( h->cellAt(p.x() - margin()) ) != 0) return;

    QString s;
    if (item->getNames().count() == 1)
        s = item->getNames()[0];
    else
        for (unsigned int i = 1; i < item->getNames().count(); i++) {
            if (i > 1) s += "<br>";
            s += "<nobr>" + item->getNames()[i] + "</nobr>";
        }

    tip(r, s);
}

/*** SearchResult ***/

SearchResult::SearchResult(int sno, const QString &tabLabel, QWidget *parent) : QVBox(parent, "searchResult")
{
    id = sno;
    this->tabLabel = tabLabel;
    visibleCount = 0;

    statusBox = new QHBox(this, "resultBox");

    QHBox *filterbox = new QHBox(statusBox, "filterBox");
    new QLabel(i18n("Filter"), filterbox);
    filterEdit = new KLineEdit(filterbox);
    connect(filterEdit, SIGNAL( textChanged(const QString &) ), this, SLOT( filterChanged() ));
    filterTimer = new QTimer(this);
    connect(filterTimer, SIGNAL( timeout() ), this, SLOT( filterTimerDone() ));

    statusLabel = new QLabel(i18n("zero of zero search results", "0/0"), statusBox);
    statusLabel->setFrameStyle(QFrame::Sunken + QFrame::Box);
    searchLabel = new QLabel(statusBox);
    searchLabel->setFrameStyle(QFrame::Sunken + QFrame::Box);
    searchLabel->setAlignment(Qt::WordBreak);
    statusBox->setStretchFactor(searchLabel,1);

    resultView = new SearchResultView(this, "searchResult");
    connect(resultView, SIGNAL(contextMenu(KListView*, QListViewItem*, const QPoint&)),
            this, SLOT(contextSearchResult(KListView*, QListViewItem*, const QPoint&)));
    connect(resultView, SIGNAL(doubleClicked(QListViewItem*)),
            this, SLOT(doubleClick(QListViewItem*)));
}

SearchResult::~SearchResult()
{
}

void SearchResult::contextSearchResult(KListView*,QListViewItem*,const QPoint& pt)
{
    QPopupMenu *pop = (QPopupMenu*)(KMLDonkey::App->factory())->container("search_actions", KMLDonkey::App);
    if (!pop)
	KMLDonkey::App->showBadInstallDialog();
    else
	pop->popup(pt);
}

void SearchResult::doubleClick(QListViewItem*)
{
    DownloadSelectedItems(false, true);
}

inline int SearchResult::searchNo() { return id; }
inline const QString& SearchResult::getTabLabel() { return tabLabel; }
inline void SearchResult::setSearchLabel(const QString &text) { searchLabel->setText(text); }
inline void SearchResult::setListFont(const QFont& font) { resultView->QListView::setFont(font); }

void SearchResult::AddItem(const ResultInfo *searchinfo)
{
    SearchResultItem* item = new SearchResultItem(resultView, searchinfo->resultNo(), searchinfo);

    bool visible = ! filterItem(item);
    if (visible) visibleCount++;
    item->setVisible(visible);

    setStatusLabel();
}

void SearchResult::DownloadSelectedItems(bool force, bool askForFilename)
{
    QStringList names;
    QString filename;
    bool ok;
    QPtrList<QListViewItem> list = resultView->selectedItems();
    SearchResultItem *item;
    for (item = (SearchResultItem*)list.first(); item; item = (SearchResultItem*)list.next()) {

        filename = item->text(0);
        if (askForFilename || ! filename.length()) {
#if KDE_IS_VERSION(3,1,90)
            filename = KInputDialog::getText( i18n("Download As"),
#else
            filename = KLineEditDlg::getText( i18n("Download As"),
#endif
                                              i18n("Choose a filename for the new download:"),
                                              filename, &ok, this );
            if (! ok) continue;
        }

        names.clear();
        names.append(filename);
        if (filename != item->text(0)) names.append(item->text(0));
        KMLDonkey::App->donkey->startDownload(names, item->getNum(), force);
    }
}

void SearchResult::filterChanged()
{
    filterTimer->stop();
    if (resultView->childCount()) filterTimer->start(500, true);
}

void SearchResult::filterTimerDone()
{
    visibleCount = 0;
    filters = QStringList::split(" ", filterEdit->text().lower().stripWhiteSpace().simplifyWhiteSpace() );
    QListViewItemIterator it(resultView);
    bool visible;
    for ( ; it.current(); ++it ) {
        visible = ! filterItem( (SearchResultItem*)it.current() );
        if(visible) visibleCount++;
        it.current()->setVisible(visible);
        if (it.current()->isSelected() && ! visible) it.current()->setSelected(false);
    }
    setStatusLabel();
}

inline void SearchResult::setStatusLabel()
{
    statusLabel->setText( i18n("visible/total search results", "%1/%2").arg(KGlobal::locale()->formatNumber(visibleCount, 0))
			  .arg(KGlobal::locale()->formatNumber(resultView->childCount(), 0)) );
}

bool SearchResult::filterItem(SearchResultItem *item)
{
    unsigned int i;
    bool minsize;
    int64 size;
    for (i = 0; i < filters.count(); i++) {
        minsize = filters[i].startsWith(">");
        if (minsize || filters[i].startsWith("<")) { // filter max/min filesize
            size = filesizeStr2Int( filters[i].right(filters[i].length() - 1) );
            if (minsize) {
                if (item->getSize() <= size) return true;
            }
            else {
                if (item->getSize() >= size) return true;
            }
        }
        else { // filter filename
            if (filters[i].startsWith("~")) { // regular expression
                QRegExp rx( filters[i].right(filters[i].length() - 1) );
                if (rx.search( item->text(0) ) < 0) return true;
            }
            else if (! item->text(0).contains(filters[i], false))
                return true;
        }
    }
    return false;
}

/*** SearchPage ***/

SearchPage::SearchPage(QWidget *parent)
    : QVBox(parent, "searchPage")
    , KMLDonkeyPage()
    , ClipboardHelper()
{
    setMargin(6);
    searchNum = 0;
    maxHits = 500;
    searchType = 0;

    splitter = new QSplitter(this);
    QScrollView *scroll = new QScrollView(splitter);
    scroll->setResizePolicy(QScrollView::AutoOneFit);
    scroll->viewport()->setBackgroundMode(PaletteBackground);
    splitter->setResizeMode(scroll, QSplitter::KeepSize);
    QVBox *box = new QVBox(scroll->viewport());
    scroll->addChild(box);
    box->setMargin(6);

    QVBox* rightBox = new QVBox(splitter);
    ed2kBox = new QHBox(rightBox);
    ed2kBox->setMargin(6);
    ed2kBox->setSpacing(6);
    QLabel* ed2kLabel = new QLabel(i18n("Enter &URL to download:"), ed2kBox);
    ed2kEntry = new KLineEdit(ed2kBox, "ed2kEntry");
    ed2kLabel->setBuddy(ed2kEntry);
    connect(ed2kEntry, SIGNAL(returnPressed()), SLOT(submitED2K()));

#if KDE_IS_VERSION(3,1,90)
    resultTabs = new KTabWidget(rightBox);
    resultTabs->setHoverCloseButton(true);
    resultTabs->setTabReorderingEnabled(true);
#else
    resultTabs = new ClosableTabWidget(rightBox);
#endif
    connect(resultTabs, SIGNAL(closeRequest(QWidget*)), this, SLOT(closeSearch(QWidget*)));

    // Keywords

    QHBox *kwbox = new QHBox(box);
    kwbox->setSpacing(2);
    new QLabel(i18n("Keywords:"), kwbox);
    keywordsEdit = new KLineEdit(kwbox);
    connect(keywordsEdit, SIGNAL( returnPressed() ), this, SLOT( startSearch() ));

    // Simple Options

    QGroupBox *simplebox = new QGroupBox(box);
    simplebox->setTitle(i18n("Simple Options"));
    simplebox->setColumnLayout(0, Qt::Vertical);
    simplebox->layout()->setSpacing(2);
    QGridLayout *simplelayout = new QGridLayout(simplebox->layout());

    QHBox *minsizebox = new QHBox(simplebox);
    simplelayout->addWidget(minsizebox , 0, 0);
    new QLabel(i18n("Min size:"), minsizebox);
    minsizeCombo = new KComboBox(minsizebox);
    minsizeCombo->setEditable(true);
    minsizeCombo->insertItem(QString(""), 0);
    minsizeCombo->insertItem("500 MB", 1);
    minsizeCombo->insertItem("100 MB", 2);
    minsizeCombo->insertItem("50 MB", 3);
    minsizeCombo->insertItem("3 MB", 4);
    minsizeCombo->insertItem("500 KB", 5);
    minsizeCombo->insertItem("500 B", 6);

    QHBox *maxsizebox = new QHBox(simplebox);
    simplelayout->addWidget(maxsizebox , 1, 0);
    new QLabel(i18n("Max size:"), maxsizebox);
    maxsizeCombo = new KComboBox(maxsizebox);
    maxsizeCombo->setEditable(true);
    maxsizeCombo->insertItem(QString(""), 0);
    maxsizeCombo->insertItem("500 MB", 1);
    maxsizeCombo->insertItem("100 MB", 2);
    maxsizeCombo->insertItem("50 MB", 3);
    maxsizeCombo->insertItem("3 MB", 4);
    maxsizeCombo->insertItem("500 KB", 5);
    maxsizeCombo->insertItem("500 B", 6);

    QHBox *mediabox = new QHBox(simplebox);
    simplelayout->addWidget(mediabox , 2, 0);
    new QLabel(i18n("Media:"), mediabox);
    mediaCombo = new KComboBox(mediabox);
    mediaCombo->setEditable(true);
    mediaCombo->insertItem(QString(""), 0);
    mediaCombo->insertItem("Audio", 1);
    mediaCombo->insertItem("Video", 2);
    mediaCombo->insertItem("Program", 3);
    mediaCombo->insertItem("Image", 4);
    mediaCombo->insertItem("Documentation", 5);
    mediaCombo->insertItem("Collection", 6);

    QHBox *formatbox = new QHBox(simplebox);
    simplelayout->addWidget(formatbox , 3, 0);
    new QLabel(i18n("file format", "Format:"), formatbox);
    formatCombo = new KComboBox(formatbox);
    formatCombo->setEditable(true);
    formatCombo->insertItem(QString(""), 0);
    formatCombo->insertItem("avi", 1);
    formatCombo->insertItem("mp3", 2);

    // MP3 Options

    QGroupBox *mp3box = new QGroupBox(box);
    mp3box->setTitle(i18n("MP3 Options"));
    mp3box->setColumnLayout(0, Qt::Vertical);
    mp3box->layout()->setSpacing(2);
    QGridLayout *mp3layout = new QGridLayout(mp3box->layout());

    QHBox *artistbox = new QHBox(mp3box);
    artistbox->setSpacing(2);
    mp3layout->addWidget(artistbox, 0, 0);
    new QLabel(i18n("Artist:"), artistbox);
    mp3ArtistEdit = new KLineEdit(artistbox);
    connect(mp3ArtistEdit, SIGNAL( returnPressed() ), this, SLOT( startSearch() ));

    QHBox *titlebox = new QHBox(mp3box);
    titlebox->setSpacing(2);
    mp3layout->addWidget(titlebox, 1, 0);
    new QLabel(i18n("Title:"), titlebox);
    mp3TitleEdit = new KLineEdit(titlebox);
    connect(mp3TitleEdit, SIGNAL( returnPressed() ), this, SLOT( startSearch() ));

    QHBox *albumbox = new QHBox(mp3box);
    albumbox->setSpacing(2);
    mp3layout->addWidget(albumbox, 2, 0);
    new QLabel(i18n("Album:"), albumbox);
    mp3AlbumEdit = new KLineEdit(albumbox);
    connect(mp3AlbumEdit, SIGNAL( returnPressed() ), this, SLOT( startSearch() ));

    QHBox *bitratebox = new QHBox(mp3box);
    mp3layout->addWidget(bitratebox , 3, 0);
    new QLabel(i18n("Bit rate:"), bitratebox);
    mp3BitrateCombo = new KComboBox(bitratebox);
    mp3BitrateCombo->setEditable(true);
    mp3BitrateCombo->insertItem(QString(""), 0);
    mp3BitrateCombo->insertItem("64", 1);
    mp3BitrateCombo->insertItem("96", 2);
    mp3BitrateCombo->insertItem("128", 3);
    mp3BitrateCombo->insertItem("160", 4);
    mp3BitrateCombo->insertItem("192", 5);

    // Max Hits

    QHBox *mhbox = new QHBox(box);
    mhbox->setSpacing(2);
    new QLabel(i18n("Max hits:"), mhbox);
    maxhitsEdit = new KLineEdit(mhbox);
    maxhitsEdit->setMaxLength(4);
    maxhitsEdit->setText( QString::number(maxHits) );

    // Type

    QHBox *stbox = new QHBox(box);
    new QLabel(i18n("search type", "Type:"), stbox);
    searchtypeCombo = new KComboBox(stbox);
    searchtypeCombo->insertItem(i18n("remote search", "Remote"), 0);
    searchtypeCombo->insertItem(i18n("local search", "Local"), 1);
    searchtypeCombo->insertItem(i18n("subscription search", "Subscribe"), 2);
    searchtypeCombo->setCurrentItem(searchType);

    // Network

    QHBox *snbox = new QHBox(box);
    new QLabel(i18n("Network:"), snbox);
    searchnetCombo = new KComboBox(snbox);
    searchnetCombo->insertItem(i18n("All Networks"));
    searchnetCombo->setCurrentItem(0);
    connect(KMLDonkey::App->donkey, SIGNAL( networkUpdated(int) ), this, SLOT( setNetworks(int) ));

    KPushButton *startButton = new KPushButton(i18n("start a search", "Search"), box);
    connect(startButton, SIGNAL( clicked() ), this, SLOT( startSearch() ));

    scroll->setMinimumSize( box->sizeHint() );

    connect(KMLDonkey::App->donkey, SIGNAL(searchUpdated(int,const ResultInfo*)), this, SLOT(searchUpdated(int,const ResultInfo*)));
}

void SearchPage::handleGenericAction(const QString& action)
{
    if (action == "copy_url") copySearchToClipboard(URL);
    else if (action == "copy_html") copySearchToClipboard(HTML);
    else if (action == "copy_hash") copySearchToClipboard(Hash);
    else if (action == "search_download") actionDownload();
    else if (action == "search_force_download") actionForceDownload();
    else if (action == "search_download_as") actionDownloadAs();
}

void SearchPage::deactivatePageActions()
{
}

QStringList SearchPage::supportedGenericActions()
{
    QStringList actions;
    SearchResult *sr = (SearchResult*)resultTabs->currentPage();

    if (sr && sr->resultView->hasFocus() && !sr->resultView->selectedItems().isEmpty()) {
	actions.append("copy_url");
	actions.append("copy_html");
	actions.append("copy_hash");
	actions.append("search_download");
	actions.append("search_force_download");
	actions.append("search_download_as");
    }

    return actions;
}

void SearchPage::plugGenericActions(QObject* object, const char* slot)
{
    connect(this, SIGNAL(genericActionsChanged(KMLDonkeyPage*)), object, slot);
}

void SearchPage::pleaseUpdateActions()
{
    emit genericActionsChanged(static_cast<KMLDonkeyPage*>(this));
}

void SearchPage::setupActions(KActionCollection* actionCollection)
{
    (void)new KAction(i18n("Activate Search Page"), 0, 0, this, SLOT(actionActivatePage()),
		      actionCollection, "activate_page_search");
}

void SearchPage::configurePrefsDialog(KMLDonkeyPreferences* prefs)
{
    prefs->searchPage->activateNewTabsCheckbox->setChecked(activateNewTabs);
    prefs->searchPage->closeTabsOnDisconnectCheckbox->setChecked(closeTabsOnDisconnect);
    prefs->searchPage->showNumbersOnTabsCheckbox->setChecked(showNumbersOnTabs);
    prefs->searchPage->showSubmitURLCheckbox->setChecked(showSubmitURL);
}

void SearchPage::applyPreferences(KMLDonkeyPreferences* prefs)
{
    activateNewTabs = prefs->searchPage->activateNewTabsCheckbox->isChecked();
    closeTabsOnDisconnect = prefs->searchPage->closeTabsOnDisconnectCheckbox->isChecked();
    showNumbersOnTabs = prefs->searchPage->showNumbersOnTabsCheckbox->isChecked();

    if (showSubmitURL != prefs->searchPage->showSubmitURLCheckbox->isChecked()) {
        showSubmitURL = prefs->searchPage->showSubmitURLCheckbox->isChecked();
        if (showSubmitURL)
            ed2kBox->show();
        else
            ed2kBox->hide();
        ed2kBox->updateGeometry();
    }

    QIntDictIterator<SearchResult> it( Results );
    for ( ; it.current(); ++it ) {
        it.current()->setListFont(KMLDonkey::App->listFont);
        resultTabs->setTabLabel(it.current(), getTabLabel(it.current()));
    }
}

inline QString SearchPage::getTabLabel(SearchResult *tab)
{
    if (! showNumbersOnTabs) return tab->getTabLabel();
    return tab->getTabLabel() + " (" + QString::number(tab->resultView->childCount()) + ")";
}

void SearchPage::saveState(KConfig* conf)
{
    conf->setGroup("Search");

    conf->writeEntry("Splitter", splitter->sizes());

    bool ok;
    int hits = maxhitsEdit->text().toInt(&ok);
    if (ok && hits) conf->writeEntry("maxHits", hits);

    conf->writeEntry("Type", searchtypeCombo->currentItem());
    conf->writeEntry("Network", searchnetCombo->currentText());

    conf->writeEntry("activateNewTabs", activateNewTabs);
    conf->writeEntry("closeTabsOnDisconnect", closeTabsOnDisconnect);
    conf->writeEntry("showNumbersOnTabs", showNumbersOnTabs);

    conf->writeEntry("showSubmitURL", showSubmitURL);
}

void SearchPage::restoreState(KConfig* conf)
{
    conf->setGroup("Search");

    splitter->setSizes(conf->readIntListEntry("Splitter"));
    int i = conf->readNumEntry("maxHits", maxHits);
    if (i && i != maxHits) {
        maxHits = i;
        maxhitsEdit->setText( QString::number(maxHits) );
    }
    i = conf->readNumEntry("Type", searchType);
    if (i && i <= 2) {
        searchType = i;
        searchtypeCombo->setCurrentItem(searchType);
    }
    searchNetwork = conf->readEntry("Network", QString::null);

    activateNewTabs = conf->readBoolEntry("activateNewTabs", true);
    closeTabsOnDisconnect = conf->readBoolEntry("closeTabsOnDisconnect", true);
    showNumbersOnTabs = conf->readBoolEntry("showNumbersOnTabs", true);

    showSubmitURL = conf->readBoolEntry("showSubmitURL", true);
    if (showSubmitURL)
        ed2kBox->show();
    else
        ed2kBox->hide();
    ed2kBox->updateGeometry();
}

void SearchPage::clear()
{
    searchnetCombo->setCurrentItem(0);
    for (int i = searchnetCombo->count() - 1; i > 0; i--) searchnetCombo->removeItem(i);
    if (closeTabsOnDisconnect) closeAllSearches();
}

void SearchPage::submitED2K()
{
    KMLDonkey::App->donkey->submitURL(ed2kEntry->text());
    ed2kEntry->clear();
}

void SearchPage::setNetworks(int no)
{
    if (KMLDonkey::App->donkey->protocolVersion() < 16) return;
    Network *net = KMLDonkey::App->donkey->findNetworkNo(no);
    if (! net) return;
    for (int i = searchnetCombo->count() - 1; i > 0; i--)
        if (searchnetCombo->text(i) == net->networkName()) { // already on the list?
            if (! net->networkEnabled())
                searchnetCombo->removeItem(i);
            return;
        }
    if (! net->networkEnabled()) return;
    if (!(net->networkFlags() & Network::NetworkHasSearch)) return;
    searchnetCombo->insertItem( net->networkName() );
    if (searchNetwork == net->networkName())
        searchnetCombo->setCurrentItem(searchnetCombo->count() - 1);
}

void SearchPage::startSearch()
{
    if (! KMLDonkey::App->donkey->isConnected()) return;

    QString keywords = keywordsEdit->text().stripWhiteSpace();

    bool ok;
    int hits = maxhitsEdit->text().toInt(&ok);
    if (ok && hits) {
        maxHits = hits;
    }
    else {
        hits = maxHits;
        maxhitsEdit->setText( QString::number(hits) );
    }

    DonkeyProtocol::SearchType type;
    switch (searchtypeCombo->currentItem()) {
        case 1:  type = DonkeyProtocol::LocalSearch; break;
        case 2:  type = DonkeyProtocol::SubscribeSearch; break;
        default: type = DonkeyProtocol::RemoteSearch; break;
    }

    long int minsize = filesizeStr2Int( minsizeCombo->currentText() );
    long int maxsize = filesizeStr2Int( maxsizeCombo->currentText() );
    QString format = formatCombo->currentText().stripWhiteSpace();
    QString media = mediaCombo->currentText().stripWhiteSpace();
    QString mp3artist = mp3ArtistEdit->text().stripWhiteSpace();
    QString mp3title = mp3TitleEdit->text().stripWhiteSpace();
    QString mp3album = mp3AlbumEdit->text().stripWhiteSpace();
    QString mp3bitrate = mp3BitrateCombo->currentText().stripWhiteSpace();

    QString caption, label;
    if (! keywords.isEmpty()) {
        caption = keywords + " ";
        label   = i18n("Keywords=\"%1\" ").arg(keywords);
    }
    if (! mp3artist.isEmpty()) {
        caption += mp3artist + " ";
        label   += i18n("Artist=\"%1\" ").arg(mp3artist);
    }
    if (! mp3title.isEmpty()) {
        caption += mp3title + " ";
        label   += i18n("Title=\"%1\" ").arg(mp3title);
    }
    if (! mp3album.isEmpty()) {
        caption += mp3album + " ";
        label   += i18n("Album=\"%1\" ").arg(mp3album);
    }
    if(! mp3bitrate.isEmpty()) label += i18n("Bitrate=%1 ").arg(mp3bitrate);
    if(! format.isEmpty()) label += i18n("Format=%1 ").arg(format);
    if(! media.isEmpty()) label += i18n("Media=%1 ").arg(media);
    if(minsize > 0) label += i18n("Min size=%1 ").arg(minsize);
    if(maxsize > 0) label += i18n("Max size=%1 ").arg(maxsize);

    if (caption.isEmpty()) return;

    // Construct the query
    QueryAnd* qa = new QueryAnd();
    qa->append(new QueryKeywords((QString&)QString::null, keywords));
    if (minsize) qa->append(new QueryMinSize(QString::null, QString::number(minsize)));
    if (maxsize) qa->append(new QueryMaxSize(QString::null, QString::number(maxsize)));
    if (!format.isEmpty()) qa->append(new QueryFormat(QString::null, format));
    if (!media.isEmpty()) qa->append(new QueryMedia(QString::null, media));
    if (!mp3artist.isEmpty()) qa->append(new QueryMp3Artist(QString::null, mp3artist));
    if (!mp3title.isEmpty()) qa->append(new QueryMp3Title(QString::null, mp3title));
    if (!mp3album.isEmpty()) qa->append(new QueryMp3Album(QString::null, mp3album));
    if (!mp3bitrate.isEmpty()) qa->append(new QueryMp3Bitrate(QString::null, mp3bitrate));

    // Simplify the query if possible
    SearchQuery* q;
    if (qa->count() == 1)
        q = qa->take(0);
    else
        q = qa;

    // Network
    int network = 0;
    if (searchnetCombo->currentItem() > 0) {
        Network *net;
        QIntDictIterator<Network> it( KMLDonkey::App->donkey->availableNetworks() );
        for ( ; it.current(); ++it ) {
            net = it.current();
            if (net->networkName() == searchnetCombo->currentText()) {
                network = net->networkNo();
                label += i18n("Network=%1").arg(net->networkName());
                break;
            }
        }
    }

    // create a SearchResult
    searchNum++;
    SearchResult *sr = new SearchResult(searchNum, caption.simplifyWhiteSpace(), resultTabs);
    connect(sr->resultView, SIGNAL(selectionChanged()), SLOT(pleaseUpdateActions()));
    connect(sr->resultView, SIGNAL(gotFocus()), SLOT(pleaseUpdateActions()));
    Results.replace(searchNum, sr);
    sr->setListFont(KMLDonkey::App->listFont);
    sr->setSearchLabel(label.simplifyWhiteSpace());
    resultTabs->addTab(sr, KGlobal::iconLoader()->loadIconSet("fileclose", KIcon::Small), getTabLabel(sr));
    if (activateNewTabs) resultTabs->setCurrentPage(resultTabs->count()-1);

    // now let's start the search
    KMLDonkey::App->donkey->startSearch(searchNum, q, maxHits, type, network);

    delete qa;
}

void SearchPage::searchUpdated(int searchnum, const ResultInfo* searchinfo)
{
    SearchResult *sr = Results[searchnum];
    if (! sr) return;
    sr->AddItem(searchinfo);
    resultTabs->setTabLabel(sr, getTabLabel(sr));
}

void SearchPage::stopSearch(int id)
{
    if (KMLDonkey::App->donkey->isConnected())
        KMLDonkey::App->donkey->stopSearch(id);
    Results.remove(id);
}

void SearchPage::stopAllSearches()
{
    if (! KMLDonkey::App->donkey->isConnected()) return;
    QIntDictIterator<SearchResult> it( Results );
    for ( ; it.current(); ++it )
        KMLDonkey::App->donkey->stopSearch( it.currentKey() );
}

void SearchPage::closeSearch(QWidget* widget)
{
    int id = static_cast<SearchResult*>(widget)->searchNo();
    stopSearch(id);
    delete widget;
    pleaseUpdateActions();
}

void SearchPage::closeAllSearches()
{
    while (resultTabs->count()) {
        int id = static_cast<SearchResult*>( resultTabs->currentPage() )->searchNo();
        stopSearch(id);
        delete resultTabs->currentPage();
    }
    pleaseUpdateActions();
}

void SearchPage::actionDownload()
{
    SearchResult *sr = (SearchResult*)resultTabs->currentPage();
    if (sr) sr->DownloadSelectedItems(false, false);
}

void SearchPage::actionForceDownload()
{
    SearchResult *sr = (SearchResult*)resultTabs->currentPage();
    if (sr) sr->DownloadSelectedItems(true, false);
}

void SearchPage::actionDownloadAs()
{
    SearchResult *sr = (SearchResult*)resultTabs->currentPage();
    if (sr) sr->DownloadSelectedItems(false, true);
}

void SearchPage::copySearchToClipboard(ClipFormat format)
{
    SearchResult *sr = (SearchResult*)resultTabs->currentPage();
    if (! sr) return;
    QStringList sl;
    QPtrList<QListViewItem> list = sr->resultView->selectedItems();
    SearchResultItem *item;
    for (item = (SearchResultItem*)list.first(); item; item = (SearchResultItem*)list.next()) {
        sl.append( item->text(0) ); // filename
        sl.append( item->text(7) ); // filehash
        sl.append( QString::number((long)item->getSize()) ); // filesize
    }
    copyToClipboard(sl, format);
}

void SearchPage::actionActivatePage()
{
    KMLDonkey::App->activatePage(this);
}


#include "search.moc"
