/* -*- c++ -*-
 *
 * sharepage.cpp
 *
 * Copyright (C) 2003 Petter E. Stokke <gibreel@gibreel.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#include <kdebug.h>
#include <klocale.h>
#include <kconfig.h>
#include <kaction.h>
#include <knuminput.h>

#include <qintdict.h>
#include <qpopupmenu.h>
#include <qclipboard.h>
#include <qregexp.h>
#include <qtimer.h>
#include <qcheckbox.h>

#include "infolist.h"
#include "prefs.h"
#include "kmldonkey.h"
#include "shareddirs.h"

#include "sharepage.h"
#include "sharepage.moc"

SharePage::SharePage(QWidget* parent)
    : QVBox(parent, "sharePage")
    , KMLDonkeyPage()
    , ClipboardHelper()
{
    shareList.setAutoDelete(true);
    uploadersCallback = 0;

    refreshTimer = new QTimer(this, "refreshTimer");

    uploadSplitter = new QSplitter(this, "uploadSplitter");
    uploadSplitter->setOrientation(QSplitter::Vertical);

    uploadView = new InfoList(uploadSplitter, "uploadView");
    uploadView->addColumn( i18n( "File Name" ) );
    uploadView->addColumn( i18n( "Network" ) );
    uploadView->addColumn( i18n( "Requests" ) );
    uploadView->addColumn( i18n( "Bytes" ) );

    uploadersView = new InfoList(uploadSplitter, "uploadersView");
    uploadersView->addColumn( i18n( "ID" ) );
    uploadersView->addColumn( i18n( "Network" ) );
    uploadersView->addColumn( i18n( "Client" ) );
    uploadersView->addColumn( i18n( "Address" ) );
    uploadersView->addColumn( i18n( "User Name" ) );
    uploadersView->addColumn( i18n( "Uploaded" ) );
    uploadersView->addColumn( i18n( "Downloaded" ) );
    uploadersView->addColumn( i18n( "File Name" ) );
    uploadersView->setSelectionMode(QListView::NoSelection);

    connect(uploadView, SIGNAL(contextMenu(KListView*, QListViewItem*, const QPoint&)),
	    this, SLOT(contextUpload(KListView*, QListViewItem*, const QPoint&)));
    connect(uploadView, SIGNAL(selectionChanged()), SLOT(clearUploadersSelection()));

    connect(uploadersView, SIGNAL(contextMenu(KListView*, QListViewItem*, const QPoint&)),
	    this, SLOT(contextUploaders(KListView*, QListViewItem*, const QPoint&)));
    connect(uploadersView, SIGNAL(selectionChanged()), SLOT(clearUploadSelection()));

    connect(KMLDonkey::App->donkey, SIGNAL(shareUpdated(int)), this, SLOT(shareUpdated(int)));
    connect(KMLDonkey::App->donkey, SIGNAL(shareRemoved(int)), this, SLOT(shareRemoved(int)));

    connect(KMLDonkey::App->donkey, SIGNAL(signalConnected()), this, SLOT(actionRefreshUploaders()));

    connect(refreshTimer, SIGNAL(timeout()), SLOT(refreshLists()));
}

void SharePage::handleGenericAction(const QString& action)
{
    if (uploadView->hasFocus()) {
	if (action == "copy_url") copyFileToClipboard(URL);
	else if (action == "copy_html") copyFileToClipboard(HTML);
	else if (action == "copy_hash") copyFileToClipboard(Hash);
    }

    else if (uploadersView->hasFocus()) {
    }
}

void SharePage::showEvent(QShowEvent*)
{
    if (KMLDonkey::App->donkey->isConnected() && ! autoUpdate) {
        actionRefreshShared();
        actionRefreshUploaders();
    }
}

void SharePage::deactivatePageActions()
{
    enableActionList(uploadActions, false);
    enableActionList(uploadersActions, false);
}

QStringList SharePage::supportedGenericActions()
{
    QStringList actions;
    QPtrList<QListViewItem> ul = uploadView->selectedItems();
    QPtrList<QListViewItem> sl = uploadersView->selectedItems();

    if (!ul.isEmpty()) {
	enableActionList(uploadActions, true);
	actions.append("copy_url");
	actions.append("copy_html");
	actions.append("copy_hash");
    }
    else enableActionList(uploadActions, false);

    if (!sl.isEmpty()) {
	enableActionList(uploadersActions, true);
    }
    else enableActionList(uploadersActions, false);

    return actions;
}

void SharePage::plugGenericActions(QObject* object, const char* slot)
{
    connect(this, SIGNAL(genericActionsChanged(KMLDonkeyPage*)), object, slot);
}

void SharePage::clearUploadSelection()
{
    uploadView->blockSignals(true);
    uploadView->clearSelection();
    uploadView->blockSignals(false);
    emit genericActionsChanged(static_cast<KMLDonkeyPage*>(this));
}

void SharePage::clearUploadersSelection()
{
    uploadersView->blockSignals(true);
    uploadersView->clearSelection();
    uploadersView->blockSignals(false);
    emit genericActionsChanged(static_cast<KMLDonkeyPage*>(this));
}

void SharePage::setupActions(KActionCollection* actionCollection)
{
    refreshShared = new KAction(i18n("&Refresh Shared File List"), "reload", 0, this, SLOT(actionRefreshShared()),
				actionCollection, "upload_refresh");
    refreshUploaders = new KAction(i18n("Refresh U&ploaders List"), "reload", 0, this, SLOT(actionRefreshUploaders()),
				   actionCollection, "uploaders_refresh");

    KMLDonkey::App->addCoreAction( new KAction(i18n("&Shared Directories..."), "fileimport", 0, this, SLOT(actionSharedDirs()),
		                   actionCollection, "shared_dirs") );

    (void)new KAction(i18n("Activate Uploads Page"), 0, 0, this, SLOT(actionActivatePage()),
		      actionCollection, "activate_page_uploads");

    deactivatePageActions();
}

void SharePage::clear()
{
    shareList.clear();
    uploadView->clear();
    uploadersView->clear();
}

void SharePage::configurePrefsDialog(KMLDonkeyPreferences* prefs)
{
    prefs->listsPage->autoUpdateCheckbox->setChecked(autoUpdate);
    prefs->listsPage->updateFrequencyEntry->setValue(updateFrequency);
}

void SharePage::applyPreferences(KMLDonkeyPreferences* prefs)
{
    if (prefs) {
	autoUpdate = prefs->listsPage->autoUpdateCheckbox->isChecked();
	updateFrequency = prefs->listsPage->updateFrequencyEntry->value();
    }

    updateTimer();

    if (KMLDonkey::App->listFont !=  uploadView->font())
        uploadView->setFont(KMLDonkey::App->listFont);
    if (KMLDonkey::App->listFont !=  uploadersView->font())
        uploadersView->setFont(KMLDonkey::App->listFont);
}

void SharePage::restoreState(KConfig* conf)
{
    conf->setGroup("Options");
    autoUpdate = conf->readBoolEntry("AutoUpdateShareLists", false);
    updateFrequency = conf->readNumEntry("ShareListUpdateFrequency", 10);

    conf->setGroup("Splitters");
    uploadSplitter->setSizes(conf->readIntListEntry("UploadSplitter"));

    uploadView->initialise(conf, "UploadView");
    uploadersView->initialise(conf, "UploadersView");

    applyPreferences();
}

void SharePage::saveState(KConfig* conf)
{
    conf->setGroup("Options");
    conf->writeEntry("AutoUpdateShareLists", autoUpdate);
    conf->writeEntry("ShareListUpdateFrequency", updateFrequency);

    conf->setGroup("Splitters");
    conf->writeEntry("UploadSplitter", uploadSplitter->sizes());

    uploadView->finalise(conf, "UploadView");
    uploadersView->finalise(conf, "UploadersView");
}

void SharePage::contextUpload(KListView*,QListViewItem*,const QPoint& pt)
{
    QPopupMenu *pop = (QPopupMenu*)(KMLDonkey::App->factory())->container("upload_actions", KMLDonkey::App);
    if (!pop)
	KMLDonkey::App->showBadInstallDialog();
    else
	pop->popup(pt);
}

void SharePage::contextUploaders(KListView*,QListViewItem*,const QPoint& pt)
{
    QPopupMenu *pop = (QPopupMenu*)(KMLDonkey::App->factory())->container("uploaders_actions", KMLDonkey::App);
    if (!pop)
	KMLDonkey::App->showBadInstallDialog();
    else
	pop->popup(pt);
}

void SharePage::actionRefreshShared()
{
    KMLDonkey::App->donkey->refreshShared();
}

void SharePage::actionSharedDirs()
{
    if (! KMLDonkey::App->donkey->isConnected()) return;
    SharedDirsDialog *dlg = new SharedDirsDialog(this, "SharedDirsDialog");
    dlg->show();
}

void SharePage::shareUpdated(int share)
{
    ShareInfo* si = KMLDonkey::App->donkey->findShareNo(share);
    if (!si) return;

    SharedFile* df = shareList[share];
    if (df) {
        df->refresh();
        return;
    }
    df = new SharedFile(uploadView, share);
    shareList.insert(share, df);
}

void SharePage::shareRemoved(int share)
{
    shareList.remove(share);
}

void SharePage::actionActivatePage()
{
    KMLDonkey::App->activatePage(this);
}

void SharePage::copyFileToClipboard(ClipFormat format)
{
    QStringList l;
    QPtrList<QListViewItem> list = uploadView->selectedItems();
    SharedFile* it;
    for (it = (SharedFile*)list.first(); it; it = (SharedFile*)list.next()) {
	const ShareInfo* fi = KMLDonkey::App->donkey->findShareNo(it->fileNo());
	if (!fi) continue;
        l.append(fi->shareName());
        l.append(FileInfo::md4ToString(fi->shareMD4()));
        l.append(QString::number((long int)fi->shareSize()));
    }
    copyToClipboard(l, format);
}

void SharePage::actionRefreshUploaders()
{
    if (! uploadersCallback) uploadersCallback = new uploadersConsoleCallback(this);
    KMLDonkey::App->donkey->sendConsoleMessage("uploaders", uploadersCallback);
}

void SharePage::clearUploadersView()
{
   uploadersView->clear();
}

void SharePage::addUploadersViewItem(QMap<QString, QString> items)
{
    UploadersFile *item = new UploadersFile(uploadersView);
    item->setText(0, items["id"]);
    item->setText(1, items["net"]);
    item->setText(2, items["client"]);
    item->setText(3, items["ip"]);
    item->setText(4, items["name"]);
    item->setText(5, items["uploaded"]);
    item->setText(6, items["downloaded"]);
    item->setText(7, items["file"]);
}

void SharePage::refreshLists()
{
    actionRefreshShared();
    actionRefreshUploaders();
}

void SharePage::updateTimer()
{
    refreshShared->setEnabled(!autoUpdate);
    refreshUploaders->setEnabled(!autoUpdate);
    if (!autoUpdate) {
	if (refreshTimer->isActive()) refreshTimer->stop();
	return;
    }

    if (refreshTimer->isActive()) {
	if (updateFrequency * 1000 != activeFrequency) {
	    activeFrequency = updateFrequency * 1000;
	    refreshTimer->changeInterval(activeFrequency);
	}
    } else {
	activeFrequency = updateFrequency * 1000;
	refreshTimer->start(activeFrequency);
    }
}

void uploadersConsoleCallback::callback(const QString& /*cmd*/, const QString& res)
{
    if(! page) return;
    page->clearUploadersView();

    QMap<QString, QString> items;
    QStringList list = QStringList::split("\n", res);
    QString s, t;
    int p1, p2;

    for (uint i = 0; i < list.count(); i++) {
        if (list[i].startsWith("[")) { // start of a new uploader
            // parse "[<NET> <ID>] <NR> <IP>:<PORT> <USERNAME> client:<CLIENT> downloaded:<DOWN> uploaded:<UP>"
            // and extract what is written in uppercase.

            items.clear();
            s = list[i];

            // get <NET> and <ID>
            p2 = s.find("]");
            if (p2 < 0) continue;
            t = s.mid(1, p2).stripWhiteSpace();
            s.remove(0, p2 + 1);
            QRegExp rxni("([a-zA-Z]+)\\s*([0-9]*)");
            p1 = t.find(rxni);
            if (p1 >= 0) {
                items["net"] = rxni.cap(1);
                items["id"] = rxni.cap(2);
            }

            // get key:value-pairs backwards
            QRegExp rx("(uploaded|downloaded|client|IP|Rating)\\s*:(\\s*)([a-zA-Z0-9\\.]+)");
            while(1) {
                p1 = s.findRev( rx );
                if (p1 < 0) break;
                items[rx.cap(1).lower()] = rx.cap(3);
                //s.remove(p1, rx.cap(1).length() + 1 + rx.cap(2).length() + rx.cap(3).length() );
                s.remove(p1, s.length() - p1 ); // remove everything what comes after the key:value-pair
            }

            // Get <USERNAME> and <ADDRESS>
            // We don't use donkey->protocolVersion() to determinate the parse-method
            // cause this way should be more flexible...
            s = s.stripWhiteSpace();
            QRegExp rxname("(Name\\s*:)");
            p1 = s.find(rxname);
            if(p1 >= 0 && ! items["ip"].isEmpty()) {
                // newer protocols (>=20 ?) use "Name: <USERNAME> IP: <ADDRESS>"
                s.remove(p1, rxname.cap(1).length());
                items["name"] = s.stripWhiteSpace();
            }
            else {
                // look for "NR IP:PORT" on the remaining "<NR> <IP>:<PORT> <USERNAME>"
                p1 = s.find(" ");
                if (p1) {
                    //items["nr"] = s.left(p1); // whatever NR means???
                    s.remove(0, p1 + 1);
                    s = s.stripWhiteSpace();
                }
                p1 = s.find(" ");
                if (p1) {
                    items["ip"] = s.left(p1);
                    s.remove(0, p1 + 1);
                    s = s.stripWhiteSpace();
                }

                // what stays is the "<USERNAME>"
                items["name"] = s;
            }

            // sometimes the ADDRESS just contains a part or the whole USERNAME
            if (items["name"].find(items["ip"]) >= 0 && ! items["ip"].find(QRegExp("[0-9\\.]+\\:[0-9]+")) >= 0)
                items["ip"] = QString::null;

        }
        else {
            p1 = list[i].find("filename:");
            if (p1 >= 0) {
                items["file"] = list[i].right(list[i].length() - p1 - 9).stripWhiteSpace();

                // We should have now all infos about a uploader.
                page->addUploadersViewItem(items);

                // finally clear the items-list just to be sure
                items.clear();
            }
        }
    }
}

