/* -*- c++ -*-
 *
 * giftprotocol.cpp
 *
 * Copyright (C) 2003 Sebastian Sauer <mail@dipe.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#include "giftprotocol.h"
//#include "giftprotocol.moc"

#include "hostiface.h"
#include "gifthost.h"

GiftProtocol::GiftProtocol(QObject *parent) : ProtocolInterface("gift", parent)
{
    kdDebug() << "GiftProtocol() constructor" << endl;
    giftError = NoError;

    message = new GiftMessage();

    connect(&sock, SIGNAL( connected() ), this, SIGNAL( signalConnected() ));
    connect(&sock, SIGNAL( readyRead() ), this, SLOT( socketReadReady() ));
    connect(&sock, SIGNAL( connectionClosed() ), this, SLOT( socketDisconnected() ));
    connect(&sock, SIGNAL( error(int) ), this, SLOT( socketError(int) ));
}

GiftProtocol::~GiftProtocol()
{
    delete message;
}

bool GiftProtocol::isConnected()
{
    return (sock.state() == QSocket::Connected);
}

bool GiftProtocol::connectToCore()
{
    kdDebug() << "GiftProtocol::connectToCore()" << endl;
    if (isConnected())
        emit signalDisconnected(NoError);

    // connect with the core
    QString user = "giFTuser";
    GiftHost *host = (GiftHost*)getHost();
    if (host) {
        kdDebug() << "GiftProtocol::connectToCore() login host=" << host->address() << " port=" << host->guiPort() << endl;
        sock.connectToHost(host->address(), (Q_UINT16)host->guiPort());
        user = host->username();
    }
    else {
        kdDebug() << "GiftProtocol::connectToCore() login default" << endl;
        sock.connectToHost("127.0.0.1", 1213);
    }

    // send the loginmessage
    //sendMessageToCore("ATTACH client(" + QString(GIFT_CLIENT_NAME) + ") version(" + QString(GIFT_CLIENT_VERSION) + ") profile(" + user + ");" );
    if( message->setMessage("ATTACH client(" + QString(GIFT_CLIENT_NAME) + ") version(" + QString(GIFT_CLIENT_VERSION) + ") profile(" + user + ");") == GiftMessageItem::smOk) {
        kdDebug() << "GiftProtocol::connectToCore() called message->setMessage() successfully" << endl;
        if(sendMessageToCore(message))
            kdDebug() << "GiftProtocol::connectToCore() called sendMessageToCore() successfully" << endl;
        else
            kdDebug() << "GiftProtocol::connectToCore() called sendMessageToCore() failed!!!" << endl;
    }
    else kdDebug() << "GiftProtocol::connectToCore() called message->setMessage() failed!!!" << endl;

    return true;
}

bool GiftProtocol::disconnectFromCore()
{
    kdDebug() << "GiftProtocol::disconnectFromCore()" << endl;
    sendMessageToCore("DETACH;"); // close connection
    sock.close();
    emit signalDisconnected(NoError);
    return false;
}

bool GiftProtocol::sendMessageToCore(const QString& message)
{
    if(sock.state() == QSocket::Idle) {
        emit socketError(giftError);
        return false;
    }
    if(message.isEmpty())
        return false;

    kdDebug() << "GiftProtocol::sendMessageToCore() message='" << message << "'" << endl;
    sock.writeBlock(message.latin1(), message.length());

    return true;
}

bool GiftProtocol::sendMessageToCore(GiftMessage* message)
{
    return sendMessageToCore( message->getMessage() );
}

void GiftProtocol::socketReadReady()
{
    kdDebug() << "GiftProtocol::socketReadReady()" << endl;

    while (sock.canReadCommandBlock()) {
        QString line = sock.readCommandBlock().stripWhiteSpace();
        switch( message->setMessage(line) ) {

            case GiftMessageItem::smOk: { // message was successfully parsed, the GiftMessage holds the content now
                QString key = message->getMainKey().upper();

                if(key == "ADDDOWNLOAD") { // Update the downloadlist
                    //TODO we need here a common FileInfo-Object!!!
                    
                    //QString SessionID = message->getMainValue()
                    //message->getArgumentValue("hash");
                    //message->getArgumentValue("state");
                    //message->getArgumentValue("transmit");
                    //message->getArgumentValue("size");
                    //message->getArgumentValue("file");
                    //message->getArgumentValue("shared");

                    //message->getArgumentValue("SOURCE");
                    //  user (username)
                    //  url (url)
                    //  statusgrl (status
                    //  status (protocol status)
                    //  start (chunk start)
                    //  transmit (chunk transmit)
                    //  total (chunk total)

                    kdDebug() << "GiftProtocol::socketReadReady() ADDDOWNLOAD !!!!!!!!!!!!!" << endl;
                }
                else if(key == "ADDUPLOAD") { // Update the uploadlist
                    kdDebug() << "GiftProtocol::socketReadReady() ADDUPLOAD !!!!!!!!!!!!!" << endl;
                }
                else if(key == "ATTACH") { // the core says we're attached now
                    giftServerName = message->getArgumentValue("server");
                    giftServerVersion = message->getArgumentValue("version");
                    emit signalConnected();
                }
                else {
                    //TODO
                    kdDebug() << "------------------------------------------------------------" << endl;
                    kdDebug() << "GiftProtocol::socketReadReady() getMainKey='" << message->getMainKey() << "' getMainValue='" << message->getMainValue() << "'" << endl;
                    //kdDebug() << "GiftProtocol::socketReadReady() getMessage='" << message->getMessage() << "'" << endl;
                    kdDebug() << "------------------------------------------------------------" << endl;
                    disconnectFromCore();
                    return;
                }
            } break;

            case GiftMessageItem::smFinalizerMissing: // if the finalizer is missing just continue reading...
                break;

            //GiftMessageItem::smParseError:
            default: { // for error-cases;
                //TODO
                kdDebug() << QString("PARSE-ERROR in GiftProtocol::socketReadReady()\n%1\n").arg(line) << endl;
                disconnectFromCore();
                return;
            } break;
        }
    }
}

void GiftProtocol::socketDisconnected()
{
    kdDebug() << "GiftProtocol::socketDisconnected()" << endl;
    emit signalDisconnected(giftError);
}

void GiftProtocol::socketError(int err)
{
    kdDebug() << "GiftProtocol::socketError() " << err << endl;
    giftError = err;
    switch (giftError) {
        case QSocket::ErrConnectionRefused:
            emit signalDisconnected(ConnectionRefusedError);
            break;
        case QSocket::ErrHostNotFound:
            emit signalDisconnected(HostNotFoundError);
            break;
        case QSocket::ErrSocketRead:
            giftError = CommunicationError;
            disconnectFromCore();
            break;
    }
}


#include "giftprotocol.moc"
