/* -*- c++ -*-
 *
 * searchcorewidgets.cpp
 *
 * Copyright (C) 2004 Sebastian Sauer <mail@dipe.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#include "searchcorewidgets.h"
#include "searchcore.h"

#include <qhbox.h>
#include <qlabel.h>
#include <qgroupbox.h>
#include <kdebug.h>
#include <klocale.h>
#include <klineedit.h>
#include <kcombobox.h>

/*** SearchCoreWidget ***/

SearchCoreWidget::SearchCoreWidget(SearchCore* c, QWidget *parent, SearchCoreWidgetContainer* container)
    : QVBox(parent)
    , m_searchcore(c)
    , m_parent(parent)
    , m_hidden(false)
    , m_container(container)
{
}

SearchQuery* SearchCoreWidget::getQuery()
{
    kdDebug() << "SearchCoreWidget::getQuery() Unsupported!" << endl;
    return 0;
}

SearchCoreWidget* SearchCoreWidget::setQuery(SearchQuery *query)
{
    //kdDebug() << "SearchCoreWidget::setQuery() query=" << query->getQuerystring() << endl;
    SearchCoreWidget* widget = 0;
    switch(query->operation()) {
        case SearchQuery::And: widget = new SearchCoreWidgetAnd(m_searchcore, m_parent, m_container); break;
        case SearchQuery::Or: widget = new SearchCoreWidgetOr(m_searchcore, m_parent, m_container); break;
        case SearchQuery::Hidden: widget = new SearchCoreWidgetHidden(m_searchcore, m_parent, m_container); break;
        case SearchQuery::AndNot: widget = new SearchCoreWidgetAndNot(m_searchcore, m_parent, m_container); break;
        case SearchQuery::Module: widget = new SearchCoreWidgetModule(m_searchcore, m_parent, m_container); break;
        case SearchQuery::Keywords: widget = new SearchCoreWidgetKeywords(m_searchcore, m_parent, m_container); break;
        case SearchQuery::MinSize: widget = new SearchCoreWidgetMinSize(m_searchcore, m_parent, m_container); break;
        case SearchQuery::MaxSize: widget = new SearchCoreWidgetMaxSize(m_searchcore, m_parent, m_container); break;
        case SearchQuery::Format: widget = new SearchCoreWidgetFormat(m_searchcore, m_parent, m_container); break;
        case SearchQuery::Media: widget = new SearchCoreWidgetMedia(m_searchcore, m_parent, m_container); break;
        case SearchQuery::Mp3Artist: widget = new SearchCoreWidgetMp3Artist(m_searchcore, m_parent, m_container); break;
        case SearchQuery::Mp3Title: widget = new SearchCoreWidgetMp3Title(m_searchcore, m_parent, m_container); break;
        case SearchQuery::Mp3Album: widget = new SearchCoreWidgetMp3Album(m_searchcore, m_parent, m_container); break;
        case SearchQuery::Mp3Bitrate: widget = new SearchCoreWidgetMp3Bitrate(m_searchcore, m_parent, m_container); break;
        default: {
            kdDebug() << QString("SearchCoreWidget::setQuery() Unhandled operation %1").arg(query->operation()) << endl;
            return 0;
        } break;
    }
    if(! widget)
        kdDebug() << QString("SearchCoreWidget::setQuery() for operation %1 returned NULL").arg(query->operation()) << endl;
    else {
        SearchCoreWidgetContainer* container = widget->getContainer();
        if(container) widget->setHidden(container->isHidden());
        if(! widget->setQuery(query)) {
            kdDebug() << QString("SearchCoreWidget::setQuery() widget->setQuery() for operation %1 returned NULL").arg(query->operation()) << endl;
            delete widget; widget = 0;
        }
    }
    return widget;
}

/*** SearchCoreWidgetContainer ***/

const QString SearchCoreWidgetContainer::getDescription()
{
    QString s;
    for(QValueList<SearchCoreWidget*>::Iterator it = m_children.begin(); it != m_children.end(); ++it) {
        QString t = (*it)->getDescription();
        if(! t.isEmpty()) s += t + QString(" ");
    }
    return s.stripWhiteSpace();
}

SearchQuery* SearchCoreWidgetContainer::getQueryList(SearchQueryList* querylist)
{
    for(QValueList<SearchCoreWidget*>::Iterator it = m_children.begin(); it != m_children.end(); ++it) {
        SearchQuery* q = (*it)->getQuery();
        if(q) querylist->append(q);
    }
    if(querylist->count() < 1) {
        delete querylist; querylist = 0;
    }
    else if(querylist->count() == 1) { // Simplify the query if possible
        return querylist->take(0);
    }
    return querylist;
}

SearchCoreWidget* SearchCoreWidgetContainer::setQuery(SearchQuery *query)
{
    //kdDebug() << "SearchCoreWidgetContainer::setQuery() query=" << query->getQuerystring() << endl;
    if(query->operation() == SearchQuery::Hidden) m_hidden = true;
    SearchQueryList* q = dynamic_cast<SearchQueryList*>(query);
    if(! q) {
        kdDebug() << "SearchCoreWidgetContainer::setQuery() Invalid SearchQueryList" << endl;
        return 0;
    }
    m_container = this;
    int i, j = q->count();
    for(i = 0; i < j; i++) {
        SearchCoreWidget* w = SearchCoreWidget::setQuery(q->at(i));
        if(w) m_children.append(w);
    }
    return (m_children.count() > 0) ? this : 0;
}

/*** SearchCoreWidgetAnd ***/

SearchQuery* SearchCoreWidgetAnd::getQuery()
{
    //kdDebug() << "SearchCoreWidgetAnd::getQuery()" << endl;
    return getQueryList(new QueryAnd());
}

/*** SearchCoreWidgetOr ***/

SearchQuery* SearchCoreWidgetOr::getQuery()
{
    //kdDebug() << "SearchCoreWidgetOr::getQuery()" << endl;
    return getQueryList(new QueryOr());
}

/*** SearchCoreWidgetAndNot ***/

SearchCoreWidgetAndNot::SearchCoreWidgetAndNot(SearchCore* c, QWidget *parent, SearchCoreWidgetContainer* container)
    : SearchCoreWidgetContainer(c, parent, container)
    , m_andmodule(0)
    , m_notmodule(0)
{
}

SearchCoreWidgetAndNot::~SearchCoreWidgetAndNot()
{
    delete m_andmodule; m_andmodule = 0;
    delete m_notmodule; m_notmodule = 0;
}

const QString SearchCoreWidgetAndNot::getDescription()
{
    if(! (m_andmodule && m_notmodule)) return QString::null;
    QString s = m_andmodule->getDescription();
    QString t = m_notmodule->getDescription();
    if(! t.isEmpty()) s += (s.isEmpty()) ? "-" + t : " -" + t;
    return s;
}

SearchQuery* SearchCoreWidgetAndNot::getQuery()
{
    //kdDebug() << "SearchCoreWidgetAndNot::getQuery()" << endl;
    if(! (m_andmodule && m_notmodule)) return 0;
    SearchQuery* qand = m_andmodule->getQuery();
    SearchQuery* qnot = m_notmodule->getQuery();
    if(qand && qnot) return new QueryAndNot(qand, qnot);
    delete qand; qand = 0;
    delete qnot; qnot = 0;
    return 0;
}

SearchCoreWidget* SearchCoreWidgetAndNot::setQuery(SearchQuery *query)
{
    //kdDebug() << "SearchCoreWidgetAndNot::setQuery() query=" << query->getQuerystring() << endl;
    QueryAndNot* q = dynamic_cast<QueryAndNot*>(query);
    if(! q) return 0;
    m_container = this;
    m_andmodule = SearchCoreWidget::setQuery(q->getQueryAnd());
    m_notmodule = SearchCoreWidget::setQuery(q->getQueryNot());
    if(! (m_andmodule && m_notmodule)) {
        delete m_andmodule; m_andmodule = 0;
        delete m_notmodule; m_notmodule = 0;
        return 0;
    }
    return this;
}

/*** SearchCoreWidgetHidden ***/

SearchQuery* SearchCoreWidgetHidden::getQuery()
{
    //kdDebug() << "SearchCoreWidgetHidden::getQuery()" << endl;
    return getQueryList(new QueryHidden());
}

/*** SearchCoreWidgetModule ***/

SearchCoreWidgetModule::SearchCoreWidgetModule(SearchCore* c, QWidget *parent, SearchCoreWidgetContainer* container)
    : SearchCoreWidgetContainer(c, parent, container)
    , m_module(0)
{
}

SearchCoreWidgetModule::~SearchCoreWidgetModule()
{
    delete m_module; m_module = 0;
}

const QString SearchCoreWidgetModule::getDescription()
{
    return m_module ? m_module->getDescription() : QString::null;
}

SearchQuery* SearchCoreWidgetModule::getQuery()
{
    //kdDebug() << "SearchCoreWidgetModule::getQuery()" << endl;
    if(! m_module) return 0;
    /*
    SearchQuery *q = m_module->getQuery();
    if(! q) return 0;
    return new QueryModule(m_name, q);
    */
    return m_module->getQuery();
}

SearchCoreWidget* SearchCoreWidgetModule::setQuery(SearchQuery *query)
{
    //kdDebug() << "SearchCoreWidgetModule::setQuery() query=" << query->getQuerystring() << endl;
    if(query->operation() != SearchQuery::Module) {
        kdDebug() << "SearchCoreWidgetModule::setQuery() query->operation()!=SearchQuery::Module This should never be the case!" << endl;
        SearchCoreWidget::setQuery(query);
        return 0;
    }
    QueryModule* q = dynamic_cast<QueryModule*>(query);
    if(! q) return 0;
    m_name = q->getName();
    QGroupBox *grp = new QGroupBox(1, Qt::Horizontal, m_name, m_parent);
    grp->setInsideSpacing(2);
    m_parent = grp;
    m_module = SearchCoreWidget::setQuery(q->getQuery());
    return m_module ? this : 0;
}

/*** SearchCoreWidgetEdit ***/

SearchCoreWidgetEdit::SearchCoreWidgetEdit(SearchCore* c, QWidget *parent, SearchCoreWidgetContainer* container)
    : SearchCoreWidget(c, parent, container)
    , m_edit(0)
    , m_combo(0)
{
}

const QString SearchCoreWidgetEdit::getValue()
{
    return m_combo ? m_combo->currentText() : m_edit ? m_edit->text() : QString::null;
}

SearchCoreWidget* SearchCoreWidgetEdit::setQuery(SearchQuery *query)
{
    return SearchCoreWidgetEdit::setQuery(query, QStringList());
}

SearchCoreWidget* SearchCoreWidgetEdit::setQuery(SearchQuery *query, const QStringList& items)
{
    //kdDebug() << "SearchCoreWidgetEdit::setQuery() query=" << query->getQuerystring() << endl;
    SearchQueryTwoStrings* q = dynamic_cast<SearchQueryTwoStrings*>(query);
    if(! q) return 0;
    QHBox *box = new QHBox(m_parent);
    box->setEnabled(! m_hidden);
    box->setSpacing(2);

    m_key = q->getKey();

    //TODO The label should reflect the parent SearchQueryList if there is one!
    new QLabel(getTitle(), box);
    if(items.isEmpty()) {
        m_edit = new KLineEdit(box);
        box->setStretchFactor(m_edit, 1);
        if(isHidden()) m_edit->setText(m_key);
        connect(m_edit, SIGNAL( returnPressed() ), m_searchcore, SLOT( startSearch() ));
    }
    else {
        m_combo = new KComboBox(box);
        box->setStretchFactor(m_combo, 1);
        m_combo->setEditable(true);
        m_combo->insertItem(isHidden() ? m_key : QString(""), 0);
        m_combo->insertStringList(items);
        connect(m_combo, SIGNAL( returnPressed() ), m_searchcore, SLOT( startSearch() ));
    }
    return this;
}

/*** SearchCoreWidgetKeywords ***/

const QString SearchCoreWidgetKeywords::getTitle()
{
    return i18n("Keywords:");
}

const QString SearchCoreWidgetKeywords::getDescription()
{
    return getValue();
}

SearchQuery* SearchCoreWidgetKeywords::getQuery()
{
    //kdDebug() << "SearchCoreWidgetKeywords::getQuery()" << endl;
    QString s = getValue();
    if(s.isEmpty()) {
        kdDebug() << "SearchCoreWidgetKeywords::getQuery() getValue().isEmpty()" << endl;
        return 0;
    }
    return new QueryKeywords(m_key, s);
}

/*** SearchCoreWidgetMinSize ***/

const QString SearchCoreWidgetMinSize::getTitle()
{
    return i18n("Min size:");
}

SearchQuery* SearchCoreWidgetMinSize::getQuery()
{
    //kdDebug() << "SearchCoreWidgetMinSize::getQuery()" << endl;
    QString s = getValue();
    if(s.isEmpty()) return 0;
    return new QueryMinSize(m_key, QString::number(filesizeStr2Int(getValue())));
}

SearchCoreWidget* SearchCoreWidgetMinSize::setQuery(SearchQuery *query)
{
    return SearchCoreWidgetEdit::setQuery(query, QStringList() << "500 MB" << "100 MB" << "50 MB" << "3 MB" << "500 KB" << "500 B");
}

/*** SearchCoreWidgetMaxSize ***/

const QString SearchCoreWidgetMaxSize::getTitle()
{
    return i18n("Max size:");
}

SearchQuery* SearchCoreWidgetMaxSize::getQuery()
{
    //kdDebug() << "SearchCoreWidgetMaxSize::getQuery()" << endl;
    QString s = getValue();
    if(s.isEmpty()) return 0;
    return new QueryMaxSize(m_key, QString::number(filesizeStr2Int(s)));
}

SearchCoreWidget* SearchCoreWidgetMaxSize::setQuery(SearchQuery *query)
{
    return SearchCoreWidgetEdit::setQuery(query, QStringList() << "500 MB" << "100 MB" << "50 MB" << "3 MB" << "500 KB" << "500 B");
}

/*** SearchCoreWidgetFormat ***/

const QString SearchCoreWidgetFormat::getTitle()
{
    return i18n("Format:");
}

SearchQuery* SearchCoreWidgetFormat::getQuery()
{
    //kdDebug() << "SearchCoreWidgetFormat::getQuery()" << endl;
    QString s = getValue();
    if(s.isEmpty()) return 0;
    return new QueryFormat(m_key, s);
}

SearchCoreWidget* SearchCoreWidgetFormat::setQuery(SearchQuery *query)
{
    return SearchCoreWidgetEdit::setQuery(query, QStringList() << "Audio" << "Video" << "Program" << "Image" << "Documentation" << "Collection");
}

/*** SearchCoreWidgetMedia ***/

const QString SearchCoreWidgetMedia::getTitle()
{
    return i18n("Media:");
}

SearchQuery* SearchCoreWidgetMedia::getQuery()
{
    //kdDebug() << "SearchCoreWidgetMedia::getQuery()" << endl;
    QString s = getValue();
    if(s.isEmpty()) return 0;
    return new QueryMedia(m_key, s);
}

SearchCoreWidget* SearchCoreWidgetMedia::setQuery(SearchQuery *query)
{
    return SearchCoreWidgetEdit::setQuery(query, QStringList() << "avi" << "mp3");
}

/*** SearchCoreWidgetMp3Artist ***/

const QString SearchCoreWidgetMp3Artist::getTitle()
{
    return i18n("Artist:");
}

const QString SearchCoreWidgetMp3Artist::getDescription()
{
    return getValue();
}

SearchQuery* SearchCoreWidgetMp3Artist::getQuery()
{
    //kdDebug() << "SearchCoreWidgetMp3Artist::getQuery()" << endl;
    QString s = getValue();
    if(s.isEmpty()) return 0;
    return new QueryMp3Artist(m_key, s);
}

/*** SearchCoreWidgetMp3Title ***/

const QString SearchCoreWidgetMp3Title::getTitle()
{
    return i18n("Title:");
}

const QString SearchCoreWidgetMp3Title::getDescription()
{
    return getValue();
}

SearchQuery* SearchCoreWidgetMp3Title::getQuery()
{
    //kdDebug() << "SearchCoreWidgetMp3Title::getQuery()" << endl;
    QString s = getValue();
    if(s.isEmpty()) return 0;
    return new QueryMp3Title(m_key, s);
}

/*** SearchCoreWidgetMp3Album ***/

const QString SearchCoreWidgetMp3Album::getTitle()
{
    return i18n("Album:");
}

const QString SearchCoreWidgetMp3Album::getDescription()
{
    return getValue();
}

SearchQuery* SearchCoreWidgetMp3Album::getQuery()
{
    //kdDebug() << "SearchCoreWidgetMp3Album::getQuery()" << endl;
    QString s = getValue();
    if(s.isEmpty()) return 0;
    return new QueryMp3Album(m_key, s);
}

/*** SearchCoreWidgetMp3Bitrate ***/

const QString SearchCoreWidgetMp3Bitrate::getTitle()
{
    return i18n("Bitrate:");
}

SearchQuery* SearchCoreWidgetMp3Bitrate::getQuery()
{
    //kdDebug() << "SearchCoreWidgetMp3Bitrate::getQuery()" << endl;
    QString s = getValue();
    if(s.isEmpty()) return 0;
    return new QueryMp3Bitrate(m_key, s);
}

SearchCoreWidget* SearchCoreWidgetMp3Bitrate::setQuery(SearchQuery *query)
{
    return SearchCoreWidgetEdit::setQuery(query, QStringList() << "64" << "96" << "128" << "160" << "192");
}

//#include "searchcorewidgets.moc"
