/*
 * $Id: infodialog.cpp,v 1.8 2003/06/30 00:34:03 gibreel Exp $
 *
 * Copyright (C) 2003 Petter E. Stokke <gibreel@gibreel.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Log: infodialog.cpp,v $
 * Revision 1.8  2003/06/30 00:34:03  gibreel
 * Added an embedded web browser class, and through it added Jigle and Bitzi
 * lookups to the file info dialog.
 *
 * Revision 1.7  2003/06/28 21:17:00  gibreel
 * The file info dialog now displays file sources. Libkmldonkey received a
 * number of updates and bugfixes to facilitate this.
 *
 * Revision 1.6  2003/06/19 21:50:45  gibreel
 * Split the various pages off into individual objects, and made a ton of code
 * cleanups, API changes, and not a few bugfixes in the process. The
 * disconnect/reconnect bug, especially, now seems to be gone.
 *
 * Revision 1.5  2003/06/09 18:10:23  gibreel
 * Configurable availability display colours. Extended the DCOP interface and
 * improved the interaction between the GUI and the applet. Added colour for
 * the download list's queued state. Cleanups, bugfixes all round.
 *
 * Revision 1.4  2003/05/12 17:14:59  gibreel
 * Typo in downloaded size display.
 *
 * Revision 1.3  2003/05/11 20:02:26  gibreel
 * Added some missing fields to the info dialog.
 *
 * Revision 1.2  2003/05/01 17:09:37  gibreel
 * File info dialog keeps and updates a percentage in the window caption.
 *
 * Revision 1.1  2003/04/13 17:59:46  gibreel
 * File info dialog and coloured availability bars.
 *
 */

#include "infodialog.h"
#include "clientlist.h"
#include "kmldonkey.h"
#include "webservices.h"

#include <kglobal.h>
#include <kconfig.h>
#include <klocale.h>
#include <kdebug.h>
#include <qlayout.h>
#include <qlabel.h>

#include <fileinfo.h>

static QLabel* makeLabel(const QString& label, QWidget* parent, bool alignTop = false)
{
    QLabel* l = new QLabel(label, parent);
    if (alignTop) l->setAlignment(Qt::AlignTop | Qt::AlignRight);
    else l->setAlignment(Qt::AlignVCenter | Qt::AlignRight);
    l->setSizePolicy(QSizePolicy::Minimum, QSizePolicy::Fixed);
    return l;
}

FileInfoDialog::FileInfoDialog(int file)
    : KDialogBase(Tabbed, i18n("File Information"),
                  Try|Close, Default, 0, 0, false)
{
    fileno = file;
    FileInfo* fi = KMLDonkey::App->donkey->findDownloadFileNo(fileno);
    if (!fi) return;

    setButtonText(Try, i18n("&Preview"));
    setButtonTip(Try, i18n("Ask MLDonkey to preview the file."));

    QFrame *frame = addPage(QString("File"));
    QGridLayout* layout = new QGridLayout(frame, 7, 2, 4);

    layout->addWidget(makeLabel(i18n("Name:"), frame), 0, 0);
    layout->addWidget(makeLabel(i18n("Complete:"), frame), 1, 0);
    layout->addWidget(makeLabel(i18n("Downloaded:"), frame), 2, 0);
    layout->addWidget(makeLabel(i18n("Speed:"), frame), 3, 0);
    layout->addWidget(makeLabel(i18n("Hash:"), frame), 4, 0);
    layout->addWidget(makeLabel(i18n("Format:"), frame), 5, 0);
    layout->addWidget(makeLabel(i18n("Alt. names:"), frame, true), 6, 0);

    nameView = new KLineEdit(fi->fileName(), frame, "nameView");
    nameView->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);
    nameView->setReadOnly(true);
    layout->addWidget(nameView, 0, 1);

    availView = new AvailabilityWidget(fileno, frame, "availView");
    layout->addWidget(availView, 1, 1);

    QHBoxLayout* l2 = new QHBoxLayout();

    sizeView = new KLineEdit(humanReadableSize(fi->fileDownloaded()), frame, "sizeView");
    sizeView->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);
    sizeView->setReadOnly(true);
    l2->addWidget(sizeView);

    l2->addWidget(makeLabel(i18n("Size:"), frame));
    totalSizeView = new KLineEdit(humanReadableSize(fi->fileSize()), frame, "totalSizeView");
    totalSizeView->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);
    totalSizeView->setReadOnly(true);
    l2->addWidget(totalSizeView);

    l2->addWidget(makeLabel(i18n("Network:"), frame));
    networkView = new KLineEdit(KMLDonkey::App->donkey->findNetworkNo(fi->fileNetwork())->networkName(), frame, "networkView");
    networkView->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);
    networkView->setReadOnly(true);
    l2->addWidget(networkView);

    layout->addLayout(l2, 2, 1);

    QHBoxLayout* l3 = new QHBoxLayout();

    speedView = new KLineEdit(humanReadableSize((int64)fi->fileSpeed()) + "b/s", frame, "speedView");
    speedView->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);
    speedView->setReadOnly(true);
    l3->addWidget(speedView);

    l3->addWidget(makeLabel(i18n("ETA:"), frame));
    etaView = new KLineEdit(calculateETA(fi), frame, "etaView");
    etaView->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);
    etaView->setReadOnly(true);
    l3->addWidget(etaView);

    layout->addLayout(l3, 3, 1);

    hashView = new KLineEdit(FileInfo::md4ToString(fi->fileMD4()), frame, "hashView");
    hashView->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);
    hashView->setReadOnly(true);
    layout->addWidget(hashView, 4, 1);

    formatView = new KLineEdit(fi->fileFormatInfo(), frame, "formatView");
    formatView->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);
    formatView->setReadOnly(true);
    layout->addWidget(formatView, 5, 1);

    namesView = new KTextBrowser(frame, "namesView");
    namesView->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::MinimumExpanding);
    namesView->setTextFormat(QTextBrowser::PlainText);
    namesView->setWordWrap(QTextBrowser::NoWrap);
    namesView->setText(fi->fileNames().join("\n"));
    layout->addWidget(namesView, 6, 1);

    layout->setColStretch(0, 1);
    layout->setColStretch(1, 100);

    QFrame *srcframe = addPage(i18n("Sources"));
    QBoxLayout* srclayout = new QVBoxLayout(srcframe);
    clientList = new ClientList(fileno, srcframe);
    srclayout->addWidget(clientList);

    clientList->restoreLayout(KGlobal::config(), "DownloadSourceList");

    if (KMLDonkey::App->donkey->findNetworkNo(fi->fileNetwork())->networkName() == "Donkey") {
	QFrame* jigleframe = addPage(i18n("Jigle"));
	(new QVBoxLayout(jigleframe))->addWidget(new JigleFileInfo(fileno, jigleframe));
	QFrame* bitziframe = addPage(i18n("Bitzi"));
	(new QVBoxLayout(bitziframe))->addWidget(new BitziFileInfo(fileno, bitziframe));
    }

    updateCaption(fi);

    setInitialSize(configDialogSize("FileInfoDialog"));
    connect(this, SIGNAL(finished()), this, SLOT(saveSize()));
    connect(KMLDonkey::App->donkey, SIGNAL(fileUpdated(int)), this, SLOT(updateFile(int)));
}

void FileInfoDialog::updateCaption(FileInfo* fi)
{
    QString caption;
    caption.sprintf("(%.2f%%) ", (fi->fileDownloaded() * 100.0) / fi->fileSize());
    caption += fi->fileName();
    setCaption(caption);
}

void FileInfoDialog::saveSize()
{
    saveDialogSize("FileInfoDialog");
    clientList->saveLayout(KGlobal::config(), "DownloadSourceList");
    deleteLater();
}

void FileInfoDialog::updateFile(int file)
{
    if (file != fileno) return;
    FileInfo* fi = KMLDonkey::App->donkey->findDownloadFileNo(fileno);
    if (!fi) {
        deleteLater();
        return;
    }
    nameView->setText(fi->fileName());
    availView->update();
    sizeView->setText(humanReadableSize(fi->fileDownloaded()));
    totalSizeView->setText(humanReadableSize(fi->fileSize()));
    speedView->setText(humanReadableSize((int64)fi->fileSpeed()) + "b/s");
    etaView->setText(calculateETA(fi));
    formatView->setText(fi->fileFormatInfo());
    updateCaption(fi);
}

#include "infodialog.moc"
