/*
 * $Id: kmldonkey.cpp,v 1.55 2003/07/30 15:30:30 dipesh Exp $
 *
 * Copyright (C) 2003 Petter E. Stokke <gibreel@gibreel.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Log: kmldonkey.cpp,v $
 * Revision 1.55  2003/07/30 15:30:30  dipesh
 * Clear Statusbar-infos on disconnect
 *
 * Revision 1.54  2003/07/27 20:24:26  gibreel
 * Added KActions for switching between pages.
 *
 * Revision 1.53  2003/07/27 18:51:42  gibreel
 * KMLDonkey didn't clean the connect menu properly if the host list was
 * changed.
 *
 * Revision 1.52  2003/07/27 18:16:21  gibreel
 * Updated TODO.
 *
 * Revision 1.51  2003/07/23 19:47:05  gibreel
 * Removed the unnecessary and KDE 3.2-specific kactionclasses.h include.
 *
 * Revision 1.50  2003/07/23 19:35:25  gibreel
 * KMLDonkey remembers the tab order (it can be rearranged if compiled with
 * KTabWidget).
 *
 * Revision 1.49  2003/07/23 18:04:39  gibreel
 * Support for multiple host definitions using HostManager. Configurable
 * charset encoding.
 *
 * Revision 1.48  2003/07/20 20:45:04  dipesh
 * Added "Configure MLDonkey" :)
 *
 * Revision 1.47  2003/07/19 20:20:32  gibreel
 * Uses autoconf to detect the presence of KTabWidget, rather than guessing its
 * presence using KDE_IS_VERSION.
 *
 * Revision 1.46  2003/07/19 14:18:22  gibreel
 * Uses KTabWidget instead of QTabWidget/ClosableTabWidget if available (KDE
 * 3.2 or CVS). ClosableTabWidget API changed to match KTabWidget's.
 *
 * Revision 1.45  2003/07/17 19:54:00  gibreel
 * Now properly calls clear() in all pages when disconnected.
 *
 * Revision 1.44  2003/07/17 15:39:12  gibreel
 * Release preparation.
 *
 * Revision 1.43  2003/07/01 23:08:23  gibreel
 * The applet and GUI now accept drag and drop events containing URLs, which
 * are passed on to the core.
 *
 * Revision 1.42  2003/06/30 23:30:35  gibreel
 * Preliminary friend list support. A ton of updates to the libkmldonkey API to
 * accommodate this, most notably improvements to the search handling
 * necessitated by mldonkey's somewhat awkward reporting of friend shares.
 *
 * Revision 1.41  2003/06/30 14:59:22  gibreel
 * Updated lists to support libkmldonkey's new removed signals. Moved pages'
 * connect statements out of KMLDonkey's constructor into their own
 * constructors. Added a debug console displaying dumps of unhandled messages.
 *
 * Revision 1.40  2003/06/29 15:04:59  gibreel
 * Fixed the Vanishing KMLDonkey bug.
 *
 * Revision 1.39  2003/06/28 21:55:13  gibreel
 * Configurable coloured source list.
 *
 * Revision 1.38  2003/06/19 22:05:52  gibreel
 * Merged the General and Donkey prefs pages back together.
 *
 * Revision 1.37  2003/06/19 21:50:45  gibreel
 * Split the various pages off into individual objects, and made a ton of code
 * cleanups, API changes, and not a few bugfixes in the process. The
 * disconnect/reconnect bug, especially, now seems to be gone.
 *
 * Revision 1.36  2003/06/19 17:04:32  gibreel
 * Minor code cleanups and attempt to fix the crash bugs on
 * disconnect/reconnect. Both kmldonkey and libkmldonkey now clean out their
 * state data when the connection with the core is lost.
 *
 * Revision 1.35  2003/06/19 15:24:21  dipesh
 * Added total number of servers to the statusbar
 *
 * Revision 1.34  2003/06/16 13:18:16  dipesh
 * Added KAction's for "Connect more servers", "Add server" and "Remove old servers"
 *
 * Revision 1.33  2003/06/14 15:29:17  dipesh
 * Fixed bad performance with a large serverlist.
 *
 * Revision 1.32  2003/06/13 22:35:07  dipesh
 * Small patch to don't slow down the serverlist on applyPrefs
 *
 * Revision 1.31  2003/06/13 18:20:01  gibreel
 * Libkmldonkey now uses references instead of pointers everywhere except where
 * it would cause an obvious performance impact, which should lead to less
 * chance of memory leaks and cleaner code in general. Almost everything that
 * should be const is now also const.
 *
 * Revision 1.30  2003/06/13 09:50:53  dipesh
 * optional coloured server- and search lists
 *
 * Revision 1.29  2003/06/09 23:45:36  gibreel
 * Availability threshold config uses a KIntNumInput instead of a QSlider.
 *
 * Revision 1.28  2003/06/09 18:10:23  gibreel
 * Configurable availability display colours. Extended the DCOP interface and
 * improved the interaction between the GUI and the applet. Added colour for
 * the download list's queued state. Cleanups, bugfixes all round.
 *
 * Revision 1.27  2003/05/30 11:53:00  dipesh
 * Optional colored Downloadlist
 *
 * Revision 1.26  2003/05/29 23:47:55  gibreel
 * Added shared file list refresh action. For some reason, though, the core
 * seems to only want to respond to the refresh request once...
 *
 * Revision 1.25  2003/05/28 20:01:44  gibreel
 * Renamed actions so they all have different names; this should make the
 * shortcut configuration dialog a little less confusing.
 *
 * Revision 1.24  2003/05/28 19:57:58  gibreel
 * Remembers the active page between restarts.
 *
 * Revision 1.23  2003/05/28 19:50:02  gibreel
 * Added facilities for launching the (as yet unwritten) MLDonkey config tool
 * from the GUI and control centre applet.
 *
 * Revision 1.22  2003/05/26 15:49:46  gibreel
 * Improved connect/disconnect code: added KActions for connecting to and
 * disconnecting from the mldonkey core, and an option for trying to stay
 * connected at all times (behaving essentially like the applet does).
 *
 * Revision 1.21  2003/05/26 14:31:46  dipesh
 * Added prefs; Statusbar-label's + save on exit
 *
 * Revision 1.20  2003/05/24 14:52:22  dipesh
 * more connect-infos in statusBar()
 *
 * Revision 1.19  2003/05/24 11:24:48  dipesh
 * Added some more Output to the statusBar()
 *
 * Revision 1.18  2003/05/20 20:56:20  gibreel
 * Resolved some compiler warnings.
 *
 * Revision 1.17  2003/05/18 17:58:22  dipesh
 * Added KAction's Download, ForceDownload and RemoveServers
 *
 * Revision 1.16  2003/05/17 17:27:22  dipesh
 * More search-stuff; added avaibility-column, use the listfont, some smaller fixes
 *
 * Revision 1.15  2003/05/15 14:25:51  gibreel
 * Console displays copyright message and GPL summary at startup.
 *
 * Revision 1.14  2003/05/12 16:03:23  gibreel
 * Switched the main KTabCtl over to a QTabWidget and added icons to the page
 * titles. How odd that KDE's "extended" tab widget doesn't support icon labels
 * when Qt's original one does.
 *
 * Revision 1.13  2003/05/11 08:34:09  dipesh
 * Adding initial Search-functionality
 *
 * Revision 1.12  2003/05/03 13:33:42  dipesh
 * *** empty log message ***
 *
 * Revision 1.11  2003/05/01 14:56:09  gibreel
 * Applied yet another patch from dipesh <dipesh@gmx.de> which remembers the
 * complete state of the toolbar and statusbar, and moves the Settings menu
 * into the kmldonkeyui.rc file.
 *
 * Revision 1.10  2003/04/27 23:00:10  gibreel
 * Double clicking on an item in the download list opens the file info dialog.
 *
 * Revision 1.9  2003/04/27 16:10:08  gibreel
 * Applied another patch from dipesh <dipesh@gmx.de> which adds the Show All
 * Servers action to the prefs dialog, and cleans up prefs handling a little.
 * The prefs dialog now also remembers its size.
 *
 * Revision 1.8  2003/04/26 18:27:44  gibreel
 * Applied a patch from dipesh <dipesh@gmx.de> for hiding non-connected
 * servers from the server list.
 *
 * Revision 1.7  2003/04/15 15:07:30  gibreel
 * Added server copy-to-clipboard actions too.
 *
 * Revision 1.6  2003/04/15 14:54:56  gibreel
 * Added copy-to-clipboard actions.
 *
 * Revision 1.5  2003/04/13 17:59:46  gibreel
 * File info dialog and coloured availability bars.
 *
 * Revision 1.4  2003/04/06 21:29:45  gibreel
 * Rudimentary server actions, and all lists now display network type.
 *
 * Revision 1.3  2003/04/06 19:22:52  gibreel
 * Added the remaining actions for the download list (but still missing the
 * file information dialog).
 *
 * Revision 1.2  2003/03/24 15:36:52  gibreel
 * More work; still not complete, but getting there.
 *
 * Revision 1.1  2003/03/23 23:40:01  gibreel
 * Preliminary version of standalone GUI.
 *
 */

#include "kmldonkey.h"
#include "infolist.h"
#include "prefs.h"
#include "fontselector.h"
#include "serverpage.h"
#include "downloadpage.h"
#include "sharepage.h"
#include "friendpage.h"
#include "consolepage.h"
#include "search.h"
#include "debugpage.h"
#include "mlconfig.h"

#include <qdragobject.h>
#include <kprinter.h>
#include <qpainter.h>
#include <qpaintdevicemetrics.h>
#include <qclipboard.h>
#include <qcheckbox.h>
#include <qsignalmapper.h>

#include <kglobal.h>
#include <klocale.h>
#include <kdebug.h>
#include <kiconloader.h>
#include <kmenubar.h>
#include <kstatusbar.h>
#include <kkeydialog.h>
#include <kaccel.h>
#include <kconfig.h>
#include <kurl.h>
#include <kmessagebox.h>
#include <knuminput.h>
#include <kcolorbutton.h>
#include <klineeditdlg.h>
#include <kcombobox.h>
#include <kcharsets.h>
#include <kedittoolbar.h>
#include <kstdaccel.h>
#include <kaction.h>
#include <kstdaction.h>

#include <donkeyprotocol.h>
#include <hostmanager.h>

#ifdef HAVE_KTABWIDGET
#include <ktabwidget.h>
#else
#include <qtabwidget.h>
#endif

KMLDonkey *KMLDonkey::App=0; // global access to the static instance-pointer

KMLDonkey::KMLDonkey()
    : DCOPObject("KMLDonkeyIface"),
      KMainWindow( 0, "KMLDonkey" )
{
    App = this;
    SaveStatOnExit = true;
    ShowStatusbarLabels = true;
    persistentReconnect = true;
    reconnect = 0;
    prefs = 0;
    donkey = new DonkeyProtocol(false, this);
    hostManager = new HostManager(this);
    lastHost = hostManager->defaultHostName();

    connectActions.setAutoDelete(true);

    setAcceptDrops(true);
    setCentralWidget(buildInterface());
    setupActions();
    resize( QSize(607, 621).expandedTo(minimumSizeHint()) );
    setAutoSaveSettings();
    restoreState(KGlobal::config());

    // Register with DCOP
    if ( !kapp->dcopClient()->isRegistered() ) {
	kapp->dcopClient()->registerAs( "kmldonkey" );
	kapp->dcopClient()->setDefaultObject( objId() );
    }

    timer = new QTimer(this);
    connect(timer, SIGNAL(timeout()), this, SLOT(checkReconnect()));
    timer->start(5000);

    connect(donkey, SIGNAL(donkeyConnected()), this, SLOT(donkeyConnected()));
    connect(donkey, SIGNAL(donkeyDisconnected(int)), this, SLOT(donkeyDisconnected(int)));
    connect(donkey, SIGNAL(clientStats(int64, int64, int64, int, int, int, int, int, int, int)),
            this, SLOT(updateStatus(int64, int64, int64, int, int, int, int, int, int, int)));

    connect(hostManager, SIGNAL(hostListUpdated()), SLOT(hostListUpdated()));

    connectDonkey();
}

void KMLDonkey::show()
{
    KMainWindow::show();
    QByteArray args;
    QDataStream stream(args, IO_WriteOnly);
    stream << true;
    emitDCOPSignal("kmldonkeyAppeared(bool)", args);
}

void KMLDonkey::hide()
{
    KMainWindow::hide();
    QByteArray args;
    QDataStream stream(args, IO_WriteOnly);
    stream << false;
    emitDCOPSignal("kmldonkeyAppeared(bool)", args);
}

void KMLDonkey::addPage(KMLDonkeyPage* page, const QIconSet& iconset, const QString& label)
{
    pageTab->addTab(dynamic_cast<QWidget*>(page), iconset, label);
    pages.append(page);
}

void KMLDonkey::removePage(KMLDonkeyPage* page)
{
    pageTab->removePage(dynamic_cast<QWidget*>(page));
    pages.remove(page);
}

void KMLDonkey::activatePage(KMLDonkeyPage* page)
{
    pageTab->showPage(dynamic_cast<QWidget*>(page));
}

void KMLDonkey::activatePage(int page)
{
    pageTab->setCurrentPage(page);
}

QWidget* KMLDonkey::buildInterface()
{
#ifdef HAVE_KTABWIDGET
    pageTab = new KTabWidget(this, "pages");
    pageTab->setTabReorderingEnabled(true);
#else
    pageTab = new QTabWidget(this, "pages");
#endif

    serverPage = new ServerPage(pageTab);
    searchPage = new SearchPage(pageTab); // Search page
    downloadPage = new DownloadPage(pageTab);
    sharePage = new SharePage(pageTab);
    friendPage = new FriendPage(pageTab);
    consolePage = new ConsolePage(pageTab);
    debugPage = 0;

    // Statusbar

    statInfo = new QLabel(this);
    statServer = new QLabel(this);
    statFiles = new QLabel(this);
    statShare = new QLabel(this);
    statTransfer = new QLabel(this);
    statRate = new QLabel(this);

    statusBar()->addWidget(statInfo,1);
    statusBar()->addWidget(statServer);
    statusBar()->addWidget(statFiles);
    statusBar()->addWidget(statShare);
    statusBar()->addWidget(statTransfer);
    statusBar()->addWidget(statRate);

    // Interface done, we now need to restore some state that needs to be done before anything else is called.

    KConfig* conf = KGlobal::config();
    KIconLoader* icons = KGlobal::iconLoader();
    conf->setGroup("State");
    QStringList pageOrder = conf->readListEntry("PageOrder");
    if (pageOrder.empty())
	pageOrder = QStringList::split(",", "serverPage,searchPage,downloadPage,sharePage,friendPage,consolePage");
    QStringList::Iterator it;
    for (it = pageOrder.begin(); it != pageOrder.end(); ++it) {
	if (*it == "serverPage")
	    addPage(serverPage, icons->loadIconSet("connect_creating", KIcon::Small), i18n("Servers"));
	else if (*it == "searchPage")
	    addPage(searchPage, icons->loadIconSet("find", KIcon::Small), i18n("Search"));
	else if (*it == "downloadPage")
	    addPage(downloadPage, icons->loadIconSet("down", KIcon::Small), i18n("Downloads"));
	else if (*it == "sharePage")
	    addPage(sharePage, icons->loadIconSet("up", KIcon::Small), i18n("Uploads"));
	else if (*it == "friendPage")
	    addPage(friendPage, icons->loadIconSet("personal", KIcon::Small), i18n("Friends"));
	else if (*it == "consolePage")
	    addPage(consolePage, icons->loadIconSet("openterm", KIcon::Small), i18n("Console"));
    }
    pageTab->setCurrentPage(conf->readNumEntry("CurrentPage"));

    return (QWidget*) pageTab;
}

void KMLDonkey::setupActions()
{
    KStdAction::quit(this, SLOT(closing()), actionCollection());

    m_toolbarAction = KStdAction::showToolbar(this, SLOT(optionsShowToolbar()), actionCollection(), "toolBar");
    m_statusbarAction = KStdAction::showStatusbar(this, SLOT(optionsShowStatusbar()), actionCollection(), "statusBar");

    KStdAction::keyBindings(this, SLOT(optionsConfigureKeys()), actionCollection(), "configure_keyBindings");
    KStdAction::configureToolbars(this, SLOT(optionsConfigureToolbars()), actionCollection(), "configure_Toolbars");
    KStdAction::preferences(this, SLOT(optionsPreferences()), actionCollection(), "configure_kmldonkey");

    (void)new KAction(i18n("Configure co&nnection..."), "connect_creating", 0, this, SLOT(optionsConfigureConnection()),
		      actionCollection(), "configure_connection");
    (void)new KAction(i18n("Configure &MLDonkey..."), "kmldonkey", 0, this, SLOT(optionsConfigureMLDonkey()),
		      actionCollection(), "configure_mldonkey");

    m_connectAction = new KActionMenu(i18n("&Connect to core"), "connect_creating",
				      actionCollection(), "connect_core");
    m_connectAction->setDelayed(true);
    connectMapper = new QSignalMapper(this);
    connect(connectMapper, SIGNAL(mapped(const QString&)), SLOT(actionConnectCore(const QString&)));
    connect(m_connectAction, SIGNAL(activated()), SLOT(actionConnectCore()));
    populateConnectMenu();

    (void)new KAction(i18n("&Disconnect from core"), "connect_no", 0, this, SLOT(actionDisconnectCore()),
                      actionCollection(), "disconnect_core");

    pageMapper = new QSignalMapper(this);
    connect(pageMapper, SIGNAL(mapped(int)), SLOT(activatePage(int)));
    for (int i=1; i<10; i++) {
	KAction* action = new KAction(i18n("Activate page %1").arg(i), 0, QString("Ctrl+%1").arg(i), pageMapper, SLOT(map()),
				      actionCollection(), QString("activate_page_%1").arg(i).latin1());
	pageMapper->setMapping(action, i - 1);
    }

    QPtrListIterator<KMLDonkeyPage> it(pages);
    for ( ; it.current() ; ++it ) (*it)->setupActions(actionCollection());

    createGUI();
}

void KMLDonkey::populateConnectMenu()
{
    QPtrListIterator<KAction> ait(connectActions);
    for (; ait.current(); ++ait)
	m_connectAction->remove(ait.current());
    connectActions.clear();
    QStringList list(hostManager->hostList());
    QStringList::iterator it;
    for (it = list.begin(); it != list.end(); ++it) {
	KAction* action = new KAction(*it, 0, this);
	connect(action, SIGNAL(activated()), connectMapper, SLOT(map()));
	connectMapper->setMapping(action, *it);
	m_connectAction->insert(action);
	connectActions.append(action);
    }
}

// Called when the quit action is activated
void KMLDonkey::closing()
{
    if (SaveStatOnExit) saveState(KGlobal::config());
    kapp->quit();
}

// Called when the window is closed by the window manager
bool KMLDonkey::queryClose()
{
    if (SaveStatOnExit) saveState(KGlobal::config());
    return true;
}

void KMLDonkey::enableDebugConsole(bool enable)
{
    if (enable && !debugPage) {
	debugPage = new DebugPage(pageTab);
	addPage(debugPage, KGlobal::iconLoader()->loadIconSet("gear", KIcon::Small), i18n("Debug"));
    } else if (!enable && debugPage) {
	removePage(debugPage);
	delete debugPage;
	debugPage = 0;
    }
}

void KMLDonkey::restoreState(KConfig* conf)
{
    conf->setGroup("Options");
    m_toolbarAction->setChecked( conf->readBoolEntry("ShowToolbar", true) );
    emit optionsShowToolbar();
    m_statusbarAction->setChecked( conf->readBoolEntry("ShowStatusbar", true) );
    emit optionsShowStatusbar();
    ShowStatusbarLabels = conf->readBoolEntry("ShowStatusbarLabels", ShowStatusbarLabels);
    SaveStatOnExit = conf->readBoolEntry("SaveStatOnExit", SaveStatOnExit);
    debugEnabled = conf->readBoolEntry("EnableDebugConsole", false);
    enableDebugConsole(debugEnabled);
    coreCharset = conf->readEntry("CoreCharset", "iso 8859-1");
    DonkeyMessage::setStringCodec(KGlobal::charsets()->codecForName(coreCharset));

    conf->setGroup("Fonts");
    consoleFont = KGlobalSettings::fixedFont();
    consoleFont = conf->readFontEntry("ConsoleFont", &consoleFont);
    listFont = KGlobalSettings::generalFont();
    listFont = conf->readFontEntry("ListFont", &listFont);

    conf->setGroup("Colors");
    QColor color;

    colorServerView = conf->readBoolEntry("colorServerView", false);
    color = QColor(0x99, 0x00, 0x00);
    colorServerNotConnected = conf->readColorEntry("NotConnected", &color);
    color = QColor(0xAA, 0xAA, 0xAA);
    colorServerBlacklisted = conf->readColorEntry("Blacklisted", &color);
    color = QColor(0x00, 0x00, 0x80);
    colorServerConnecting = conf->readColorEntry("Connecting", &color);
    color = QColor(0x15, 0x70, 0x15);
    colorServerConnected = conf->readColorEntry("Connected", &color);

    colorSearchView = conf->readBoolEntry("colorSearchView", false);
    color = QColor(0x99, 0x00, 0x00);
    colorSearchFewSources = conf->readColorEntry("SearchFewSources", &color);
    color = QColor(0x15, 0x70, 0x15);
    colorSearchManySources = conf->readColorEntry("SearchManySources", &color);
    color = QColor(0xAA, 0xAA, 0xAA);
    colorSearchAlreadyDone = conf->readColorEntry("SearchAlreadyDone", &color);
    searchThreshold = conf->readNumEntry("SearchThreshold", 5);

    colorDownloadView = conf->readBoolEntry("colorDownloadView", false);
    color = QColor(0x15, 0x70, 0x15);
    colorDownloadDownloading = conf->readColorEntry("Downloading", &color);
    color = QColor(0xAA, 0xAA, 0xAA);
    colorDownloadPaused = conf->readColorEntry("Paused", &color);
    color = QColor(0x99, 0x00, 0x00);
    colorDownloadLooking = conf->readColorEntry("Looking", &color);
    color = QColor(0x00, 0x00, 0x00);
    colorDownloadQueued = conf->readColorEntry("Queued", &color);

    color = Qt::red;
    availabilityColours.setColor(QColorGroup::Background, conf->readColorEntry("NoSources", &color));
    color = Qt::green;
    availabilityColours.setColor(QColorGroup::Foreground, conf->readColorEntry("Complete", &color));
    color = QColor(240, 255, 128, QColor::Hsv);
    availabilityColours.setColor(QColorGroup::Dark, conf->readColorEntry("FewSources", &color));
    color = QColor(240, 128, 255, QColor::Hsv);
    availabilityColours.setColor(QColorGroup::Light, conf->readColorEntry("ManySources", &color));

    availabilityThreshold = conf->readNumEntry("Threshold", 25);

    colorSourceView = conf->readBoolEntry("colorSourceView", false);
    color = QColor(0x99, 0x00, 0x00);
    colorSourceNotConnected = conf->readColorEntry("SourceNotConnected", &color);
    color = QColor(0xAA, 0xAA, 0xAA);
    colorSourceBlacklisted = conf->readColorEntry("SourceBlacklisted", &color);
    color = QColor(0x00, 0x00, 0x80);
    colorSourceConnecting = conf->readColorEntry("SourceConnecting", &color);
    color = QColor(0xA0, 0x64, 0x09);
    colorSourceQueued = conf->readColorEntry("SourceQueued", &color);
    color = QColor(0x15, 0x70, 0x15);
    colorSourceDownloading = conf->readColorEntry("SourceDownloading", &color);

    // Pages
    QPtrListIterator<KMLDonkeyPage> pit(pages);
    for ( ; pit.current() ; ++pit ) (*pit)->restoreState(conf);
}

void KMLDonkey::saveState(KConfig* conf)
{
    conf->setGroup("State");
    conf->writeEntry("CurrentPage", pageTab->currentPageIndex());
    int i;
    QStringList pageOrder;
    for (i=0; i<pageTab->count(); i++)
	pageOrder.append(pageTab->page(i)->name());
    conf->writeEntry("PageOrder", pageOrder);
    conf->setGroup("Options");
    conf->writeEntry("ShowToolbar", m_toolbarAction->isChecked());
    conf->writeEntry("ShowStatusbar", m_statusbarAction->isChecked());
    conf->writeEntry("ShowStatusbarLabels", ShowStatusbarLabels);
    conf->writeEntry("SaveStatOnExit", SaveStatOnExit);
    conf->writeEntry("EnableDebugConsole", debugEnabled);
    conf->writeEntry("CoreCharset", coreCharset);

    // Pages
    QPtrListIterator<KMLDonkeyPage> it(pages);
    for ( ; it.current() ; ++it ) (*it)->saveState(conf);
}

void KMLDonkey::actionConnectCore()
{
    reconnect = 0;
    connectDonkey();
}

void KMLDonkey::actionConnectCore(const QString& host)
{
    if (hostManager->validHostName(host)) {
	lastHost = host;
	reconnect = 0;
	connectDonkey();
    }
}

void KMLDonkey::actionDisconnectCore()
{
    reconnect = 0;
    disconnectDonkey();
}

void KMLDonkey::dragEnterEvent(QDragEnterEvent* event)
{
    event->accept(QUriDrag::canDecode(event));
}

void KMLDonkey::dropEvent(QDropEvent* event)
{
    QStringList uri;

    if (QUriDrag::decodeToUnicodeUris(event, uri))
    {
	QStringList::Iterator it;
	for (it = uri.begin(); it != uri.end(); ++it)
	    donkey->submitURL(*it);
    }
}

void KMLDonkey::optionsShowToolbar()
{
    // this is all very cut and paste code for showing/hiding the
    // toolbar
    if (m_toolbarAction->isChecked())
        toolBar()->show();
    else
        toolBar()->hide();
}

void KMLDonkey::optionsShowStatusbar()
{
    // this is all very cut and paste code for showing/hiding the
    // statusbar
    if (m_statusbarAction->isChecked())
        statusBar()->show();
    else
        statusBar()->hide();
}

void KMLDonkey::optionsConfigureKeys()
{
    KKeyDialog::configureKeys(actionCollection(), xmlFile());
}

void KMLDonkey::optionsConfigureToolbars()
{
    saveMainWindowSettings(KGlobal::config(), autoSaveGroup());
    KEditToolbar dlg(actionCollection());
    connect(&dlg, SIGNAL(newToolbarConfig()), this, SLOT(newToolbarConfig()));
    if (dlg.exec()) {
	createGUI();
    }
}

void KMLDonkey::newToolbarConfig()
{
    // ...if you use any action list, use plugActionList on each here...
    applyMainWindowSettings(KGlobal::config(), autoSaveGroup());
}

void KMLDonkey::optionsPreferences()
{
   if (! prefs) { // on first access create the KMLDonkeyPreferences instance
        prefs = new KMLDonkeyPreferences(this);
        connect(prefs, SIGNAL(applyClicked()), this, SLOT(applyPreferences()));
        connect(prefs, SIGNAL(okClicked()), this, SLOT(applyPreferences()));
    }

    prefs->generalPage->ShowToolbarCheckbox->setChecked( m_toolbarAction->isChecked() );
    prefs->generalPage->ShowStatusbarCheckbox->setChecked( m_statusbarAction->isChecked() );
    prefs->generalPage->StatusbarLabelsCheckbox->setChecked( ShowStatusbarLabels );
    prefs->generalPage->SaveStatCheckbox->setChecked( SaveStatOnExit );
    prefs->generalPage->PersistentReconnectCheckbox->setChecked( persistentReconnect );
    prefs->generalPage->DebugConsoleCheckbox->setChecked( debugEnabled );
    prefs->generalPage->setEncoding(coreCharset);

    prefs->colorPage->colorServerCheckbox->setChecked(colorServerView);
    prefs->colorPage->colorServerNotConnected->setColor(colorServerNotConnected);
    prefs->colorPage->colorServerBlacklisted->setColor(colorServerBlacklisted);
    prefs->colorPage->colorServerConnecting->setColor(colorServerConnecting);
    prefs->colorPage->colorServerConnected->setColor(colorServerConnected);
    emit prefs->colorPage->colorServer(colorServerView);

    prefs->colorPage->colorSearchCheckbox->setChecked(colorSearchView);
    prefs->colorPage->colorSearchFewSources->setColor(colorSearchFewSources);
    prefs->colorPage->colorSearchManySources->setColor(colorSearchManySources);
    prefs->colorPage->colorSearchAlreadyDone->setColor(colorSearchAlreadyDone);
    prefs->colorPage->searchSourcesThresholdSlider->setValue(searchThreshold);
    emit prefs->colorPage->colorSearch(colorSearchView);

    prefs->colorPage->colorDownloadCheckbox->setChecked(colorDownloadView);
    prefs->colorPage->downloadingColorSelect->setColor(colorDownloadDownloading);
    prefs->colorPage->pausedColorSelect->setColor(colorDownloadPaused);
    prefs->colorPage->lookingColorSelect->setColor(colorDownloadLooking);
    prefs->colorPage->queuedColorSelect->setColor(colorDownloadQueued);
    emit prefs->colorPage->colorDownload(colorDownloadView);

    prefs->colorPage->noSourcesColorSelect->setColor(availabilityColours.background());
    prefs->colorPage->fewSourcesColorSelect->setColor(availabilityColours.dark());
    prefs->colorPage->manySourcesColorSelect->setColor(availabilityColours.light());
    prefs->colorPage->completeColorSelect->setColor(availabilityColours.foreground());
    prefs->colorPage->availabilityThresholdSlider->setValue(availabilityThreshold);

    prefs->colorPage->colorSourcesCheckbox->setChecked(colorSourceView);
    prefs->colorPage->colorSourceNotConnected->setColor(colorSourceNotConnected);
    prefs->colorPage->colorSourceBlacklisted->setColor(colorSourceBlacklisted);
    prefs->colorPage->colorSourceConnecting->setColor(colorSourceConnecting);
    prefs->colorPage->colorSourceQueued->setColor(colorSourceQueued);
    prefs->colorPage->colorSourceDownloading->setColor(colorSourceDownloading);

    prefs->fontPage->consoleFontSelect->setFont(consoleFont);
    prefs->fontPage->listFontSelect->setFont(listFont);

    QPtrListIterator<KMLDonkeyPage> it(pages);
    for ( ; it.current() ; ++it ) (*it)->configurePrefsDialog(prefs);

    prefs->show();
}

void KMLDonkey::applyPreferences()
{
    if (! prefs) return; // just to be sure ;-)

    m_toolbarAction->setChecked( prefs->generalPage->ShowToolbarCheckbox->isChecked() );
    emit optionsShowToolbar();
    m_statusbarAction->setChecked( prefs->generalPage->ShowStatusbarCheckbox->isChecked() );
    emit optionsShowStatusbar();
    ShowStatusbarLabels = prefs->generalPage->StatusbarLabelsCheckbox->isChecked();
    SaveStatOnExit = prefs->generalPage->SaveStatCheckbox->isChecked();
    persistentReconnect = prefs->generalPage->PersistentReconnectCheckbox->isChecked();
    debugEnabled = prefs->generalPage->DebugConsoleCheckbox->isChecked();
    enableDebugConsole(debugEnabled);

    QString newCharset = prefs->generalPage->getEncoding();
    if (newCharset != coreCharset) {
	coreCharset = newCharset;
	DonkeyMessage::setStringCodec(KGlobal::charsets()->codecForName(coreCharset));
	connectDonkey();
    }

    KConfig* conf = KGlobal::config();

    conf->setGroup("Colors");

    colorServerView = prefs->colorPage->colorServerCheckbox->isChecked();
    colorServerNotConnected = prefs->colorPage->colorServerNotConnected->color();
    colorServerBlacklisted = prefs->colorPage->colorServerBlacklisted->color();
    colorServerConnecting = prefs->colorPage->colorServerConnecting->color();
    colorServerConnected = prefs->colorPage->colorServerConnected->color();
    conf->writeEntry("colorServerView", colorServerView);
    conf->writeEntry("NotConnected", colorServerNotConnected);
    conf->writeEntry("Blacklisted", colorServerBlacklisted);
    conf->writeEntry("Connecting", colorServerConnecting);
    conf->writeEntry("Connected", colorServerConnected);

    colorSearchView = prefs->colorPage->colorSearchCheckbox->isChecked();
    colorSearchFewSources = prefs->colorPage->colorSearchFewSources->color();
    colorSearchManySources = prefs->colorPage->colorSearchManySources->color();
    colorSearchAlreadyDone = prefs->colorPage->colorSearchAlreadyDone->color();
    searchThreshold = prefs->colorPage->searchSourcesThresholdSlider->value();
    conf->writeEntry("colorSearchView", colorSearchView);
    conf->writeEntry("SearchFewSources", colorSearchFewSources);
    conf->writeEntry("SearchManySources", colorSearchManySources);
    conf->writeEntry("SearchAlreadyDone", colorSearchAlreadyDone);
    conf->writeEntry("SearchThreshold", searchThreshold);

    colorDownloadView = prefs->colorPage->colorDownloadCheckbox->isChecked();
    colorDownloadDownloading = prefs->colorPage->downloadingColorSelect->color();
    colorDownloadPaused = prefs->colorPage->pausedColorSelect->color();
    colorDownloadLooking = prefs->colorPage->lookingColorSelect->color();
    colorDownloadQueued = prefs->colorPage->queuedColorSelect->color();
    conf->writeEntry("colorDownloadView", colorDownloadView);
    conf->writeEntry("Downloading", colorDownloadDownloading);
    conf->writeEntry("Paused", colorDownloadPaused);
    conf->writeEntry("Looking", colorDownloadLooking);
    conf->writeEntry("Queued", colorDownloadQueued);

    availabilityColours.setColor(QColorGroup::Background, prefs->colorPage->noSourcesColorSelect->color());
    availabilityColours.setColor(QColorGroup::Dark, prefs->colorPage->fewSourcesColorSelect->color());
    availabilityColours.setColor(QColorGroup::Light, prefs->colorPage->manySourcesColorSelect->color());
    availabilityColours.setColor(QColorGroup::Foreground, prefs->colorPage->completeColorSelect->color());
    availabilityThreshold = prefs->colorPage->availabilityThresholdSlider->value();
    conf->writeEntry("NoSources", availabilityColours.background());
    conf->writeEntry("FewSources", availabilityColours.dark());
    conf->writeEntry("ManySources", availabilityColours.light());
    conf->writeEntry("Complete", availabilityColours.foreground());
    conf->writeEntry("Threshold", availabilityThreshold);

    colorSourceView = prefs->colorPage->colorSourcesCheckbox->isChecked();
    colorSourceNotConnected = prefs->colorPage->colorSourceNotConnected->color();
    colorSourceBlacklisted = prefs->colorPage->colorSourceBlacklisted->color();
    colorSourceConnecting = prefs->colorPage->colorSourceConnecting->color();
    colorSourceQueued = prefs->colorPage->colorSourceQueued->color();
    colorSourceDownloading = prefs->colorPage->colorSourceDownloading->color();
    conf->writeEntry("colorSourceView", colorSourceView);
    conf->writeEntry("SourceNotConnected", colorSourceNotConnected);
    conf->writeEntry("SourceBlacklisted", colorSourceBlacklisted);
    conf->writeEntry("SourceConnecting", colorSourceConnecting);
    conf->writeEntry("SourceQueued", colorSourceQueued);
    conf->writeEntry("SourceDownloading", colorSourceDownloading);

    conf->setGroup("Fonts");

    listFont = prefs->fontPage->listFontSelect->font();
    consoleFont = prefs->fontPage->consoleFontSelect->font();

    conf->writeEntry("ConsoleFont", consoleFont);
    conf->writeEntry("ListFont", listFont);

    QPtrListIterator<KMLDonkeyPage> it(pages);
    for ( ; it.current() ; ++it ) (*it)->applyPreferences(prefs);

    saveState(conf); // finally save everything
}

void KMLDonkey::optionsConfigureConnection()
{
    QStringList foo;
    foo.append("kcmdonkey");
    QString err;
    if (KApplication::kdeinitExec("kcmshell", foo, &err))
        KMessageBox::detailedError(this, i18n("Failed to launch the MLDonkey control panel applet."), err);
}

void KMLDonkey::optionsConfigureMLDonkey()
{
    if (!donkey->isConnected()) {
	KMessageBox::error(this, i18n("You must connect to a running MLDonkey core before you can configure it!"));
	return;
    }
    MLDonkeyConfig* cfg = new MLDonkeyConfig(this);
    if (cfg) cfg->show();
}

void KMLDonkey::changeStatusbar(const QString& text)
{
    // display the text on the statusbar
    statInfo->setText(text);
}

void KMLDonkey::changeCaption(const QString& text)
{
    // display the text on the caption
    setCaption(text);
}

void KMLDonkey::hostListUpdated()
{
    populateConnectMenu();
    if (!hostManager->validHostName(lastHost))
	lastHost = hostManager->defaultHostName();
    connectDonkey();
}

void KMLDonkey::connectDonkey()
{
    donkey->connectDonkey(hostManager->hostProperties(lastHost));
    changeStatusbar(i18n("Connecting..."));
}

void KMLDonkey::disconnectDonkey()
{
    donkey->disconnectDonkey();
}

void KMLDonkey::donkeyDisconnected(int err)
{
    changeStatusbar(i18n("Disconnected."));

    statServer->setText(QString::null);
    statFiles->setText(QString::null);
    statShare->setText(QString::null);
    statTransfer->setText(QString::null);
    statRate->setText(QString::null);

    switch (err) {
    case DonkeyProtocol::AuthenticationError:
	KMessageBox::error(this, i18n("Incorrect password."));
        reconnect = 0;
	break;
    case DonkeyProtocol::ConnectionRefusedError:
	if (!reconnect) KMessageBox::error(this, i18n("Connection refused."));
	reconnect = persistentReconnect;
	break;
    case DonkeyProtocol::HostNotFoundError:
	KMessageBox::error(this, i18n("KMLDonkey Communication Error"),
			      i18n("Host not found."));
        reconnect = 0;
	break;
    case DonkeyProtocol::SocketReadError:
	if (!reconnect) KMessageBox::error(this, i18n("Read error. Applet disconnected."));
	reconnect = persistentReconnect;
	break;
    case DonkeyProtocol::ObsoleteProtocolError:
	KMessageBox::error(this,
			      i18n("Your mldonkey core uses an obsolete communications protocol. "
				   "Please upgrade it to a more recent version!"));
        reconnect = 0;
	break;
    default:
	reconnect = persistentReconnect;
	break;
    }

    QPtrListIterator<KMLDonkeyPage> it(pages);
    for ( ; it.current() ; ++it ) (*it)->clear();
}

void KMLDonkey::donkeyConnected()
{
    const DonkeyHost& host = hostManager->hostProperties(lastHost);
    changeStatusbar(i18n("Connected %1@%2:%3 (%4)")
		    .arg(host.username())
		    .arg(host.address())
		    .arg(QString::number(host.guiPort()))
		    .arg(host.name()));
    reconnect = 0;
}

void KMLDonkey::checkReconnect()
{
    if (persistentReconnect && reconnect && (!donkey || donkey->donkeyStatus() == QSocket::Idle))
        connectDonkey();
}

void KMLDonkey::sendConsoleMessage(const QString& txt)
{
    consolePage->sendConsoleMessage(txt);
}

void KMLDonkey::updateStatus(int64 ul, int64 dl, int64, int nsh, int tul, int tdl, int, int, int ndl, int ncp)
{
    statServer->setText( (ShowStatusbarLabels ? i18n("Server") + " " : QString::null)
			 + QString::number(donkey->connectedServerCount()) + "/"
			 + QString::number(donkey->totalServerCount()) );
    statFiles->setText( (ShowStatusbarLabels ? i18n("Files") + " " : QString::null)
			+ QString::number(ncp) + "/" + QString::number(ndl) );
    statShare->setText( (ShowStatusbarLabels ? i18n("Shared") + " " : QString::null)
			+ QString::number(nsh) );
    statTransfer->setText( (ShowStatusbarLabels ? i18n("Transfer") + " " : QString::null)
			   + humanReadableSize(ul) + "/" + humanReadableSize(dl) );
    statRate->setText( (ShowStatusbarLabels ? i18n("Speed") + " " : QString::null)
		       + QString::number((double)tul / 1024.0,'f',1) + "/"
		       + QString::number((double)tdl / 1024.0,'f',1) );
}


/*
 * DCOP methods (as defined in kmldonkeyiface.h)
 */

void KMLDonkey::submitURL(QString url)
{
    donkey->submitURL(url);
}

void KMLDonkey::consoleCommand(QString command)
{
    sendConsoleMessage(command);
}



#include "kmldonkey.moc"
