/*
 * $Id: mlconfig.cpp,v 1.7 2003/07/30 13:46:30 dipesh Exp $
 *
 * Copyright (C) 2003 Petter E. Stokke <gibreel@gibreel.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Log: mlconfig.cpp,v $
 * Revision 1.7  2003/07/30 13:46:30  dipesh
 * added directory-browser
 *
 * Revision 1.6  2003/07/26 13:38:58  dipesh
 * Be more flexible with Networks-subitems.
 *
 * Revision 1.5  2003/07/25 16:46:15  gibreel
 * Casting some QStrings explicitly so as not to confuse certain older
 * compilers.
 *
 * Revision 1.4  2003/07/25 15:42:53  dipesh
 * Added the enableNetwork() donkey-function and changed "Configure MLDonkey" to
 * handle enable/disable Networks another way. The way used previous was working
 * as well, but the core didn't send something back to show that that a Network
 * was enabled/disabled. So, to be able to work at the search/network combo always
 * with the actual enabled networks the new function is used now. Stupid
 * core-behaviour anyway.
 *
 * Revision 1.3  2003/07/22 20:43:25  gibreel
 * Got rid of a compile warning.
 *
 * Revision 1.2  2003/07/22 19:16:03  dipesh
 * Just renamed the saveSize() slot to closeDialog() :-)
 *
 * Revision 1.1  2003/07/20 20:47:41  dipesh
 * Added the initial "Configure MLDonkey" feature.
 *
 */

#include "mlconfig.h"
#include "kmldonkey.h"

#include <qstring.h>
#include <qmap.h>
#include <qlayout.h>
#include <qlabel.h>
#include <qvbox.h>
#include <qhbox.h>
#include <qregexp.h>

#include <kglobal.h>
#include <kconfig.h>
#include <klocale.h>
#include <ktextedit.h>
#include <qcheckbox.h>
#include <qlineedit.h>

#include <kdebug.h>
#include <klineeditdlg.h>
#include <kpushbutton.h>
#include <kfiledialog.h>

/*** MLDonkeyConfigPageItem ***/

MLDonkeyConfigPageItem::MLDonkeyConfigPageItem(QWidget *parent, const QString& name, const QString& value)
{
    this->parent = parent;

    QString n = name;
    n.replace(QRegExp("\\-|_"), " ");

    if (value == "true" || value == "false") { // boolean
        Type = MLDonkeyConfigPageItem::etCheckbox;
        checkbox = new QCheckBox(parent);
        checkbox->setText(n);
        checkbox->setChecked(value == "true");
    }
    else if (value.find("\n") >= 0) { // multiline string
        Type = MLDonkeyConfigPageItem::etTextedit;
        QVBox *b = new QVBox(parent);
        new QLabel(n, b);
        textedit = new KTextEdit(b);
        textedit->setTextFormat(Qt::PlainText);
        textedit->setWordWrap(QTextEdit::NoWrap);
        textedit->setText(value);
    }
    else if (name.find(QRegExp("^directory|directory$")) >= 0) {
        Type = MLDonkeyConfigPageItem::etDirectory;
        QHBox *b = new QHBox(parent);
        new QLabel(n, b);
        lineedit = new KLineEdit(b);
        lineedit->setText(value);
        KPushButton *btn = new KPushButton(b);
        btn->setText("...");
        connect(btn, SIGNAL( clicked() ), this, SLOT( slotDirBtnClicked() ));
    }
    else { // string
        Type = MLDonkeyConfigPageItem::etLineedit;
        QHBox *b = new QHBox(parent);
        new QLabel(n, b);
        lineedit = new KLineEdit(b);
        lineedit->setText(value);
    }
}

void MLDonkeyConfigPageItem::slotDirBtnClicked()
{
    QString dir = KFileDialog::getExistingDirectory(lineedit->text(), parent, i18n("Choose directory"));
    if (dir) lineedit->setText(dir);
}

QString MLDonkeyConfigPageItem::getValue()
{
    switch (Type) {
        case MLDonkeyConfigPageItem::etCheckbox:
            return checkbox->isChecked() ? "true" : "false";
        case MLDonkeyConfigPageItem::etDirectory:
        case MLDonkeyConfigPageItem::etTextedit:
            return textedit->text();
        case MLDonkeyConfigPageItem::etLineedit:
            return lineedit->text();
    }
    return QString::null;
}

/*** MLDonkeyConfigPage ***/

MLDonkeyConfigPage::MLDonkeyConfigPage(QWidget *parent)
    : QScrollView(parent)
{
    setResizePolicy(QScrollView::AutoOneFit);
    viewport()->setBackgroundMode(PaletteBackground);

    box = new QVBox();
    box->setSpacing(6);
    box->setMargin(11);
    addChild(box);

    setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Minimum);
    setMinimumSize(sizeHint());
}

void MLDonkeyConfigPage::addOption(const QString& name, const QString& value)
{
    items[name] = new MLDonkeyConfigPageItem(box, name, value);
}

void MLDonkeyConfigPage::addEnableNetwork(const QString& name, const int nwnum, const bool enable)
{
    QCheckBox *cb = new QCheckBox(box);
    cb->setText(name);
    cb->setChecked(enable);
    enableNets.replace(nwnum, cb);
}

/*** MLDonkeyConfig ***/

MLDonkeyConfig::MLDonkeyConfig(QWidget* parent)
    : KDialogBase(TreeList, i18n("Configure MLDonkey"),
                  Ok|Cancel, Ok, parent) // |Default
{
    setRootIsDecorated(false);

    // define how to present the options
    // first String is the title and the second a RegExp witch have to match to the optionname to be on this page.
    QString defaultpage = i18n("Generals"); // default page
    tree.insert(i18n("Client"),              "(^client_)|^force_client_ip$");
    tree.insert(i18n("Restrictions"),        "^(max_|maximal_|min_|minimal_|ban_|black_list)|allow_|allowed_|propagate_");
    tree.insert(i18n("GUI"),                 "(^gui_|_gui$)|_gui_|^options_");
    tree.insert(i18n("Log"),                 "^log_");
    tree.insert(i18n("Chat"),                "^(chat|max_dialog_history$)");
    tree.insert(i18n("Telnet"),              "^(telnet|term)_");
    tree.insert(i18n("Web"),                 "html|http|new_print_search|commands_frame_height");
    tree.insert(i18n("Mail"),                "mail|smtp|filename_in_subject");

    tree.insert(i18n("Networks"),            "^enable_overnet$"); // the Networks-page is something special cause here we place the enableNetwork-options as well
    tree.insert(i18n("Networks")+"/$1",      "(.+)-(.+)");
    tree.insert(i18n("Networks")+"/Donkey",  "(^(emule|donkey)|^(force_high_id|port|protocol_version|enable_server|use_server_id|upload_power|initial_score)$)");
    tree.insert(i18n("Networks")+"/Overnet", "^overnet_");

    // add pages and fill them with options
    treeMap::Iterator treeit;
    options = KMLDonkey::App->donkey->OptionsList();
    QString name, value, page;
    for (QMap<QString, QString>::Iterator optionsit = options.begin(); optionsit != options.end(); ++optionsit) {
        name = optionsit.key();
        value = optionsit.data();
        page = defaultpage;

        for (treeit = tree.begin(); treeit != tree.end(); ++treeit) {
            if (treeit.data().isEmpty()) continue;
            QRegExp rx( treeit.data() );
            if (rx.search(name) < 0) continue;
            page = treeit.key();
            page.replace("$1", rx.cap(1));
            break;
        }

        if (page == defaultpage && name.startsWith("enable_")) continue; // ignore the enable_<network> options cause we handle it another way

        if (! pages[page]) { // create page(s) if they don't already exists
            QString title;
            QStringList titlelist = QStringList::split("/", page);
            for (unsigned int i = 0; i < titlelist.count(); i++) {
                title += i == 0 ? titlelist[i] : "/" + titlelist[i];
                if (pages[title]) continue;
                pages[title] = new MLDonkeyConfigPage((QWidget*) addVBoxPage(QStringList::split("/", title), i18n("Configure %1").arg(title)) );
            }
        }

        pages[page]->addOption(name, value);
    }

    // finally add the enable-<network> stuff
    if (pages[i18n("Networks")]) {
        Network *net;
        QIntDictIterator<Network> nwit( KMLDonkey::App->donkey->availableNetworks() );
        for ( ; nwit.current(); ++nwit ) {
            net = nwit.current();
            pages[i18n("Networks")]->addEnableNetwork("enable " + net->networkName(), net->networkNo(), net->networkEnabled());
        }
    }

    // align the widgets top
    for (pagesMap::Iterator pagesit = pages.begin(); pagesit != pages.end(); ++pagesit)
        pagesit.data()->box->setStretchFactor(new QWidget(pagesit.data()->box), 1);

    setInitialSize(configDialogSize("MLConfigDialog"));
    connect(this, SIGNAL(finished()), this, SLOT(closeDialog()));
    connect(this, SIGNAL(okClicked()), this, SLOT(saveOptions()));

    // show default page
    if (pages[defaultpage]) {
        QWidget *w = (QWidget*) pages[defaultpage]->parent();
        if (w) showPage(pageIndex(w));
    }
}

void MLDonkeyConfig::saveOptions()
{
    // set the modified options
    QMap<QString, QString> opts;
    QString s, t;
    for (pagesMap::Iterator pagesit = pages.begin(); pagesit != pages.end(); ++pagesit) {
        for (QMap<QString, MLDonkeyConfigPageItem*>::Iterator it = pagesit.data()->items.begin(); it != pagesit.data()->items.end(); ++it) {
            s = options[it.key()].isEmpty() ? QString("") : options[it.key()];
            t = it.data()->getValue().isEmpty() ? QString("") : it.data()->getValue();
            if (s == t) continue; // only save modified options
            opts[it.key()] = t;
        }
    }
    KMLDonkey::App->donkey->setOptions(opts);

    // enable/disable networks
    if (pages[i18n("Networks")]) {
        Network *net;
        QIntDictIterator<QCheckBox> nwit( pages[i18n("Networks")]->enableNets );
        for ( ; nwit.current(); ++nwit ) {
            net = (Network*) KMLDonkey::App->donkey->findNetworkNo( (int)nwit.currentKey() );
            if (! net) continue;
            if (net->networkEnabled() == nwit.current()->isChecked()) continue; // modified?
            KMLDonkey::App->donkey->enableNetwork(nwit.currentKey(), nwit.current()->isChecked());
        }
    }
}

void MLDonkeyConfig::closeDialog()
{
    saveDialogSize("MLConfigDialog");
    delayedDestruct();
}

#include "mlconfig.moc"
