/* -*- c++ -*-
 *
 * $Id: search.cpp
 *
 * Copyright (C) 2003 Petter E. Stokke <gibreel@gibreel.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Log: search.cpp,v $
 * Revision 1.36  2003/07/27 20:24:26  gibreel
 * Added KActions for switching between pages.
 *
 * Revision 1.35  2003/07/25 15:42:53  dipesh
 * Added the enableNetwork() donkey-function and changed "Configure MLDonkey" to
 * handle enable/disable Networks another way. The way used previous was working
 * as well, but the core didn't send something back to show that that a Network
 * was enabled/disabled. So, to be able to work at the search/network combo always
 * with the actual enabled networks the new function is used now. Stupid
 * core-behaviour anyway.
 *
 * Revision 1.34  2003/07/19 20:20:32  gibreel
 * Uses autoconf to detect the presence of KTabWidget, rather than guessing its
 * presence using KDE_IS_VERSION.
 *
 * Revision 1.33  2003/07/19 14:18:22  gibreel
 * Uses KTabWidget instead of QTabWidget/ClosableTabWidget if available (KDE
 * 3.2 or CVS). ClosableTabWidget API changed to match KTabWidget's.
 *
 * Revision 1.32  2003/07/17 20:45:49  gibreel
 * Added copy icons to all copy actions.
 *
 * Revision 1.31  2003/07/16 17:38:28  dipesh
 * Added Filetype column
 *
 * Revision 1.30  2003/07/14 19:14:44  dipesh
 * display multible filenames as tooltip
 *
 * Revision 1.29  2003/06/30 23:30:35  gibreel
 * Preliminary friend list support. A ton of updates to the libkmldonkey API to
 * accommodate this, most notably improvements to the search handling
 * necessitated by mldonkey's somewhat awkward reporting of friend shares.
 *
 * Revision 1.28  2003/06/30 14:59:22  gibreel
 * Updated lists to support libkmldonkey's new removed signals. Moved pages'
 * connect statements out of KMLDonkey's constructor into their own
 * constructors. Added a debug console displaying dumps of unhandled messages.
 *
 * Revision 1.27  2003/06/29 10:56:30  dipesh
 * converted search-pref "Ask for filename on starting a new download" to the
 * KAction "Download as..." to be more flexible and remember some more
 * searchpage values.
 *
 * Revision 1.26  2003/06/28 22:28:44  gibreel
 * All DonkeyMessage typedefs are now unsigned - having some of them signed
 * caused problems.
 *
 * Revision 1.25  2003/06/28 10:42:25  dipesh
 * Added search-pref "Ask for filename on starting a new download".
 *
 * Revision 1.24  2003/06/27 12:50:39  dipesh
 * added search-pref to optional display numbers of files found on tabs,
 * searchfilter does understand filesizes (fe. >2MB), use function-inline
 * where it make sense and some other things.
 *
 * Revision 1.23  2003/06/26 14:40:46  gibreel
 * Fixed a bug that would copy search URLs with the size as a float instead of
 * an int.
 *
 * Revision 1.22  2003/06/23 18:26:29  dipesh
 * Added initial search-prefs, more later
 *
 * Revision 1.21  2003/06/23 12:59:51  dipesh
 * Added KAction's "Copy search URL/HTML/Hash to Clipboard"
 *
 * Revision 1.20  2003/06/20 00:05:44  dipesh
 * use restoreState/saveState, great work done with the last commits Petter! :-)
 *
 * Revision 1.19  2003/06/19 21:50:45  gibreel
 * Split the various pages off into individual objects, and made a ton of code
 * cleanups, API changes, and not a few bugfixes in the process. The
 * disconnect/reconnect bug, especially, now seems to be gone.
 *
 * Revision 1.18  2003/06/19 17:04:32  gibreel
 * Minor code cleanups and attempt to fix the crash bugs on
 * disconnect/reconnect. Both kmldonkey and libkmldonkey now clean out their
 * state data when the connection with the core is lost.
 *
 * Revision 1.17  2003/06/18 21:37:12  dipesh
 * Some small corrections
 *
 * Revision 1.16  2003/06/14 12:12:57  dipesh
 * Corrected update of networklist
 *
 * Revision 1.15  2003/06/14 02:46:41  dipesh
 * specify which network to search if the GUI protocol is >= 16
 *
 * Revision 1.14  2003/06/13 18:20:01  gibreel
 * Libkmldonkey now uses references instead of pointers everywhere except where
 * it would cause an obvious performance impact, which should lead to less
 * chance of memory leaks and cleaner code in general. Almost everything that
 * should be const is now also const.
 *
 * Revision 1.13  2003/06/13 09:50:53  dipesh
 * optional coloured server- and search lists
 *
 * Revision 1.12  2003/05/24 22:52:04  gibreel
 * Created some classes for constructing search queries in a more flexible
 * manner, and updated the code accordingly where necessary.
 *
 * Revision 1.11  2003/05/24 10:02:46  dipesh
 * fix for a small bug I note
 *
 * Revision 1.10  2003/05/20 17:00:17  dipesh
 * Added search-result filter
 *
 * Revision 1.9  2003/05/19 12:25:07  dipesh
 * fixed Availability-column
 *
 * Revision 1.8  2003/05/18 17:58:22  dipesh
 * Added KAction's Download, ForceDownload and RemoveServers
 *
 * Revision 1.7  2003/05/17 17:27:22  dipesh
 * More search-stuff; added avaibility-column, use the listfont, some smaller fixes
 *
 * Revision 1.6  2003/05/17 15:00:04  dipesh
 * fixed sorting of column "Size" in SearchResultItem and made the sorting
 * case-insensitive in InfoItem
 *
 * Revision 1.5  2003/05/17 10:52:24  dipesh
 * Added more search-options
 *
 * Revision 1.4  2003/05/12 00:21:11  gibreel
 * Added a close button to the search result tabs.
 *
 * Revision 1.3  2003/05/11 19:38:02  dipesh
 * Forgot something, sorry
 *
 * Revision 1.2  2003/05/11 18:55:15  dipesh
 * Replaced the used KTabCtl with a QTabWidget to get more flexibility and fix
 * a small repaint-Bug.
 *
 * Revision 1.1  2003/05/11 08:34:09  dipesh
 * Adding initial Search-functionality
 *
 *
 */

#include <qpainter.h>
#include <qlabel.h>
#include <qvbox.h>
#include <qgroupbox.h>
#include <qscrollview.h>
#include <qsplitter.h>
#include <qcheckbox.h>
#include <qlayout.h>
#include <qpopupmenu.h>
#include <qclipboard.h>
#include <qheader.h>

#include <kdebug.h>
#include <klocale.h>
#include <kconfig.h>
#include <klineedit.h>
#include <kpushbutton.h>
#include <kmessagebox.h>
#include <klistview.h>
#include <kcombobox.h>
#include <kaction.h>
#include <klineeditdlg.h>
#include <kmimetype.h>
#include <kiconloader.h>

#include "kmldonkey.h"
#include "search.h"
#include "searchquery.h"
#include "prefs.h"

#ifdef HAVE_KTABWIDGET
#include <ktabwidget.h>
#else
#include "closabletab.h"
#endif

long int filesizeStr2Int(QString filesize)
{
    long int result;
    QString s = filesize.lower().stripWhiteSpace();
    if (s.isEmpty()) return 0;
    bool ok;

    if (s.endsWith("mb") || s.endsWith("kb")) s.remove(s.length()-1, 1);

    if (s.endsWith("m")) { // megabytes
        result = s.left(s.length()-1).stripWhiteSpace().toLong() * 1048576;
    }
    else if (s.endsWith("k")) { // kilobytes
        result = s.left(filesize.length()-1).stripWhiteSpace().toLong() * 1024;
    }
    else if (s.endsWith("b")) { // bytes
        result = s.left(filesize.length()-1).stripWhiteSpace().toLong();
    }
    else { // only numbers means bytes as well
        result = s.toLong(&ok);
        if (! ok) result = 0;
    }

    return (result < 0 ? 0 : result);
}

/*** SearchResultItem ***/

SearchResultItem::SearchResultItem(KListView *parent, int num, int32 avail, const ResultInfo *searchinfo) : InfoItem(parent,num)
{
    number = num;
    names = searchinfo->resultNames();
    filesize = searchinfo->resultSize();
    availability = avail;
    alreadydone = searchinfo->resultAlreadyDone();
}

SearchResultItem::~SearchResultItem()
{
}

double SearchResultItem::numeric(int col) const
{
    switch (col) {
        case 1:  return (double)filesize;
        case 2:  return (double)availability;
        default: return 0.0;
    }
}

bool SearchResultItem::isNumeric(int col) const
{
    switch (col) {
        case 1:
        case 2: return true;
        default: return false;
    }
}

void SearchResultItem::paintCell(QPainter* p, const QColorGroup& cg, int col, int w, int align)
{
    QColorGroup colgrp(cg);
    if (KMLDonkey::App->colorSearchView) {
        if (alreadydone) {
            colgrp.setColor(QColorGroup::Text, KMLDonkey::App->colorSearchAlreadyDone);
        }
        else if (availability < (int32)KMLDonkey::App->searchThreshold) {
            colgrp.setColor(QColorGroup::Text, KMLDonkey::App->colorSearchFewSources);
        }
        else {
            colgrp.setColor(QColorGroup::Text, KMLDonkey::App->colorSearchManySources);
        }
    }

    p->save();
    InfoItem::paintCell(p, colgrp, col, w, align);
    p->restore();
}

inline int SearchResultItem::getNum() { return number; }
inline QStringList SearchResultItem::getNames() { return names; }
inline int64 SearchResultItem::getSize() { return filesize; }

/*** SearchResultView ***/

SearchResultView::SearchResultView(QWidget *parent, const char *name)
    : InfoList(parent, name),
      QToolTip(viewport())
{
    addColumn( i18n("File name"), 400);

    addColumn( i18n("Size") );
    setColumnWidthMode(1, QListView::Maximum);

    addColumn( i18n("Availability") );
    addColumn( i18n("Network") );

    addColumn( i18n("Filetype") );
    setColumnWidthMode(4, QListView::Maximum);

    addColumn( i18n("Format") );
    addColumn( i18n("Comment") );

    addColumn( i18n("Hash") );
    setColumnWidthMode(7, QListView::Maximum);
}

void SearchResultView::maybeTip(const QPoint& p)
{
    SearchResultItem* item = (SearchResultItem*)itemAt(p);
    if (! item) return;

    QRect r(itemRect(item));
    if (! r.isValid()) return;

    QHeader* h = header();
    if (h && h->mapToLogical( h->cellAt(p.x() - margin()) ) != 0) return;

    QString s;
    for (unsigned int i = 0; i < item->getNames().count(); i++) {
        if (i) s += "<br>";
        s += "<nobr>" + item->getNames()[i] + "</nobr>";
    }

    tip(r, s);
}

/*** SearchResult ***/

SearchResult::SearchResult(int sno, const QString &tabLabel, QWidget *parent) : QVBox(parent, "searchResult")
{
    id = sno;
    this->tabLabel = tabLabel;
    visibleCount = 0;

    statusBox = new QHBox(this, "resultBox");

    QHBox *filterbox = new QHBox(statusBox, "filterBox");
    new QLabel(i18n("Filter"), filterbox);
    filterEdit = new KLineEdit(filterbox);
    connect(filterEdit, SIGNAL( textChanged(const QString &) ), this, SLOT( filterChanged() ));
    filterTimer = new QTimer(this);
    connect(filterTimer, SIGNAL( timeout() ), this, SLOT( filterTimerDone() ));

    statusLabel = new QLabel("0/0", statusBox);
    statusLabel->setFrameStyle(QFrame::Sunken + QFrame::Box);
    searchLabel = new QLabel(statusBox);
    searchLabel->setFrameStyle(QFrame::Sunken + QFrame::Box);
    searchLabel->setAlignment(Qt::WordBreak);
    statusBox->setStretchFactor(searchLabel,1);

    resultView = new SearchResultView(this, "searchResult");
    connect(resultView, SIGNAL(contextMenu(KListView*, QListViewItem*, const QPoint&)),
            this, SLOT(contextSearchResult(KListView*, QListViewItem*, const QPoint&)));
}

SearchResult::~SearchResult()
{
}

void SearchResult::contextSearchResult(KListView*,QListViewItem*,const QPoint& pt)
{
    QPopupMenu *pop = (QPopupMenu*)(KMLDonkey::App->factory())->container("search_actions", KMLDonkey::App);
    pop->popup(pt);
}

inline int SearchResult::searchNo() { return id; }
inline const QString& SearchResult::getTabLabel() { return tabLabel; }
inline void SearchResult::setSearchLabel(const QString &text) { searchLabel->setText(text); }
inline void SearchResult::setListFont(const QFont& font) { resultView->QListView::setFont(font); }

void SearchResult::AddItem(const ResultInfo *searchinfo)
{
    int32 avail = 0;
    const QMap<QString,QVariant>& tags = searchinfo->resultTags();
    QMapConstIterator<QString,QVariant> it;
    for ( it = tags.begin(); it != tags.end(); ++it ) {
       if (it.key() == "availability") {
          avail = it.data().toInt();
          break;
       }
    }

    SearchResultItem* item = new SearchResultItem(resultView, searchinfo->resultNo(), avail, searchinfo);
    item->setText(0, searchinfo->resultName() );
    item->setText(1, humanReadableSize(searchinfo->resultSize()) );
    item->setText(2, QString::number(avail) );

    Network* nw = KMLDonkey::App->donkey->findNetworkNo( searchinfo->resultNetwork() );
    item->setText(3, (nw) ? nw->networkName() : QString::number(searchinfo->resultNetwork()) );

    item->setText(4, KMimeType::findByURL( "file://" + searchinfo->resultName(), 0, false, true)->comment() );
    item->setText(5, searchinfo->resultFormat() );
    item->setText(6, searchinfo->resultComment() );
    item->setText(7, FileInfo::md4ToString(searchinfo->resultMD4()) );

    bool visible = ! filterItem(item);
    if (visible) visibleCount++;
    item->setVisible(visible);

    setStatusLabel();
}

void SearchResult::DownloadSelectedItems(bool force, bool askForFilename)
{
    QStringList names;
    QString filename;
    bool ok;
    QPtrList<QListViewItem> list = resultView->selectedItems();
    SearchResultItem *item;
    for (item = (SearchResultItem*)list.first(); item; item = (SearchResultItem*)list.next()) {

        filename = item->text(0);
        if (askForFilename || ! filename.length()) {
            filename = KLineEditDlg::getText( i18n("Download as..."),
                                              i18n("Choose a filename for the new download"),
                                              filename, &ok, this );
            if (! ok) continue;
        }

        names.clear();
        names.append(filename);
        if (filename != item->text(0)) names.append(item->text(0));
        KMLDonkey::App->donkey->startDownload(names, item->getNum(), force);
    }
}

void SearchResult::filterChanged()
{
    filterTimer->stop();
    if (resultView->childCount()) filterTimer->start(500, true);
}

void SearchResult::filterTimerDone()
{
    visibleCount = 0;
    filters = QStringList::split(" ", filterEdit->text().lower().stripWhiteSpace().simplifyWhiteSpace() );
    QListViewItemIterator it(resultView);
    bool visible;
    for ( ; it.current(); ++it ) {
        visible = ! filterItem( (SearchResultItem*)it.current() );
        if(visible) visibleCount++;
        it.current()->setVisible(visible);
        if (it.current()->isSelected() && ! visible) it.current()->setSelected(false);
    }
    setStatusLabel();
}

inline void SearchResult::setStatusLabel()
{
    statusLabel->setText( QString::number(visibleCount) + "/" + QString::number(resultView->childCount()) );
}

bool SearchResult::filterItem(SearchResultItem *item)
{
    unsigned int i;
    bool minsize;
    int64 size;
    for (i = 0; i < filters.count(); i++) {
        minsize = filters[i].startsWith(">");
        if (minsize || filters[i].startsWith("<")) { // filter max/min filesize
            size = filesizeStr2Int( filters[i].right(filters[i].length() - 1) );
            if (size > 0) {
                if (minsize) {
                    if (item->getSize() <= size) return true;
                }
                else {
                    if (item->getSize() >= size) return true;
                }
            }
        }
        else { // filter filename
            if(! item->text(0).contains(filters[i], false)) return true;
        }
    }
    return false;
}

/*** SearchPage ***/

SearchPage::SearchPage(QWidget *parent)
    : QVBox(parent, "searchPage")
    , KMLDonkeyPage()
    , ClipboardHelper()
{
    setMargin(6);
    searchNum = 0;
    maxHits = 500;
    searchType = 0;

    splitter = new QSplitter(this);
    QScrollView *scroll = new QScrollView(splitter);
    scroll->setResizePolicy(QScrollView::AutoOneFit);
    scroll->viewport()->setBackgroundMode(PaletteBackground);
    splitter->setResizeMode(scroll, QSplitter::KeepSize);
    QVBox *box = new QVBox(scroll->viewport());
    scroll->addChild(box);
    box->setMargin(6);

#ifdef HAVE_KTABWIDGET
    resultTabs = new KTabWidget(splitter);
    resultTabs->setHoverCloseButton(true);
    resultTabs->setTabReorderingEnabled(true);
#else
    resultTabs = new ClosableTabWidget(splitter);
#endif
    connect(resultTabs, SIGNAL(closeRequest(QWidget*)), this, SLOT(closeSearch(QWidget*)));

    // Keywords

    QHBox *kwbox = new QHBox(box);
    new QLabel(i18n("Keywords"), kwbox);
    keywordsEdit = new KLineEdit(kwbox);
    connect(keywordsEdit, SIGNAL( returnPressed() ), this, SLOT( startSearch() ));

    // Simple Options

    QGroupBox *simplebox = new QGroupBox(box);
    simplebox->setTitle(i18n("Simple Options"));
    simplebox->setColumnLayout(0, Qt::Vertical);
    simplebox->layout()->setSpacing(2);
    QGridLayout *simplelayout = new QGridLayout(simplebox->layout());

    QHBox *minsizebox = new QHBox(simplebox);
    simplelayout->addWidget(minsizebox , 0, 0);
    new QLabel(i18n("Min size"), minsizebox);
    minsizeCombo = new KComboBox(minsizebox);
    minsizeCombo->setEditable(true);
    minsizeCombo->insertItem("", 0);
    minsizeCombo->insertItem("500 MB", 1);
    minsizeCombo->insertItem("100 MB", 2);
    minsizeCombo->insertItem("50 MB", 3);
    minsizeCombo->insertItem("3 MB", 4);
    minsizeCombo->insertItem("500 KB", 5);
    minsizeCombo->insertItem("500 B", 6);

    QHBox *maxsizebox = new QHBox(simplebox);
    simplelayout->addWidget(maxsizebox , 1, 0);
    new QLabel(i18n("Max size"), maxsizebox);
    maxsizeCombo = new KComboBox(maxsizebox);
    maxsizeCombo->setEditable(true);
    maxsizeCombo->insertItem("", 0);
    maxsizeCombo->insertItem("500 MB", 1);
    maxsizeCombo->insertItem("100 MB", 2);
    maxsizeCombo->insertItem("50 MB", 3);
    maxsizeCombo->insertItem("3 MB", 4);
    maxsizeCombo->insertItem("500 KB", 5);
    maxsizeCombo->insertItem("500 B", 6);

    QHBox *mediabox = new QHBox(simplebox);
    simplelayout->addWidget(mediabox , 2, 0);
    new QLabel(i18n("Media"), mediabox);
    mediaCombo = new KComboBox(mediabox);
    mediaCombo->setEditable(true);
    mediaCombo->insertItem("", 0);
    mediaCombo->insertItem("Audio", 1);
    mediaCombo->insertItem("Video", 2);
    mediaCombo->insertItem("Program", 3);
    mediaCombo->insertItem("Image", 4);
    mediaCombo->insertItem("Documentation", 5);
    mediaCombo->insertItem("Collection", 6);

    QHBox *formatbox = new QHBox(simplebox);
    simplelayout->addWidget(formatbox , 3, 0);
    new QLabel(i18n("Format"), formatbox);
    formatCombo = new KComboBox(formatbox);
    formatCombo->setEditable(true);
    formatCombo->insertItem("", 0);
    formatCombo->insertItem("avi", 1);
    formatCombo->insertItem("mp3", 2);

    // MP3 Options

    QGroupBox *mp3box = new QGroupBox(box);
    mp3box->setTitle(i18n("MP3 Options"));
    mp3box->setColumnLayout(0, Qt::Vertical);
    mp3box->layout()->setSpacing(2);
    QGridLayout *mp3layout = new QGridLayout(mp3box->layout());

    QHBox *artistbox = new QHBox(mp3box);
    mp3layout->addWidget(artistbox, 0, 0);
    new QLabel(i18n("Artist"), artistbox);
    mp3ArtistEdit = new KLineEdit(artistbox);
    connect(mp3ArtistEdit, SIGNAL( returnPressed() ), this, SLOT( startSearch() ));

    QHBox *titlebox = new QHBox(mp3box);
    mp3layout->addWidget(titlebox, 1, 0);
    new QLabel(i18n("Title"), titlebox);
    mp3TitleEdit = new KLineEdit(titlebox);
    connect(mp3TitleEdit, SIGNAL( returnPressed() ), this, SLOT( startSearch() ));

    QHBox *albumbox = new QHBox(mp3box);
    mp3layout->addWidget(albumbox, 2, 0);
    new QLabel(i18n("Album"), albumbox);
    mp3AlbumEdit = new KLineEdit(albumbox);
    connect(mp3AlbumEdit, SIGNAL( returnPressed() ), this, SLOT( startSearch() ));

    QHBox *bitratebox = new QHBox(mp3box);
    mp3layout->addWidget(bitratebox , 3, 0);
    new QLabel(i18n("Bitrate"), bitratebox);
    mp3BitrateCombo = new KComboBox(bitratebox);
    mp3BitrateCombo->setEditable(true);
    mp3BitrateCombo->insertItem("", 0);
    mp3BitrateCombo->insertItem("64", 1);
    mp3BitrateCombo->insertItem("96", 2);
    mp3BitrateCombo->insertItem("128", 3);
    mp3BitrateCombo->insertItem("160", 4);
    mp3BitrateCombo->insertItem("192", 5);

    // Max Hits

    QHBox *mhbox = new QHBox(box);
    new QLabel(i18n("Max hits"), mhbox);
    maxhitsEdit = new KLineEdit(mhbox);
    maxhitsEdit->setMaxLength(4);
    maxhitsEdit->setText( QString::number(maxHits) );

    // Type

    QHBox *stbox = new QHBox(box);
    new QLabel(i18n("Type"), stbox);
    searchtypeCombo = new KComboBox(stbox);
    searchtypeCombo->insertItem(i18n("Remote"), 0);
    searchtypeCombo->insertItem(i18n("Local"), 1);
    searchtypeCombo->insertItem(i18n("Subscribe"), 2);
    searchtypeCombo->setCurrentItem(searchType);

    // Network

    QHBox *snbox = new QHBox(box);
    new QLabel(i18n("Network"), snbox);
    searchnetCombo = new KComboBox(snbox);
    searchnetCombo->insertItem(i18n("All Networks"));
    searchnetCombo->setCurrentItem(0);
    connect(KMLDonkey::App->donkey, SIGNAL( networkUpdated(int) ), this, SLOT( setNetworks(int) ));

    KPushButton *startButton = new KPushButton(i18n("Search"), box);
    connect(startButton, SIGNAL( clicked() ), this, SLOT( startSearch() ));

    scroll->setMinimumSize( box->sizeHint() );

    connect(KMLDonkey::App->donkey, SIGNAL(searchUpdated(int,const ResultInfo*)), this, SLOT(searchUpdated(int,const ResultInfo*)));
}

void SearchPage::setupActions(KActionCollection* actionCollection)
{
    (void)new KAction(i18n("&Download"), "down", 0, this, SLOT(actionDownload()),
                      actionCollection, "search_download");
    (void)new KAction(i18n("&Force download"), "down", 0, this, SLOT(actionForceDownload()),
                      actionCollection, "search_force_download");
    (void)new KAction(i18n("Download &as..."), "down", 0, this, SLOT(actionDownloadAs()),
                      actionCollection, "search_download_as");

    (void)new KAction(i18n("Copy search &URL to clipboard"), "editcopy", 0, this, SLOT(actionCopyURL()),
                      actionCollection, "search_copy_url");
    (void)new KAction(i18n("Copy search HTML to clipboard"), "editcopy", 0, this, SLOT(actionCopyHTML()),
                      actionCollection, "search_copy_html");
    (void)new KAction(i18n("Copy search hash to clipboard"), "editcopy", 0, this, SLOT(actionCopyHash()),
                      actionCollection, "search_copy_hash");

    (void)new KAction(i18n("Activate search page"), 0, 0, this, SLOT(actionActivatePage()),
		      actionCollection, "activate_page_search");
}

void SearchPage::configurePrefsDialog(KMLDonkeyPreferences* prefs)
{
    prefs->searchPage->activateNewTabsCheckbox->setChecked(activateNewTabs);
    prefs->searchPage->closeTabsOnDisconnectCheckbox->setChecked(closeTabsOnDisconnect);
    prefs->searchPage->showNumbersOnTabsCheckbox->setChecked(showNumbersOnTabs);
}

void SearchPage::applyPreferences(KMLDonkeyPreferences* prefs)
{
    activateNewTabs = prefs->searchPage->activateNewTabsCheckbox->isChecked();
    closeTabsOnDisconnect = prefs->searchPage->closeTabsOnDisconnectCheckbox->isChecked();
    showNumbersOnTabs = prefs->searchPage->showNumbersOnTabsCheckbox->isChecked();

    QIntDictIterator<SearchResult> it( Results );
    for ( ; it.current(); ++it ) {
        it.current()->setListFont(KMLDonkey::App->listFont);
        resultTabs->setTabLabel(it.current(), getTabLabel(it.current()));
    }
}

inline QString SearchPage::getTabLabel(SearchResult *tab)
{
    if (! showNumbersOnTabs) return tab->getTabLabel();
    return tab->getTabLabel() + " (" + QString::number(tab->resultView->childCount()) + ")";
}

void SearchPage::saveState(KConfig* conf)
{
    conf->setGroup("Search");

    conf->writeEntry("Splitter", splitter->sizes());

    bool ok;
    int hits = maxhitsEdit->text().toInt(&ok);
    if (ok && hits) conf->writeEntry("maxHits", hits);

    conf->writeEntry("Type", searchtypeCombo->currentItem());
    conf->writeEntry("Network", searchnetCombo->currentText());

    conf->writeEntry("activateNewTabs", activateNewTabs);
    conf->writeEntry("closeTabsOnDisconnect", closeTabsOnDisconnect);
    conf->writeEntry("showNumbersOnTabs", showNumbersOnTabs);
}

void SearchPage::restoreState(KConfig* conf)
{
    conf->setGroup("Search");

    splitter->setSizes(conf->readIntListEntry("Splitter"));
    int i = conf->readNumEntry("maxHits", maxHits);
    if (i && i != maxHits) {
        maxHits = i;
        maxhitsEdit->setText( QString::number(maxHits) );
    }
    i = conf->readNumEntry("Type", searchType);
    if (i && i <= 2) {
        searchType = i;
        searchtypeCombo->setCurrentItem(searchType);
    }
    searchNetwork = conf->readEntry("Network", "");

    activateNewTabs = conf->readBoolEntry("activateNewTabs", true);
    closeTabsOnDisconnect = conf->readBoolEntry("closeTabsOnDisconnect", true);
    showNumbersOnTabs = conf->readBoolEntry("showNumbersOnTabs", true);
}

void SearchPage::clear()
{
    searchnetCombo->setCurrentItem(0);
    for (int i = searchnetCombo->count() - 1; i > 0; i--) searchnetCombo->removeItem(i);
    if (closeTabsOnDisconnect) closeAllSearches();
}

void SearchPage::setNetworks(int no)
{
    if (KMLDonkey::App->donkey->protocolVersion() < 16) return;
    Network *net = KMLDonkey::App->donkey->findNetworkNo(no);
    if (! net) return;
    for (int i = searchnetCombo->count() - 1; i > 0; i--)
        if (searchnetCombo->text(i) == net->networkName()) { // already on the list?
            if (! net->networkEnabled())
                searchnetCombo->removeItem(i);
            return;
        }
    if (! net->networkEnabled()) return;
    searchnetCombo->insertItem( net->networkName() );
    if (searchNetwork == net->networkName())
        searchnetCombo->setCurrentItem(searchnetCombo->count() - 1);
}

void SearchPage::startSearch()
{
    if (! KMLDonkey::App->donkey->isConnected()) return;

    QString keywords = keywordsEdit->text().stripWhiteSpace();

    bool ok;
    int hits = maxhitsEdit->text().toInt(&ok);
    if (ok && hits) {
        maxHits = hits;
    }
    else {
        hits = maxHits;
        maxhitsEdit->setText( QString::number(hits) );
    }

    DonkeyProtocol::SearchType type;
    switch (searchtypeCombo->currentItem()) {
        case 1:  type = DonkeyProtocol::LocalSearch; break;
        case 2:  type = DonkeyProtocol::SubscribeSearch; break;
        default: type = DonkeyProtocol::RemoteSearch; break;
    }

    long int minsize = filesizeStr2Int( minsizeCombo->currentText() );
    long int maxsize = filesizeStr2Int( maxsizeCombo->currentText() );
    QString format = formatCombo->currentText().stripWhiteSpace();
    QString media = mediaCombo->currentText().stripWhiteSpace();
    QString mp3artist = mp3ArtistEdit->text().stripWhiteSpace();
    QString mp3title = mp3TitleEdit->text().stripWhiteSpace();
    QString mp3album = mp3AlbumEdit->text().stripWhiteSpace();
    QString mp3bitrate = mp3BitrateCombo->currentText().stripWhiteSpace();

    QString caption, label;
    if (! keywords.isEmpty()) {
        caption = keywords + " ";
        label   = i18n("Keywords") + "=\"" + keywords + "\" ";
    }
    if (! mp3artist.isEmpty()) {
        caption += mp3artist + " ";
        label   += i18n("Artist") + "=\"" + mp3artist + "\" ";
    }
    if (! mp3title.isEmpty()) {
        caption += mp3title + " ";
        label   += i18n("Title") + "=\"" + mp3title + "\" ";
    }
    if (! mp3album.isEmpty()) {
        caption += mp3album + " ";
        label   += i18n("Album") + "=\"" + mp3album + "\" ";
    }
    if(! mp3bitrate.isEmpty()) label += i18n("Bitrate") + "=" + mp3bitrate + " ";
    if(! format.isEmpty()) label += i18n("Format") + "=" + format + " ";
    if(! media.isEmpty()) label += i18n("Media") + "=" + media + " ";
    if(minsize > 0) label += i18n("Min size") + "=" + QString::number(minsize) + " ";
    if(maxsize > 0) label += i18n("Max size") + "=" + QString::number(maxsize) + " ";

    if (caption.isEmpty()) return;

    // Construct the query
    QueryAnd* qa = new QueryAnd();
    qa->append(new QueryKeywords((QString&)QString::null, keywords));
    if (minsize) qa->append(new QueryMinSize(QString::null, QString::number(minsize)));
    if (maxsize) qa->append(new QueryMaxSize(QString::null, QString::number(maxsize)));
    if (!format.isEmpty()) qa->append(new QueryFormat(QString::null, format));
    if (!media.isEmpty()) qa->append(new QueryMedia(QString::null, media));
    if (!mp3artist.isEmpty()) qa->append(new QueryMp3Artist(QString::null, mp3artist));
    if (!mp3title.isEmpty()) qa->append(new QueryMp3Title(QString::null, mp3title));
    if (!mp3album.isEmpty()) qa->append(new QueryMp3Album(QString::null, mp3album));
    if (!mp3bitrate.isEmpty()) qa->append(new QueryMp3Bitrate(QString::null, mp3bitrate));

    // Simplify the query if possible
    SearchQuery* q;
    if (qa->count() == 1)
        q = qa->take(0);
    else
        q = qa;

    // Network
    int network = 0;
    if (searchnetCombo->currentItem() > 0) {
        Network *net;
        QIntDictIterator<Network> it( KMLDonkey::App->donkey->availableNetworks() );
        for ( ; it.current(); ++it ) {
            net = it.current();
            if (net->networkName() == searchnetCombo->currentText()) {
                network = net->networkNo();
                label += i18n("Network") + "=" + net->networkName();
                break;
            }
        }
    }

    // create a SearchResult
    searchNum++;
    SearchResult *sr = new SearchResult(searchNum, caption.stripWhiteSpace(), resultTabs);
    Results.replace(searchNum, sr);
    sr->setListFont(KMLDonkey::App->listFont);
    sr->setSearchLabel(label.stripWhiteSpace());
    resultTabs->addTab(sr, KGlobal::iconLoader()->loadIconSet("find", KIcon::Small), getTabLabel(sr));
    if (activateNewTabs) resultTabs->setCurrentPage(resultTabs->count()-1);

    // now let's start the search
    KMLDonkey::App->donkey->startSearch(searchNum, q, maxHits, type, network);

    delete qa;
}

void SearchPage::searchUpdated(int searchnum, const ResultInfo* searchinfo)
{
    SearchResult *sr = Results[searchnum];
    if (! sr) return;
    sr->AddItem(searchinfo);
    resultTabs->setTabLabel(sr, getTabLabel(sr));
}

void SearchPage::closeSearch(QWidget* widget)
{
    int id = dynamic_cast<SearchResult*>(widget)->searchNo();
    KMLDonkey::App->donkey->stopSearch(id);
    Results.remove(id);
    delete widget;
}

void SearchPage::closeAllSearches()
{
    while (resultTabs->count())
        closeSearch(resultTabs->currentPage());
}

void SearchPage::actionDownload()
{
    SearchResult *sr = (SearchResult*)resultTabs->currentPage();
    if (sr) sr->DownloadSelectedItems(false, false);
}

void SearchPage::actionForceDownload()
{
    SearchResult *sr = (SearchResult*)resultTabs->currentPage();
    if (sr) sr->DownloadSelectedItems(true, false);
}

void SearchPage::actionDownloadAs()
{
    SearchResult *sr = (SearchResult*)resultTabs->currentPage();
    if (sr) sr->DownloadSelectedItems(false, true);
}

void SearchPage::copySearchToClipboard(ClipFormat format)
{
    SearchResult *sr = (SearchResult*)resultTabs->currentPage();
    if (! sr) return;
    QStringList sl;
    QPtrList<QListViewItem> list = sr->resultView->selectedItems();
    SearchResultItem *item;
    for (item = (SearchResultItem*)list.first(); item; item = (SearchResultItem*)list.next()) {
        sl.append( item->text(0) ); // filename
        sl.append( item->text(7) ); // filehash
        sl.append( QString::number((long)item->getSize()) ); // filesize
    }
    copyToClipboard(sl, format);
}

void SearchPage::actionCopyURL() { copySearchToClipboard(URL); }
void SearchPage::actionCopyHTML() { copySearchToClipboard(HTML); }
void SearchPage::actionCopyHash() { copySearchToClipboard(Hash); }

void SearchPage::actionActivatePage()
{
    KMLDonkey::App->activatePage(this);
}


#include "search.moc"
