/* -*- c++ -*-
 *
 * $Id: sharepage.cpp,v 1.5 2003/07/30 15:03:33 dipesh Exp $
 *
 * Copyright (C) 2003 Petter E. Stokke <gibreel@gibreel.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Log: sharepage.cpp,v $
 * Revision 1.5  2003/07/30 15:03:33  dipesh
 * Added "Share directory" and "Unshare directory"
 *
 * Revision 1.4  2003/07/27 20:24:26  gibreel
 * Added KActions for switching between pages.
 *
 * Revision 1.3  2003/06/30 23:30:35  gibreel
 * Preliminary friend list support. A ton of updates to the libkmldonkey API to
 * accommodate this, most notably improvements to the search handling
 * necessitated by mldonkey's somewhat awkward reporting of friend shares.
 *
 * Revision 1.2  2003/06/30 14:59:22  gibreel
 * Updated lists to support libkmldonkey's new removed signals. Moved pages'
 * connect statements out of KMLDonkey's constructor into their own
 * constructors. Added a debug console displaying dumps of unhandled messages.
 *
 * Revision 1.1  2003/06/19 21:50:45  gibreel
 * Split the various pages off into individual objects, and made a ton of code
 * cleanups, API changes, and not a few bugfixes in the process. The
 * disconnect/reconnect bug, especially, now seems to be gone.
 *
 */

#include <klocale.h>
#include <kconfig.h>
#include <kaction.h>
#include <klineeditdlg.h>

#include <qintdict.h>
#include <qpopupmenu.h>
#include <qclipboard.h>

#include "infolist.h"
#include "prefs.h"
#include "kmldonkey.h"
#include "sharepage.h"
#include "sharepage.moc"

SharePage::SharePage(QWidget* parent)
    : QVBox(parent, "sharePage")
    , KMLDonkeyPage()
    , ClipboardHelper()
{
    shareList.setAutoDelete(true);

    uploadSplitter = new QSplitter(this, "uploadSplitter");
    uploadSplitter->setOrientation(QSplitter::Vertical);

    uploadView = new InfoList(uploadSplitter, "uploadView");
    uploadView->addColumn( i18n( "File name" ) );
    uploadView->addColumn( i18n( "Network" ) );
    uploadView->addColumn( i18n( "Requests" ) );
    uploadView->addColumn( i18n( "Bytes" ) );

    uploadersView = new InfoList(uploadSplitter, "uploadersView");
    uploadersView->addColumn( i18n( "ID" ) );
    uploadersView->addColumn( i18n( "Network" ) );
    uploadersView->addColumn( i18n( "Client" ) );
    uploadersView->addColumn( i18n( "File" ) );
    uploadersView->addColumn( i18n( "Sent" ) );
    uploadersView->setDisabled(true);

    connect(uploadView, SIGNAL(contextMenu(KListView*, QListViewItem*, const QPoint&)),
	    this, SLOT(contextUpload(KListView*, QListViewItem*, const QPoint&)));

    connect(KMLDonkey::App->donkey, SIGNAL(shareUpdated(int)), this, SLOT(shareUpdated(int)));
    connect(KMLDonkey::App->donkey, SIGNAL(shareRemoved(int)), this, SLOT(shareRemoved(int)));
}

void SharePage::setupActions(KActionCollection* actionCollection)
{
    (void)new KAction(i18n("&Refresh shared file list"), "reload", 0, this, SLOT(actionRefreshShared()),
                      actionCollection, "upload_refresh");
    (void)new KAction(i18n("&Share directory"), "fileimport", 0, this, SLOT(actionShareDirectory()),
                      actionCollection, "share_directory");
    (void)new KAction(i18n("&Unshare directory"), "fileexport", 0, this, SLOT(actionUnshareDirectory()),
                      actionCollection, "unshare_directory");

    (void)new KAction(i18n("Activate uploads page"), 0, 0, this, SLOT(actionActivatePage()),
		      actionCollection, "activate_page_uploads");
}

void SharePage::clear()
{
    shareList.clear();
    uploadView->clear();
}

void SharePage::applyPreferences(KMLDonkeyPreferences*)
{
    if (KMLDonkey::App->listFont !=  uploadView->font())
        uploadView->setFont(KMLDonkey::App->listFont);
    if (KMLDonkey::App->listFont !=  uploadersView->font())
        uploadersView->setFont(KMLDonkey::App->listFont);
}

void SharePage::restoreState(KConfig* conf)
{
    conf->setGroup("Splitters");
    uploadSplitter->setSizes(conf->readIntListEntry("UploadSplitter"));

    uploadView->restoreLayout(conf, "UploadView");
    uploadersView->restoreLayout(conf, "UploadersView");

    applyPreferences();
}

void SharePage::saveState(KConfig* conf)
{
    conf->setGroup("Splitters");
    conf->writeEntry("UploadSplitter", uploadSplitter->sizes());

    uploadView->saveLayout(conf, "UploadView");
    uploadersView->saveLayout(conf, "UploadersView");
}

void SharePage::contextUpload(KListView*,QListViewItem*,const QPoint& pt)
{
    QPopupMenu *pop = (QPopupMenu*)(KMLDonkey::App->factory())->container("upload_actions", KMLDonkey::App);
    pop->popup(pt);
}

void SharePage::actionRefreshShared()
{
    KMLDonkey::App->donkey->refreshShared();
}

void SharePage::actionShareDirectory()
{
    bool ok;
    QString dir = KLineEditDlg::getText( i18n("Share Directory"),
                                         i18n("Enter the new directory to share (fe. '/home/user/')."),
                                         QString::null, &ok, this );
    if (ok) {
        if (dir.right(1) != "/") dir += "/";
        KMLDonkey::App->donkey->sendConsoleMessage("share " + dir);
    }
}

void SharePage::actionUnshareDirectory()
{
    bool ok;
    QString dir = KLineEditDlg::getText( i18n("Unshare Directory"),
                                         i18n("Enter the directory to unshare (fe. '/home/user/')."),
                                         QString::null, &ok, this );
    if (ok) {
        if (dir.right(1) != "/") dir += "/";
        KMLDonkey::App->donkey->sendConsoleMessage("unshare " + dir);
    }
}

void SharePage::shareUpdated(int share)
{
    ShareInfo* si = KMLDonkey::App->donkey->findShareNo(share);
    if (!si) return;

    SharedFile* df = shareList[share];
    if (df) {
        df->repaint();
        return;
    }
    df = new SharedFile(uploadView, share);
    shareList.insert(share, df);
}

void SharePage::shareRemoved(int share)
{
    shareList.remove(share);
}

void SharePage::actionActivatePage()
{
    KMLDonkey::App->activatePage(this);
}

