/*
 * $Id: donkeyprotocol.cpp,v 1.36 2003/07/28 17:10:25 dipesh Exp $
 *
 * Copyright (C) 2003 Petter E. Stokke <gibreel@gibreel.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Log: donkeyprotocol.cpp,v $
 * Revision 1.36  2003/07/28 17:10:25  dipesh
 * Fixed that searching for something that was previous (in the same
 * session) searched didn't returned anything cause of ML's
 * caching beaviour.
 *
 * Revision 1.35  2003/07/25 15:42:53  dipesh
 * Added the enableNetwork() donkey-function and changed "Configure MLDonkey" to
 * handle enable/disable Networks another way. The way used previous was working
 * as well, but the core didn't send something back to show that that a Network
 * was enabled/disabled. So, to be able to work at the search/network combo always
 * with the actual enabled networks the new function is used now. Stupid
 * core-behaviour anyway.
 *
 * Revision 1.34  2003/07/22 21:01:55  gibreel
 * Added the HostManager class for dealing with mldonkeyrc (MLDonkey connection
 * definitions as written by kcmdonkey).
 *
 * Revision 1.33  2003/07/20 20:45:04  dipesh
 * Added "Configure MLDonkey" :)
 *
 * Revision 1.32  2003/07/18 17:08:54  gibreel
 * Fixed a bug where DonkeyProtocol's tables weren't cleared on a manual
 * disconnect, and the friend table wasn't ever cleared. Replaced remaining
 * qDebug() calls with kdDebug().
 *
 * Revision 1.31  2003/07/17 19:55:35  gibreel
 * Removed some obsolete debug output pertaining to client lists.
 *
 * Revision 1.30  2003/06/30 23:30:35  gibreel
 * Preliminary friend list support. A ton of updates to the libkmldonkey API to
 * accommodate this, most notably improvements to the search handling
 * necessitated by mldonkey's somewhat awkward reporting of friend shares.
 *
 * Revision 1.29  2003/06/30 14:53:42  gibreel
 * Still hunting bugs in source management: implemented a lot of missing
 * message handlers pertaining to cleanup of client records. Quite a few bug
 * fixes. Protocol object now emits *Removed() signals corresponding to
 * *Updated() signals instead of sending update signals for removed records and
 * leaving the user to determine whether to remove the records.
 *
 * Revision 1.28  2003/06/29 13:39:27  gibreel
 * Tried to optimise the particularly memory hungry file source lists.
 *
 * Revision 1.27  2003/06/28 21:17:00  gibreel
 * The file info dialog now displays file sources. Libkmldonkey received a
 * number of updates and bugfixes to facilitate this.
 *
 * Revision 1.26  2003/06/27 12:43:08  dipesh
 * Added isConnected();
 *
 * Revision 1.25  2003/06/19 21:47:45  gibreel
 * DonkeyProtocol now keeps track of the amount of connected servers, and
 * provides methods for querying this and the total server count.
 *
 * Revision 1.24  2003/06/19 17:04:32  gibreel
 * Minor code cleanups and attempt to fix the crash bugs on
 * disconnect/reconnect. Both kmldonkey and libkmldonkey now clean out their
 * state data when the connection with the core is lost.
 *
 * Revision 1.23  2003/06/13 18:20:01  gibreel
 * Libkmldonkey now uses references instead of pointers everywhere except where
 * it would cause an obvious performance impact, which should lead to less
 * chance of memory leaks and cleaner code in general. Almost everything that
 * should be const is now also const.
 *
 * Revision 1.22  2003/06/09 21:08:50  gibreel
 * Added the setOption method for configuring mldonkey.
 *
 * Revision 1.21  2003/05/29 23:47:55  gibreel
 * Added shared file list refresh action. For some reason, though, the core
 * seems to only want to respond to the refresh request once...
 *
 * Revision 1.20  2003/05/24 22:52:04  gibreel
 * Created some classes for constructing search queries in a more flexible
 * manner, and updated the code accordingly where necessary.
 *
 * Revision 1.19  2003/05/21 00:07:42  gibreel
 * Updated libkmldonkey to support protocol 16. Added the addServer method.
 *
 * Revision 1.18  2003/05/20 20:56:20  gibreel
 * Resolved some compiler warnings.
 *
 * Revision 1.17  2003/05/19 10:59:30  dipesh
 * fixed Keywords-only search
 *
 * Revision 1.16  2003/05/18 17:58:22  dipesh
 * Added KAction's Download, ForceDownload and RemoveServers
 *
 * Revision 1.15  2003/05/17 10:52:24  dipesh
 * Added more search-options
 *
 * Revision 1.14  2003/05/12 00:20:36  gibreel
 * Added a method for forgetting searches.
 *
 * Revision 1.13  2003/05/11 08:33:01  dipesh
 * Added initial Search-functionality.
 *
 * Revision 1.12  2003/04/26 11:07:16  gibreel
 * Fixed a crash bug when the core sends an unseen server number.
 *
 * Revision 1.11  2003/04/15 20:34:35  gibreel
 * Objects are removed from memory when they expire. This should clear up a
 * pretty serious memory semi-leak.
 *
 * Revision 1.10  2003/04/06 21:29:14  gibreel
 * Added some methods for controlling server lists.
 *
 * Revision 1.9  2003/03/24 15:36:18  gibreel
 * Removed some debug cruft.
 *
 * Revision 1.8  2003/03/24 13:05:25  gibreel
 * Improved message handling efficiency: when several messages are received on
 * the socket at once, we process them in a loop rather than the old recursive
 * mechanism. Sometimes, when connecting as a full GUI, the core would send so
 * many messages at once that DonkeySocket would overload the stack. This
 * should no longer happen. Note that the sense of the readyMessage() signal
 * has changed - it is no longer emitted for every message that arrives, only
 * when the DonkeySocket feels it is time to process some messages.
 *
 * Revision 1.7  2003/03/23 23:34:59  gibreel
 * A lot of API additions, especially lists keeping track of shared files and
 * clients.
 *
 * Revision 1.6  2003/03/10 14:52:02  gibreel
 * Support for GUI protocol 14. Specifically: Authentication with username, new
 * download file state "Queued", and support for the new message types in the
 * DonkeyProtocol class.
 *
 * Revision 1.5  2003/03/08 20:13:20  gibreel
 * Switched readString/writeString in DonkeyMessage from using pointers to
 * QStrings to using references, which leads to far less hassle, shorter code,
 * and less chance of memory leaks.
 *
 * Revision 1.4  2003/03/08 17:08:12  gibreel
 * Lots of minor API changes.
 *
 * Revision 1.3  2003/03/07 21:51:18  gibreel
 * Changed the MD4 storage in the FileInfo object from an array of int to a
 * QByteArray, added conversion functions between that and QString, and methods
 * to search downloaded files by hash. The applet now uses the file hash
 * instead of the fileno when storing and restoring priorities, as the fileno
 * changes between core restarts.
 *
 * Revision 1.2  2003/03/07 20:35:37  gibreel
 * Added submitURL() method and option to disable flushing of the output buffer
 * on disconnect. Removed a ton of debug messages.
 *
 * Revision 1.1.1.1  2003/03/07 11:50:20  gibreel
 * Initial import.
 *
 */

#include <kdebug.h>

#include "donkeyprotocol.h"
#include "donkeyprotocol.moc"



DonkeyProtocol::DonkeyProtocol( bool poll, QObject *parent, const char *name )
    : QObject( parent, name)
{
    uname = "admin";
    passwd = "";
    donkeyError = NoError;
    cstate = connectedservers = 0;
    proto = MIN_PROTOCOL_VERSION;
    wantpoll = poll;
    download.setAutoDelete(true);
    downloaded.setAutoDelete(true);
    servers.setAutoDelete(true);
    networks.setAutoDelete(true);
    clients.setAutoDelete(true);
    shares.setAutoDelete(true);
    searches.setAutoDelete(true);
    unmappedResults.setAutoDelete(true);
    connect(&sock, SIGNAL(readyMessage()), this, SLOT(processMessage()));
    connect(&sock, SIGNAL(connectionClosed()), this, SLOT(socketDisconnected()));
    connect(&sock, SIGNAL(error(int)), this, SLOT(socketError(int)));
    connect(&sock, SIGNAL(delayedCloseFinished()), this, SLOT(socketDisconnected()));
}

DonkeyProtocol::~DonkeyProtocol()
{
}

void DonkeyProtocol::setPassword(const QString& username, const QString& pwd)
{
    uname = username;
    passwd = pwd;
}

void DonkeyProtocol::setPassword(const QString& pwd)
{
    uname = "admin";
    passwd = pwd;
}

const QString& DonkeyProtocol::username()
{
    return uname;
}

const QString& DonkeyProtocol::password()
{
    return passwd;
}

bool DonkeyProtocol::isConnected()
{
    return (sock.state() == QSocket::Connected);
}

void DonkeyProtocol::connectDonkey()
{
    donkeyError = NoError;
    if (isConnected())
	emit donkeyDisconnected(NoError);
    flushState();
    sock.connectDonkey();
}

void DonkeyProtocol::connectDonkey(const QString& host, Q_UINT16 port)
{
    if (isConnected())
	emit donkeyDisconnected(NoError);
    flushState();
    sock.connectDonkey(host, port);
}

void DonkeyProtocol::connectDonkey(const DonkeyHost& host)
{
    if (isConnected())
	emit donkeyDisconnected(NoError);
    flushState();
    setPassword(host.username(), host.password());
    sock.connectDonkey(host.address(), host.guiPort());
}

void DonkeyProtocol::flushState()
{
    download.clear();
    downloaded.clear();
    servers.clear();
    networks.clear();
    clients.clear();
    shares.clear();
    searches.clear();
    unmappedResults.clear();
    options.clear();
    friends.clear();
}

void DonkeyProtocol::disconnectDonkey(int force)
{
    cstate = 0;
    if (force)
	sock.clearPendingData();
    sock.close();
    kdDebug() << "Socket closed." << endl;
    if (sock.state() == QSocket::Idle)
	emit donkeyDisconnected(donkeyError);
    flushState();
}

void DonkeyProtocol::socketDisconnected()
{
    emit donkeyDisconnected(donkeyError);
    flushState();
}

const QSocket::State DonkeyProtocol::donkeyStatus()
{
    switch (sock.state()) {
    case QSocket::Connected:
	if (!cstate) return QSocket::Connecting;
    default:
	return sock.state();
    }
}

void DonkeyProtocol::socketError(int err)
{
    switch (err) {
    case QSocket::ErrConnectionRefused:
	emit donkeyDisconnected(ConnectionRefusedError);
	break;
    case QSocket::ErrHostNotFound:
	emit donkeyDisconnected(HostNotFoundError);
	break;
    case QSocket::ErrSocketRead:
	donkeyError = SocketReadError;
	disconnectDonkey();
	break;
    }
}

void DonkeyProtocol::pruneClientRecord(int clientno)
{
    QIntDictIterator<FileInfo> it(download);
    for (; it.current(); ++it)
	it.current()->removeSource(clientno);
    if (friends.remove(clientno))
	emit friendRemoved(clientno);
}

const int DonkeyProtocol::protocolVersion()
{
    return proto;
}

void DonkeyProtocol::processMessage()
{
    DonkeyMessage* msg;
    while ((msg = sock.popMessage())) {
        DonkeyMessage* out;
        QString baz;
        emit messageReceived(msg);
        switch (msg->opcode()) {
        case CoreProtocol:
            proto = msg->readInt32();
            kdDebug() << "CoreProtocol message, version " << proto << endl;
            if (proto < MIN_PROTOCOL_VERSION) {
                kdDebug() << "Obsolete protocol version!" << endl;
                donkeyError = ObsoleteProtocolError;
                disconnectDonkey();
                break;
            }
            out = new DonkeyMessage(GuiProtocol);
            out->writeInt32(MAX_PROTOCOL_VERSION);
            sock.sendMessage(*out);
            if (proto > MAX_PROTOCOL_VERSION)
                proto = MAX_PROTOCOL_VERSION;
            kdDebug() << "Using protocol " << proto << endl;
            delete out;
            if (wantpoll) {
                out = new DonkeyMessage(GuiExtensions);
                out->writeInt16(1);
                out->writeInt32(1);
                out->writeInt8(1);
                sock.sendMessage(*out);
                delete out;
            }
            if (proto < 14) out = new DonkeyMessage(Password_v1);
            else out = new DonkeyMessage(Password);
            out->writeString(passwd);
            if (proto >= 14) out->writeString(uname);
            sock.sendMessage(*out);
            delete out;
            cstate = 1;
            emit donkeyConnected();
            break;
        case Console:
            baz = msg->readString();
            emit consoleMessage(baz);
            break;
        case BadPassword:
            kdDebug() << "Bad password!" << endl;
            donkeyError = AuthenticationError;
            disconnectDonkey();
            break;
        case Client_stats_v1:
        case Client_stats_v2:
        case Client_stats_v3:
            kdDebug() << "Obsolete client stats message received" << endl;
            break;
        case Client_stats:
        {
            int64 ul,dl,sh;
            int32 nsh,tul,tdl,uul,udl,ndl,ncp;
            ul = msg->readInt64();
            dl = msg->readInt64();
            sh = msg->readInt64();
            nsh = msg->readInt32();
            tul = msg->readInt32();
            tdl = msg->readInt32();
            uul = msg->readInt32();
            udl = msg->readInt32();
            ndl = msg->readInt32();
            ncp = msg->readInt32();
            emit clientStats(ul,dl,sh,nsh,tul,tdl,uul,udl,ndl,ncp);
        } break;
        case File_add_source:
        {
            int fn = msg->readInt32();
	    int cl = msg->readInt32();
	    FileInfo* file = findDownloadFileNo(fn);
	    if (file) {
		file->addSource(cl);
		emit fileUpdated(fn);
		emit fileSourceUpdated(fn,cl);
	    }
        } break;
	case File_remove_source:
	{
	    int fn = msg->readInt32();
	    int cl = msg->readInt32();
	    kdDebug() << "File_remove_source " << fn << " " << cl << endl;
	    FileInfo* file = findDownloadFileNo(fn);
	    if (file) {
		file->removeSource(cl);
		emit fileUpdated(fn);
		emit fileSourceRemoved(fn,cl);
	    }
	};
        case File_update_availability:
        {
            int fn = msg->readInt32();
            int cl = msg->readInt32();
            QString av = msg->readString();
            FileInfo* file = findDownloadFileNo(fn);
            if (file) {
                file->updateAvailability(cl, av);
                emit fileUpdated(fn);
		emit fileSourceUpdated(fn,cl);
            }
        } break;
        case Client_info:
        {
            ClientInfo* client = new ClientInfo(msg,  proto);
            clients.replace(client->clientNo(), client);
	    switch (client->clientType()) {
	    case ClientInfo::FriendClient:
		if (!friends.contains(client->clientNo())) {
		    friends.append(client->clientNo());
		    emit friendUpdated(client->clientNo());
		}
		break;
	    default:
		if (friends.remove(client->clientNo()))
		    emit friendRemoved(client->clientNo());
	    }
            switch (client->clientState()) {
            case ClientInfo::Removed:
		emit clientRemoved(client->clientNo());
		pruneClientRecord(client->clientNo());
                clients.remove(client->clientNo());
                break;
            default:
		emit clientUpdated(client->clientNo());
                break;
            }
        } break;
        case Client_state:
        {
	    int clno = msg->readInt32();
            ClientInfo* client = findClientNo(clno);
            if (!client) {
		refreshClientInfo(clno);
                break;
            }
            client->setClientState(msg, proto);
            switch (client->clientState()) {
            case ClientInfo::Removed:
		emit clientRemoved(clno);
		pruneClientRecord(clno);
                clients.remove(clno);
                break;
            default:
		emit clientUpdated(clno);
                break;
            }
        } break;
	case Client_friend:
	{
	    int clno = msg->readInt32();
	    ClientInfo* client = findClientNo(clno);
	    if (!client) {
		refreshClientInfo(clno);
		break;
	    }
	    client->setClientType(msg, proto);
	    switch (client->clientType()) {
	    case ClientInfo::FriendClient:
		if (!friends.contains(clno)) {
		    friends.append(clno);
		    emit friendUpdated(clno);
		}
		break;
	    default:
		if (friends.remove(clno))
		    emit friendRemoved(clno);
	    }
	    emit clientUpdated(clno);
	} break;
	/*
	case Client_file:
	{
	    kdDebug() << "Client_file message: " << msg->readInt32() << " \"" << msg->readString() << "\" " << msg->readInt32() << endl;
	} break;
	*/
        case DownloadFiles_v1:
        case DownloadFiles_v2:
        case DownloadFiles_v3:
            kdDebug() << "Obsolete download files message received" << endl;
            break;
        case DownloadFiles_v4:
        case DownloadFiles:
        {
            download.clear();
            int i, j = msg->readInt16();
            for (i=0; i<j; i++) {
                FileInfo* fi = new FileInfo(msg, proto);
		FileInfo* fo = findDownloadFileNo(fi->fileNo());
		if (fo)
		    fo->updateFileInfo(fi);
		else
		    downloaded.replace(fi->fileNo(), fi);
                emit fileUpdated(fi->fileNo());
            }
            emit updatedDownloadFiles();
        } break;
        case File_downloaded_v1:
        case File_downloaded:
        {
            int fn = msg->readInt32();
            FileInfo* fi = findDownloadFileNo(fn);
            if (fi) {
                fi->updateDownloadStatus(msg, proto);
                switch (fi->fileState()) {
                case FileInfo::Shared:
                case FileInfo::Cancelled:
                case FileInfo::Aborted:
		    emit fileRemoved(fn);
                    download.remove(fi->fileNo());
                    break;
                default:
		    emit fileUpdated(fn);
                    break;
                }
            }
        } break;
        case DownloadedFiles_v1:
            kdDebug() << "Obsolete downloaded files message received" << endl;
            break;
        case DownloadedFiles_v2:
        case DownloadedFiles:
        {
            downloaded.clear();
            int i, j = msg->readInt16();
            for (i=0; i<j; i++) {
                FileInfo* fi = new FileInfo(msg, proto);
		FileInfo* fo = findDownloadFileNo(fi->fileNo());
		if (fo)
		    fo->updateFileInfo(fi);
		else
		    downloaded.replace(fi->fileNo(), fi);
            }
            emit updatedDownloadedFiles();
        } break;
        case File_info_v1:
        case File_info_v2:
            kdDebug() << "Obsolete file info message received" << endl;
            break;
        case File_info_v3:
        case File_info:
        {
            FileInfo* fi = findDownloadFileNo(msg->readInt32());
	    msg->resetPosition();
	    if (!fi) {
		fi = new FileInfo(msg, proto);
		download.replace(fi->fileNo(), fi);
	    } else {
		fi->updateFileInfo(msg, proto);
	    }
            switch (fi->fileState()) {
            case FileInfo::Shared:
            case FileInfo::Cancelled:
            case FileInfo::Aborted:
		emit fileRemoved(fi->fileNo());
                download.remove(fi->fileNo());
                break;
            default:
		emit fileUpdated(fi->fileNo());
                break;
            }
        } break;
        case ConnectedServers:
        {
            servers.clear();
	    connectedservers = 0;
            int i, j = msg->readInt16();
            for (i=0; i<j; i++) {
                ServerInfo* si = new ServerInfo(msg, proto);
                servers.replace(si->serverNo(), si);
		if (si->serverState() == ServerInfo::Connected || si->serverState() == ServerInfo::Connected2) connectedservers++;
            }
            emit updatedConnectedServers();
        } break;
        case Server_info_v1:
        case Server_info:
        {
            ServerInfo* si = new ServerInfo(msg, proto);
	    ServerInfo* oldsi = findServerNo(si->serverNo());
	    ServerInfo::State newstate = si->serverState(), oldstate = oldsi ? oldsi->serverState() : ServerInfo::NotConnected;
	    if ((oldstate == ServerInfo::Connected || oldstate == ServerInfo::Connected2)
		&& (newstate != ServerInfo::Connected && newstate != ServerInfo::Connected2))
		connectedservers--;
	    if ((oldstate != ServerInfo::Connected && oldstate != ServerInfo::Connected2)
		&& (newstate == ServerInfo::Connected || newstate == ServerInfo::Connected2))
		connectedservers++;
            servers.replace(si->serverNo(), si);
            switch (si->serverState()) {
            case ServerInfo::Removed:
		emit serverRemoved(si->serverNo());
                servers.remove(si->serverNo());
                break;
            default:
		emit serverUpdated(si->serverNo());
                break;
            }
        } break;
        case Server_state:
        {
            int sn = msg->readInt32();
            ServerInfo* si = findServerNo(sn);
            if (!si) {
                kdDebug() << "Core sent an invalid server number!" << endl;
                break;
            }
	    ServerInfo::State oldstate = si->serverState();
            si->updateServerState(msg, proto);
	    ServerInfo::State newstate = si->serverState();
	    if ((oldstate == ServerInfo::Connected || oldstate == ServerInfo::Connected2)
		&& (newstate != ServerInfo::Connected && newstate != ServerInfo::Connected2))
		connectedservers--;
	    if ((oldstate != ServerInfo::Connected && oldstate != ServerInfo::Connected2)
		&& (newstate == ServerInfo::Connected || newstate == ServerInfo::Connected2))
		connectedservers++;
            switch (si->serverState()) {
            case ServerInfo::Removed:
		emit serverRemoved(sn);
                servers.remove(si->serverNo());
                break;
            default:
		emit serverUpdated(sn);
                break;
            }
        } break;
        case Network_info:
        {
            Network* nw = new Network(msg, proto);
            networks.replace(nw->networkNo(), nw);
            emit networkUpdated(nw->networkNo());
        } break;
        case Shared_file_info_v1:
        case Shared_file_info:
        {
            ShareInfo* si = new ShareInfo(msg, proto);
            shares.replace(si->shareNo(), si);
            emit shareUpdated(si->shareNo());
        } break;
	case Shared_file_upload:
	{
	    ShareInfo* si = findShareNo(msg->readInt32());
	    if (si) {
		si->updateShare(msg, proto);
		emit shareUpdated(si->shareNo());
	    }
	} break;
	case Shared_file_unshared:
	{
	    int shno = msg->readInt32();
	    if (shares.remove(shno))
		emit shareRemoved(shno);
	} break;
        case Options_info:
        {
            int i, j = msg->readInt16();
            for (i=0; i<j; i++) {
                QString key, value;
                key = msg->readString();
                value = msg->readString();
                options.replace(key,value);
            }
        } break;

        // Following both messages are used to transfer the Searchresults back.
        // First Result_info's are recieved. Later the Search_result's defines
        // to what Search the previous noted Result_info's belongs too.
        // MLDonkey does cache searches. So, if you start again a search for
        // something that was previously searched for, it returns only
        // Search_result's.
        case Result_info:
        {
            ResultInfo *si = new ResultInfo(msg);
            unmappedResults.replace(si->resultNo(), si);
        } break;
        case Search_result:
        {
            int searchnum = msg->readInt32();
            int resultnum = msg->readInt32();
	    ResultInfo* ri = unmappedResults[resultnum];
	    if (!ri) return;
            SearchInfo *si = searches[searchnum];
	    if (!si) {
		si = new SearchInfo(searchnum);
		searches.replace(searchnum, si);
	    }
	    si->addResult(ri);
	    emit searchUpdated(searchnum, ri);
        } break;

	case Client_file:
	{
	    int clno = msg->readInt32();
	    QString dir = msg->readString();
	    int result = msg->readInt32();
	    emit clientFileListing(clno, dir, result);
	} break;

	case CleanTables:
	{
	    QIntDict<ClientInfo> newClients;
	    int i, j = msg->readInt16();
	    for (i=0; i<j; i++) {
		ClientInfo* cl = clients.take(msg->readInt32());
		if (cl) newClients.replace(cl->clientNo(), cl);
	    }
	    QIntDictIterator<ClientInfo> cit(clients);
	    for (; cit.current(); ++cit) {
		emit clientRemoved(cit.current()->clientNo());
		pruneClientRecord(cit.current()->clientNo());
	    }
	    clients = newClients;

	    QIntDict<ServerInfo> newServers;
	    j = msg->readInt16();
	    for (i=0; i<j; i++) {
		ServerInfo* si = servers.take(msg->readInt32());
		if (si) newServers.replace(si->serverNo(), si);
	    }
	    QIntDictIterator<ServerInfo> sit(servers);
	    for (; sit.current(); ++sit)
		emit serverRemoved(sit.current()->serverNo());
	    servers = newServers;

	    // Don't know if operator=() copies the autodelete state, best to be certain.
	    clients.setAutoDelete(true);
	    servers.setAutoDelete(true);
	} break;

	case MessageFromClient:
	{
	    int clno = msg->readInt32();
	    if (!findClientNo(clno))
		refreshClientInfo(clno);
	    emit messageFromClient(clno, msg->readString());
	} break;

        default:
            emit unhandledMessage(msg);
            break;
        }
        delete msg;
    }
}

void DonkeyProtocol::updateDownloadFiles()
{
    sock.sendMessage(DonkeyMessage(GetDownloadFiles));
}

void DonkeyProtocol::updateDownloadedFiles()
{
    sock.sendMessage(DonkeyMessage(GetDownloadedFiles));
}

void DonkeyProtocol::updateConnectedServers()
{
    sock.sendMessage(DonkeyMessage(GetConnectedServers));
}

const QIntDict<FileInfo>& DonkeyProtocol::downloadFiles()
{
    return download;
}

const QIntDict<FileInfo>& DonkeyProtocol::downloadedFiles()
{
    return downloaded;
}

const QIntDict<ServerInfo>& DonkeyProtocol::connectedServers()
{
    return servers;
}

const QIntDict<Network>& DonkeyProtocol::availableNetworks()
{
    return networks;
}

const QIntDict<ClientInfo>& DonkeyProtocol::clientList()
{
    return clients;
}

const QIntDict<ShareInfo>& DonkeyProtocol::sharedFiles()
{
    return shares;
}

const QValueList<int>& DonkeyProtocol::friendList()
{
    return friends;
}

FileInfo* DonkeyProtocol::findDownloadFileNo(int fileno)
{
    return download[fileno];
}

FileInfo* DonkeyProtocol::findDownloadFileByHash(const QByteArray& hash)
{
    for (QIntDictIterator<FileInfo> it(download); it.current(); ++it) {
	if (it.current()->fileMD4() == hash) return it;
    }
    return NULL;
}

FileInfo* DonkeyProtocol::findDownloadFileByHash(const QString& hash)
{
    QByteArray foo = FileInfo::stringToMd4(hash);
    return findDownloadFileByHash(foo);
}

FileInfo* DonkeyProtocol::findDownloadedFileNo(int fileno)
{
    return downloaded[fileno];
}

ServerInfo* DonkeyProtocol::findServerNo(int serverno)
{
    return servers[serverno];
}

Network* DonkeyProtocol::findNetworkNo(int nwno)
{
    return networks[nwno];
}

ClientInfo* DonkeyProtocol::findClientNo(int clno)
{
    return clients[clno];
}

ShareInfo* DonkeyProtocol::findShareNo(int shno)
{
    return shares[shno];
}

void DonkeyProtocol::sendConsoleMessage(const QString& msg)
{
    DonkeyMessage out(Command);
    out.writeString(msg);
    sock.sendMessage(out);
}

void DonkeyProtocol::saveFile(int fileno, const QString& name)
{
    DonkeyMessage out(SaveFile);
    out.writeInt32(fileno);
    out.writeString(name);
    sock.sendMessage(out);
}

void DonkeyProtocol::pauseFile(int fileno, bool pause)
{
    DonkeyMessage out(SwitchDownload);
    out.writeInt32(fileno);
    out.writeInt8((int8)!pause);
    sock.sendMessage(out);
}

void DonkeyProtocol::cancelFile(int fileno)
{
    DonkeyMessage out(RemoveDownload_query);
    out.writeInt32(fileno);
    sock.sendMessage(out);
}

void DonkeyProtocol::setFilePriority(int fileno, int pri)
{
    if (proto < 12) return;
    DonkeyMessage out(SetFilePriority);
    out.writeInt32(fileno);
    out.writeInt32(pri);
    sock.sendMessage(out);
}

void DonkeyProtocol::verifyFileChunks(int fileno)
{
    DonkeyMessage out(VerifyAllChunks);
    out.writeInt32(fileno);
    sock.sendMessage(out);
}

void DonkeyProtocol::getFileFormat(int fileno)
{
    DonkeyMessage out(QueryFormat);
    out.writeInt32(fileno);
    sock.sendMessage(out);
}

void DonkeyProtocol::previewFile(int fileno)
{
    DonkeyMessage out(Preview);
    out.writeInt32(fileno);
    sock.sendMessage(out);
}

void DonkeyProtocol::retryFile(int fileno)
{
    DonkeyMessage out(ConnectAll);
    out.writeInt32(fileno);
    sock.sendMessage(out);
}

void DonkeyProtocol::submitURL(const QString& url)
{
    DonkeyMessage out(Url);
    out.writeString(url);
    sock.sendMessage(out);
}

void DonkeyProtocol::sendMessage(const DonkeyMessage& msg)
{
    sock.sendMessage(msg);
}

void DonkeyProtocol::connectMoreServers()
{
    sock.sendMessage(DonkeyMessage(ConnectMore_query));
}

void DonkeyProtocol::cleanOldServers()
{
    sock.sendMessage(DonkeyMessage(CleanOldServers));
}

void DonkeyProtocol::addServer(int network, const QString& ip, int16 port)
{
    DonkeyMessage out(AddServer_query);
    out.writeInt32(network);
    out.writeInt8(1); // True means following IP address is a string
    out.writeString(ip);
    out.writeInt16(port);
    sock.sendMessage(out);
}

void DonkeyProtocol::removeServer(int serverno)
{
    DonkeyMessage out(RemoveServer_query);
    out.writeInt32(serverno);
    sock.sendMessage(out);
}

void DonkeyProtocol::getServerInfo(int serverno)
{
    DonkeyMessage out(GetServer_info);
    out.writeInt32(serverno);
    sock.sendMessage(out);
}

void DonkeyProtocol::getServerUsers(int serverno)
{
    DonkeyMessage out(GetServer_users);
    out.writeInt32(serverno);
    sock.sendMessage(out);
}

void DonkeyProtocol::connectServer(int serverno)
{
    DonkeyMessage out(ConnectServer);
    out.writeInt32(serverno);
    sock.sendMessage(out);
}

void DonkeyProtocol::disconnectServer(int serverno)
{
    DonkeyMessage out(DisconnectServer);
    out.writeInt32(serverno);
    sock.sendMessage(out);
}

SearchInfo* DonkeyProtocol::findSearchNo(int num)
{
    return searches[num];
}

const QIntDict<SearchInfo>& DonkeyProtocol::activeSearches()
{
    return searches;
}

void DonkeyProtocol::startSearch(int searchNum,
                                 SearchQuery* query,
                                 int maxHits,
                                 SearchType searchType,
                                 int network)
{
    DonkeyMessage out(Search_query);
    out.writeInt32((int32)searchNum); // search_num to identify searches
    query->writeQuery(out); // write the query into the message
    out.writeInt32((int32)maxHits); // search_max_hits
    out.writeInt8((int8)searchType); // search_type
    if (proto >= 16) out.writeInt32((int32)network); // network to search
    sock.sendMessage(out);
}

void DonkeyProtocol::stopSearch(int searchNum)
{
    DonkeyMessage out((proto >= 15) ? CloseSearch : ForgetSearch);
    out.writeInt32(searchNum);
    if (proto >= 15) out.writeInt8(true);
    sock.sendMessage(out);
}

void DonkeyProtocol::startDownload(const QStringList &names, int num, bool force)
{
    DonkeyMessage out((proto < 14) ? Download_query_v1 : Download_query);

    out.writeInt16(names.count());
    for (int i = 0; i < (int)names.count(); i++) out.writeString(names[i]);

    out.writeInt32(num);
    if (proto > 13) out.writeInt8(force);

    sock.sendMessage(out);
}

void DonkeyProtocol::refreshShared()
{
    DonkeyMessage out(RefreshUploadStats);
    sock.sendMessage(out);
}

void DonkeyProtocol::refreshFileInfo(int fileno)
{
    DonkeyMessage out(GetFile_info);
    out.writeInt32(fileno);
    sock.sendMessage(out);
}

void DonkeyProtocol::refreshClientInfo(int clientno)
{
    DonkeyMessage out(GetClient_info);
    out.writeInt32(clientno);
    sock.sendMessage(out);
}

uint DonkeyProtocol::connectedServerCount()
{
    return connectedservers;
}

uint DonkeyProtocol::totalServerCount()
{
    return servers.count();
}

void DonkeyProtocol::searchForFriend(const QString& name)
{
    DonkeyMessage out(FindFriend);
    out.writeString(name);
    sock.sendMessage(out);
}

void DonkeyProtocol::addClientFriend(int client)
{
    DonkeyMessage out(AddClientFriend);
    out.writeInt32(client);
    sock.sendMessage(out);
}

void DonkeyProtocol::removeFriend(int client)
{
    DonkeyMessage out(RemoveFriend);
    out.writeInt32(client);
    sock.sendMessage(out);
}

void DonkeyProtocol::connectFriend(int client)
{
    DonkeyMessage out(ConnectFriend);
    out.writeInt32(client);
    sock.sendMessage(out);
}

const ResultInfo* DonkeyProtocol::findClientFile(int fileno)
{
    return unmappedResults[fileno];
}

QMap<QString, QString> DonkeyProtocol::OptionsList()
{
    return options;
}

void DonkeyProtocol::setOption(const QString& option, const QString& value)
{
    DonkeyMessage out(SetOption);
    out.writeString(option);
    out.writeString(value);
    sock.sendMessage(out);
}

void DonkeyProtocol::setOptions(QMap<QString, QString> opts)
{
    if (opts.count() < 1) return;
    DonkeyMessage out(SaveOptions_query);
    out.writeInt16( (int16) opts.count() );
    for (QMap<QString, QString>::Iterator it = opts.begin(); it != opts.end(); ++it) {
        out.writeString( it.key() );
        out.writeString( it.data() );
    }
    sock.sendMessage(out);
}

void DonkeyProtocol::enableNetwork(int nwno, bool enable)
{
    Network* nw = networks[nwno];
    if (! nw) return;
    DonkeyMessage out(EnableNetwork);
    out.writeInt32( (int32) nw->networkNo() );
    out.writeInt8( (int8) (enable ? 1 : 0) );
    sock.sendMessage(out);
}
