/*
 * $Id: donkeysocket.cpp,v 1.7 2003/07/18 17:08:54 gibreel Exp $
 *
 * Copyright (C) 2003 Petter E. Stokke <gibreel@gibreel.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Log: donkeysocket.cpp,v $
 * Revision 1.7  2003/07/18 17:08:54  gibreel
 * Fixed a bug where DonkeyProtocol's tables weren't cleared on a manual
 * disconnect, and the friend table wasn't ever cleared. Replaced remaining
 * qDebug() calls with kdDebug().
 *
 * Revision 1.6  2003/06/13 18:20:02  gibreel
 * Libkmldonkey now uses references instead of pointers everywhere except where
 * it would cause an obvious performance impact, which should lead to less
 * chance of memory leaks and cleaner code in general. Almost everything that
 * should be const is now also const.
 *
 * Revision 1.5  2003/03/24 13:05:25  gibreel
 * Improved message handling efficiency: when several messages are received on
 * the socket at once, we process them in a loop rather than the old recursive
 * mechanism. Sometimes, when connecting as a full GUI, the core would send so
 * many messages at once that DonkeySocket would overload the stack. This
 * should no longer happen. Note that the sense of the readyMessage() signal
 * has changed - it is no longer emitted for every message that arrives, only
 * when the DonkeySocket feels it is time to process some messages.
 *
 * Revision 1.4  2003/03/08 20:13:20  gibreel
 * Switched readString/writeString in DonkeyMessage from using pointers to
 * QStrings to using references, which leads to far less hassle, shorter code,
 * and less chance of memory leaks.
 *
 * Revision 1.3  2003/03/08 17:08:12  gibreel
 * Lots of minor API changes.
 *
 * Revision 1.2  2003/03/07 20:35:38  gibreel
 * Added submitURL() method and option to disable flushing of the output buffer
 * on disconnect. Removed a ton of debug messages.
 *
 * Revision 1.1.1.1  2003/03/07 11:50:20  gibreel
 * Initial import.
 *
 */

#include <kdebug.h>
#include <qsocket.h>
#include <stdlib.h>

#include "donkeysocket.h"
#include "donkeysocket.moc"



DonkeySocket::DonkeySocket( QObject *parent, const char *name )
    : QSocket( parent, name)
{
    fifo.setAutoDelete(TRUE);
    mlHost = QString::null;
    mlPort = 4001;
    connect(this, SIGNAL(readyRead()), SLOT(readMessage()));
}

DonkeySocket::~DonkeySocket()
{
}

void DonkeySocket::connectDonkey()
{
    rstate = 0; ct = 0; sz = 0;
    kdDebug() << "Connecting to " << mlHost << ":" << mlPort << endl;
    connectToHost(mlHost, mlPort);
}

void DonkeySocket::connectDonkey(const QString& host, Q_UINT16 port)
{
    mlHost = host;
    mlPort = port;
    connectDonkey();
}

int DonkeySocket::sendMessage(const DonkeyMessage& msg)
{
    unsigned char op[4];
    int sz = msg.size() + 2;
    op[0] = (unsigned char)(sz & 0xff);
    op[1] = (unsigned char)((sz >> 8) & 0xff);
    op[2] = (unsigned char)((sz >> 16) & 0xff);
    op[3] = (unsigned char)((sz >> 24) & 0xff);
    Q_LONG r = writeBlock((const char *)op, 4);
    if (r != 4) return 0;
    op[0] = (unsigned char)(msg.opcode() & 0xff);
    op[1] = (unsigned char)((msg.opcode() >> 8) & 0xff);
    r = writeBlock((const char *)op, 2);
    if (r != 2) return 0;
    r = writeBlock((const char*)msg.data(), msg.size());
    return r == (Q_LONG)msg.size();
}

void DonkeySocket::readMessage()
{
    int done = 1;
    while (bytesAvailable() && done) {
        done = 0;
        if (!rstate && bytesAvailable() >= 4) {
            ct = 0; rstate = 1;
            sz = getch();
            sz |= getch() << 8;
            sz |= getch() << 16;
            sz |= getch() << 24;
            sz -= 2;
            done = 1;
        }
        if (rstate == 1 && (int)bytesAvailable() >= 2) {
            int opcode = getch();
            opcode |= getch() << 8;
            cur = new DonkeyMessage(opcode, sz);
            rstate = 2;
            done = 1;
        }
        if (rstate == 2 && (int)bytesAvailable()) {
            int a = (int)bytesAvailable() > (sz - ct) ? sz - ct : bytesAvailable();
            char* data = (char*)malloc(bytesAvailable());
            if (!data) exit(1);
            int r = (int)readBlock(data, a);
            if (r < 0) { rstate = 0; free(data); delete cur; kdDebug() << "Read error!" << endl; return; }
            cur->feedBuffer(data, r);
            free(data);
            ct += r;
            if (ct >= sz) {
                cur->resetPosition();
                fifo.enqueue(cur);
                cur = NULL;
                rstate = sz = ct = 0;
            }
            done = 1;
        }
    }
    if (!fifo.isEmpty())
	emit readyMessage();
}

uint DonkeySocket::queueSize()
{
    return fifo.count();
}

DonkeyMessage* DonkeySocket::popMessage()
{
    return fifo.dequeue();
}
