/*
 * $Id: fileinfo.cpp,v 1.9 2003/06/29 13:39:28 gibreel Exp $
 *
 * Copyright (C) 2003 Petter E. Stokke <gibreel@gibreel.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Log: fileinfo.cpp,v $
 * Revision 1.9  2003/06/29 13:39:28  gibreel
 * Tried to optimise the particularly memory hungry file source lists.
 *
 * Revision 1.8  2003/06/28 22:34:53  gibreel
 * Fixed a serious memory leaking bug that was recently introduced.
 *
 * Revision 1.7  2003/06/28 21:17:00  gibreel
 * The file info dialog now displays file sources. Libkmldonkey received a
 * number of updates and bugfixes to facilitate this.
 *
 * Revision 1.6  2003/06/13 18:20:02  gibreel
 * Libkmldonkey now uses references instead of pointers everywhere except where
 * it would cause an obvious performance impact, which should lead to less
 * chance of memory leaks and cleaner code in general. Almost everything that
 * should be const is now also const.
 *
 * Revision 1.5  2003/04/06 21:28:48  gibreel
 * Added missing fileNetwork() method.
 *
 * Revision 1.4  2003/03/23 23:34:59  gibreel
 * A lot of API additions, especially lists keeping track of shared files and
 * clients.
 *
 * Revision 1.3  2003/03/08 20:13:21  gibreel
 * Switched readString/writeString in DonkeyMessage from using pointers to
 * QStrings to using references, which leads to far less hassle, shorter code,
 * and less chance of memory leaks.
 *
 * Revision 1.2  2003/03/07 21:51:18  gibreel
 * Changed the MD4 storage in the FileInfo object from an array of int to a
 * QByteArray, added conversion functions between that and QString, and methods
 * to search downloaded files by hash. The applet now uses the file hash
 * instead of the fileno when storing and restoring priorities, as the fileno
 * changes between core restarts.
 *
 * Revision 1.1.1.1  2003/03/07 11:50:21  gibreel
 * Initial import.
 *
 */


#include <kdebug.h>

#include "fileinfo.h"



FileInfo::FileInfo(DonkeyMessage* msg, int proto)
{
    updateFileInfo(msg, proto);
}

FileInfo::~FileInfo()
{
}

void FileInfo::updateFileInfo(FileInfo* fi)
{
    network = fi->fileNetwork();
    names = fi->fileNames();
    md4 = fi->fileMD4();
    size = fi->fileSize();
    downloaded = fi->fileDownloaded();
    nlocations = fi->fileNLocations();
    nclients = fi->fileNClients();
    state = fi->fileState();
    chunks = fi->fileChunks();
    availability = fi->fileAvailability();
    speed = fi->fileSpeed();
    chunks_age = fi->fileChunksAge();
    age = fi->fileAge();
    format = fi->fileFormat();
    formatinfo = fi->fileFormatInfo();
    name = fi->fileName();
    lastseen = fi->fileLastSeen();
    priority = fi->filePriority();
}

void FileInfo::updateFileInfo(DonkeyMessage* msg, int proto)
{
    int i,j;
    num = msg->readInt32();
    network = msg->readInt32();
    j = msg->readInt16();
    names.clear();
    for (i=0; i<j; i++)
	names.append(msg->readString());
    md4 = QByteArray(16);
    for (i=0; i<16; i++)
	md4[i] = msg->readInt8();
    size = msg->readInt32();
    downloaded = msg->readInt32();
    nlocations = msg->readInt32();
    nclients = msg->readInt32();
    state = (State)msg->readInt8();
    if (state == 6)
	abortedmsg = msg->readString();
    chunks = msg->readString();
    availability = msg->readString();
    speed = msg->readFloat();
    j = msg->readInt16();
    chunks_age.clear();
    for (i=0; i<j; i++)
	chunks_age.append(msg->readFloat());
    age = msg->readFloat();
    format = msg->readInt8();
    switch (format) {
    case 1:
    {
	QString foo = msg->readString();
	QString bar = msg->readString();
	formatinfo = foo + " " + bar;
    } break;
    case 2:
    {
	QString foo = msg->readString();
	int i = msg->readInt32();
	int j = msg->readInt32();
	int k = msg->readInt32();
	int l = msg->readInt32();
	formatinfo.sprintf("AVI %s %dx%d %.2ffps rate %d", foo.latin1(), i, j, (double)k / 1000.0, l);
    } break;
    case 3:
    {
	QString foo = msg->readString();
	QString bar = msg->readString();
	QString baz = msg->readString();
	QString quux = msg->readString();
	msg->readString();
	int i = msg->readInt32();
	msg->readInt32();
	formatinfo.sprintf("MP3 Ti:%s Ar:%s Al:%s Yr:%s Tr:%d", foo.latin1(),
			   bar.latin1(), baz.latin1(), quux.latin1(), i);
    } break;
    default:
	formatinfo = "Unknown format";
	break;
    }
    if (proto >= 8)
	name = msg->readString();
    else
	name = names[0];
    if (proto >= 9)
	lastseen = msg->readInt32();
    else
	lastseen = -1;
    if (proto >= 12)
	priority = msg->readInt32();
    else
	priority = 0;
    if (priority > 0x40000000) priority -= 0x80000000;
}

const int FileInfo::fileNo()
{
    return num;
}

const int FileInfo::fileNetwork()
{
    return network;
}

const QString& FileInfo::fileName()
{
    return name;
}

const QStringList& FileInfo::fileNames()
{
    return names;
}

const QByteArray& FileInfo::fileMD4()
{
    return md4;
}

const int64 FileInfo::fileSize()
{
    return size;
}

const int64 FileInfo::fileDownloaded()
{
    return downloaded;
}

const int FileInfo::fileNLocations()
{
    return nlocations;
}

const int FileInfo::fileNClients()
{
    return nclients;
}

const FileInfo::State FileInfo::fileState()
{
    return state;
}

const QString& FileInfo::fileAbortedMsg()
{
    return abortedmsg;
}

const QString& FileInfo::fileChunks()
{
    return chunks;
}

const QString& FileInfo::fileAvailability()
{
    return availability;
}

const double FileInfo::fileSpeed()
{
    return speed;
}

const QValueList<double>& FileInfo::fileChunksAge()
{
    return chunks_age;
}

const double FileInfo::fileAge()
{
    return age;
}

const int FileInfo::fileFormat()
{
    return format;
}

const QString& FileInfo::fileFormatInfo()
{
    return formatinfo;
}

const int FileInfo::fileLastSeen()
{
    return lastseen;
}

const int FileInfo::filePriority()
{
    return priority;
}

const QMap<int,QString>& FileInfo::fileSources()
{
    return sources;
}


void FileInfo::setFileName(const QString& newname)
{
    name = newname;
}

void FileInfo::addSource(int source)
{
    if (!sources.contains(source))
	sources.insert(source, QString::null);
}

void FileInfo::removeSource(int source)
{
    sources.remove(source);
}

void FileInfo::updateAvailability(int source, const QString& avail)
{
    sources.insert(source, avail);
}

void FileInfo::updateDownloadStatus(DonkeyMessage* msg, int)
{
    downloaded = msg->readInt32();
    speed = msg->readFloat();
    if (msg->opcode() >= 46)
        lastseen = msg->readInt32();
}


QString FileInfo::md4ToString(const QByteArray& hash)
{
    char foo[33], bar[16];
    int i;
    foo[0] = 0;
    for (i=0; i<16; i++) {
	sprintf(bar, "%02x", (unsigned char)hash[i]);
	strcat(foo, bar);
    }
    return QString(foo).upper();
}

static const QString hexmap = "0123456789ABCDEF";

QByteArray FileInfo::stringToMd4(const QString& hash)
{
    QByteArray out(16);
    if (hash.length() != 32) return out;
    QString in = hash.upper();
    int b;
    for (int i=0; i<(int)in.length(); i+=2) {
	b = hexmap.find(in[i]) << 4;
	b |= hexmap.find(in[i+1]);
	out[i>>1] = b;
    }
    return out;
}

