/* -*- c++ -*-
 * $Id: fileinfo.h,v 1.8 2003/07/23 18:06:12 gibreel Exp $
 *
 * Copyright (C) 2003 Petter E. Stokke <gibreel@gibreel.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Log: fileinfo.h,v $
 * Revision 1.8  2003/07/23 18:06:12  gibreel
 * Added API doc generation using the KDE doc generation build system. Briefly
 * documented as many classes as necessary to produce useful documentation.
 *
 * Revision 1.7  2003/06/28 21:17:00  gibreel
 * The file info dialog now displays file sources. Libkmldonkey received a
 * number of updates and bugfixes to facilitate this.
 *
 * Revision 1.6  2003/06/13 18:20:02  gibreel
 * Libkmldonkey now uses references instead of pointers everywhere except where
 * it would cause an obvious performance impact, which should lead to less
 * chance of memory leaks and cleaner code in general. Almost everything that
 * should be const is now also const.
 *
 * Revision 1.5  2003/04/06 21:28:48  gibreel
 * Added missing fileNetwork() method.
 *
 * Revision 1.4  2003/03/23 23:34:59  gibreel
 * A lot of API additions, especially lists keeping track of shared files and
 * clients.
 *
 * Revision 1.3  2003/03/10 14:52:02  gibreel
 * Support for GUI protocol 14. Specifically: Authentication with username, new
 * download file state "Queued", and support for the new message types in the
 * DonkeyProtocol class.
 *
 * Revision 1.2  2003/03/07 21:51:18  gibreel
 * Changed the MD4 storage in the FileInfo object from an array of int to a
 * QByteArray, added conversion functions between that and QString, and methods
 * to search downloaded files by hash. The applet now uses the file hash
 * instead of the fileno when storing and restoring priorities, as the fileno
 * changes between core restarts.
 *
 * Revision 1.1.1.1  2003/03/07 11:50:21  gibreel
 * Initial import.
 *
 */

#ifndef __libkmldonkey_fileinfo_h__
#define __libkmldonkey_fileinfo_h__

#include <qstring.h>
#include <qstringlist.h>
#include <qvaluelist.h>
#include <qcstring.h>
#include <qmap.h>
#include "donkeymessage.h"

//! Representation of a file.

class FileInfo
{

    public:

    //! File state.
    enum State {
	Downloading = 0,
	Paused,
	Complete,
	Shared,
	Cancelled,
	New,
	Aborted,
	Queued
    };

    FileInfo(DonkeyMessage* msg, int proto);
    ~FileInfo();

    void updateFileInfo(DonkeyMessage* msg, int proto);
    void updateFileInfo(FileInfo* fi);

    //! The numerical ID of the file.
    const int fileNo();
    //! The network the file is on.
    const int fileNetwork();
    //! The file name.
    const QString& fileName();
    //! Alternative file names found on the network.
    const QStringList& fileNames();
    //! The file's MD4 hash.
    const QByteArray& fileMD4();
    //! The file size.
    const int64 fileSize();
    //! Bytes downloaded so far.
    const int64 fileDownloaded();
    const int fileNLocations();
    const int fileNClients();
    //! The file state.
    const State fileState();
    //! If state is Aborted, this is the reason why.
    const QString& fileAbortedMsg();
    //! String describing file chunks downloaded, one char for each chunk.
    const QString& fileChunks();
    //! String describing chunk availability, one char for each chunk.
    const QString& fileAvailability();
    //! Current download speed.
    const double fileSpeed();
    //! The age of individual chunks.
    const QValueList<double>& fileChunksAge();
    //! The time the download was started (seconds since Epoch).
    const double fileAge();
    //! The file format.
    const int fileFormat();
    //! A string describing the file format in more detail.
    const QString& fileFormatInfo();
    //! Seconds since the file was last seen on the network.
    const int fileLastSeen();
    //! The file priority.
    const int filePriority();
    //! File availability described by a map of client IDs to strings similar to fileChunks().
    const QMap<int,QString>& fileSources();

    void setFileName(const QString& newname);

    void addSource(int source);
    void removeSource(int source);
    void updateAvailability(int source, const QString& avail);
    void updateDownloadStatus(DonkeyMessage* msg, int proto);

    //! Converts a QByteArray containing an MD4 hash to a readable string.
    static QString md4ToString(const QByteArray& hash);
    //! Converts a string describing an MD4 hash into a QByteArray.
    static QByteArray stringToMd4(const QString& hash);

private:

    int num, network;
    QString name;
    QStringList names;
    QByteArray md4;
    int64 size, downloaded;
    int nlocations, nclients;
    State state;
    QString abortedmsg;
    QString chunks, availability;
    double speed;
    QValueList<double> chunks_age;
    double age;
    int format;
    QString formatinfo;
    int lastseen, priority;

    QMap<int,QString> sources;
};

#endif
