/*
 * $Id: submit.cpp,v 1.7 2003/07/23 18:03:47 gibreel Exp $
 *
 * Copyright (C) 2003 Petter E. Stokke <gibreel@gibreel.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * $Log: submit.cpp,v $
 * Revision 1.7  2003/07/23 18:03:47  gibreel
 * Support for multiple host definitions using HostManager.
 *
 * Revision 1.6  2003/05/29 20:37:54  gibreel
 * Cleaned up KAboutData structures for console display.
 *
 * Revision 1.5  2003/05/01 21:24:21  gibreel
 * Added option --read-html to control whether to attempt to read and parse
 * HTTP URLs for ed2k content, or simply send them directly to the donkey (as
 * would be the case with BitTorrent URLs).
 *
 * Revision 1.4  2003/03/10 14:52:02  gibreel
 * Support for GUI protocol 14. Specifically: Authentication with username, new
 * download file state "Queued", and support for the new message types in the
 * DonkeyProtocol class.
 *
 * Revision 1.3  2003/03/08 17:08:13  gibreel
 * Lots of minor API changes.
 *
 * Revision 1.2  2003/03/08 13:55:14  gibreel
 * Static version string replaced by the VERSION macro provided by autoconf.
 *
 * Revision 1.1  2003/03/07 20:53:17  gibreel
 * Removed the ed2k handler Python scripts, and replaced them with a KDE/C++
 * application.
 *
 */

#include <kdebug.h>
#include <kconfig.h>
#include <klocale.h>
#include <kapplication.h>
#include <kaboutdata.h>
#include <kcmdlineargs.h>
#include <qmessagebox.h>
#include <qregexp.h>
#include "donkeyprotocol.h"
#include "config.h"
#include "submit.h"
#include "submit.moc"


#define MESSAGEBOX_TITLE "KMLDonkey URL error"


Submitter::Submitter(bool recursive, QObject* parent, char* name) : QObject(parent,name)
{
    isRecursive = recursive;
    done = auth = 0;
    dict.setAutoDelete(true);

    hostManager = new HostManager(this);
    donkey = new DonkeyProtocol(true, this);
    connect(donkey, SIGNAL(donkeyDisconnected(int)), this, SLOT(donkeyDisconnected(int)));
    connect(donkey, SIGNAL(donkeyConnected()), this, SLOT(donkeyAuthenticating()));
    connect(donkey, SIGNAL(updatedConnectedServers()), this, SLOT(donkeyConnected()));
    donkey->setPassword(uname, pwd);
    donkey->connectDonkey(hostManager->defaultHost());
}

void Submitter::donkeyDisconnected(int err)
{
    kdDebug() << "Donkey disconnected: " << err << endl;
    switch (err) {
    case DonkeyProtocol::ConnectionRefusedError:
	QMessageBox::critical(NULL, MESSAGEBOX_TITLE,
			      "Connection refused!");
	break;
    case DonkeyProtocol::HostNotFoundError:
	QMessageBox::critical(NULL, MESSAGEBOX_TITLE,
			      "Host not found!");
	break;
    case DonkeyProtocol::SocketReadError:
	QMessageBox::critical(NULL, MESSAGEBOX_TITLE,
			      "Read error, donkey has disconnected!");
	break;
    case DonkeyProtocol::ObsoleteProtocolError:
	QMessageBox::critical(NULL, MESSAGEBOX_TITLE,
			      "Obsolete core protocol!\n\nPlease upgrade your mldonkey core.");
	break;
    default:
	if (!auth || err == DonkeyProtocol::AuthenticationError)
	    QMessageBox::critical(NULL, MESSAGEBOX_TITLE,
				  "Authentication error!");
	break;
    }
    emit quit();
}

void Submitter::donkeyAuthenticating()
{
    donkey->updateConnectedServers();
    kdDebug() << "Authenticating..." << endl;
}

void Submitter::donkeyConnected()
{
    kdDebug() << "Donkey authenticated." << endl;
    auth = 1;
    sendURLs();
}

void Submitter::sendURLs()
{
    if (!auth) return;
    while (list.size()) {
	QString foo = list.first().prettyURL();
	donkey->submitURL(foo);
	kdDebug() << "Submitted URL: " << foo.latin1() << endl;
	list.pop_front();
    }
    if (done && dict.isEmpty()) donkey->disconnectDonkey(0);
}

void Submitter::submitURL(KURL& url)
{
    if (url.protocol() == "ed2k" || !isRecursive) {
	list.append(url);
	sendURLs();
    } else {
	KIO::TransferJob *job = KIO::get(url);
	connect(job, SIGNAL(result(KIO::Job*)), this, SLOT(jobResult(KIO::Job*)));
	connect(job, SIGNAL(data(KIO::Job*, const QByteArray&)), this, SLOT(jobData(KIO::Job*, const QByteArray&)));
	dict.insert(job->progressId(), new QString());
	kdDebug() << "Job " << job->progressId() << " fetching URL: " << url.prettyURL() << endl;
    }
}

void Submitter::allDone()
{
    done = 1;
    sendURLs();
}

void Submitter::jobData(KIO::Job* job, const QByteArray& data)
{
    dict.find(job->progressId())->append(data);
}

void Submitter::jobResult(KIO::Job* job)
{
    if (job->error())
	QMessageBox::critical(NULL, MESSAGEBOX_TITLE, job->errorString(), QMessageBox::Abort, QMessageBox::NoButton);

    QString* data = dict.find(job->progressId());
    kdDebug() << "Job " << job->progressId() << " done." << endl;

    QRegExp rx("(ed2k://[|]file[|][^|]+[|]\\d+[|][0-9a-fA-F]+[|])");
    int pos = 0;

    while (pos >= 0) {
	pos = rx.search(*data, pos);
	if (pos > -1) {
	    KURL url(rx.cap(1));
	    submitURL(url);
	    pos += rx.matchedLength();
	}
    }

    dict.remove(job->progressId());
    sendURLs();
}

static KCmdLineOptions options[] = {
    { "r", 0, 0 },
    { "read-html", I18N_NOOP("Load HTTP URLs and add ed2k links contained in the retrieved documents rather than submitting them as is."), 0 },
    { "+url", I18N_NOOP("URL (or multiple URLs) to process."), 0 },
    { 0,0,0 }
};

int main(int argc, char **argv)
{
    KAboutData about("kmldonkey", I18N_NOOP("kmldonkey_submit"), VERSION,
		     I18N_NOOP("Submit a URL to mldonkey"),
                     KAboutData::License_GPL_V2, I18N_NOOP("Copyright (C) 2003 Petter E. Stokke"),
		     I18N_NOOP("Part of the KMLDonkey package."),
		     "http://www.gibreel.net/projects/kmldonkey/", "gibreel@gibreel.net");
    about.addAuthor("Petter E. Stokke", 0, "gibreel@gibreel.net");
    about.setTranslator(I18N_NOOP("_: NAME OF TRANSLATORS\nYour names"), I18N_NOOP("_: EMAIL OF TRANSLATORS\nYour emails"));
    KCmdLineArgs::init(argc, argv, &about);
    KCmdLineArgs::addCmdLineOptions(options);
    KApplication::addCmdLineOptions();
    KApplication app;
    KCmdLineArgs* args = KCmdLineArgs::parsedArgs();

    if (!args->count()) args->usage();

    Submitter sub(args->isSet("read-html"), &app);
    app.connect(&sub, SIGNAL(quit()), SLOT(quit()));
    for (int i = 0; i < args->count(); i++) {
	KURL foo = args->url(i);
	sub.submitURL(foo);
    }
    sub.allDone();

    app.exec();
}
