/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: viewer.h,v 1.2 2004/06/14 15:16:33 jd Exp $

   $Log: viewer.h,v $
   Revision 1.2  2004/06/14 15:16:33  jd
   no more drawing neither resize in windowing

   Revision 1.1  2004/05/21 23:20:54  jd
   viewer for window


 
   Created 05/17/04 by Jean-Dominique Frattini <zionarea@free.fr>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*!\file libs/graphics/viewer.h
  \brief viewer link with gl.
*/

#ifndef LEG_LIBS_GRAPHICS_VIEWER_H
#define LEG_LIBS_GRAPHICS_VIEWER_H

#include "leg/support/graphics/viewmode.h"

namespace leg
{
namespace libs
{
namespace graphics
{

using leg::support::graphics::projection;
using leg::support::graphics::modelview;
//using leg::support::graphics::texture;

//! Viewing set for window.
/*!
 * Set the OpenGL viewing stuff that links with the window and eventually the
 * camera too.
 * These functions could dynamically allocate and create types given in the
 * template arguments. So the types should be the real type because 
 * polymorphism isn't used here.
 *
 * status:
 *    in development.
 *
 * info:
 *    none.
 */
template
<
   class T3DProjection,
   class T2DProjection,
   class TViewport,
   class TViewmode
>
class Viewer
{
   protected:

   typedef TViewport	   Viewport;	     // the viewport.
   typedef T3DProjection   Projection3D;     // the 3d projection matrix.
   typedef T2DProjection   Projection2D;     // 2d projection matrix.
   typedef TViewmode	   Viewmatrix;	     // the viewing matrix mode.

   Viewport	  *viewport;
   Projection3D	  *projection_3d;
   Projection2D	  *projection_2d;
   Viewmatrix	  *viewmatrix;

   public:

   Viewer ():  viewport (0),
	       projection_3d (0),
	       projection_2d (0),
	       viewmatrix (new Viewmatrix())
   {
   }

   virtual ~Viewer()
   {
      delete viewmatrix;
      delete viewport;
      delete projection_3d;
      delete projection_2d;
      
      viewmatrix = 0;
      viewport = 0;
      projection_3d = 0;
      projection_2d = 0;
   }

   //! Calls the viewport to execute.
   void
   SetViewport()
   {
      viewport->Set();
   }
   
   //! Set the viewport to those actual values.
   void
   SetViewport (int x, int y, unsigned int w, unsigned int h)
   {
      if (viewport){
	 delete viewport;
	 viewport = 0;
      }
      viewport = new Viewport (x,y,w,h);
   }

   Viewport&
   GetViewport()
   {
      return *viewport;
   }

   void
   SetProjection3D()
   {
      projection_3d->Set();
   }
   
   void
   SetProjection3D (float fovy, float ratio, float near, float far)
   {
      if (projection_3d){
	 delete projection_3d;
	 projection_3d = 0;
      }
      projection_3d = new Projection3D (fovy,ratio,near,far);
   }

   Projection3D&
   GetProjection3D()
   {
      return *projection_3d;
   }

   void
   SetProjection2D()
   {
      projection_2d->Set();
   }
   
   void
   SetProjection2D (double left, double right, double bottom, double top, double near, double far)
   {
      if (projection_2d){
	 delete projection_2d;
	 projection_2d = 0;
      }
      projection_2d = new Projection2D (left,right,bottom,top,near,far);
   }

   Projection2D&
   GetProjection2D()
   {
      return *projection_2d;
   }

   Viewmatrix&
   GetViewmatrix()
   {
      return *viewmatrix;
   }

   template <typename ViewMode>
   void
   SetViewmatrix (ViewMode mode)
   {
      viewmatrix->Set (mode);
   }

   //! TODO: check the arguments to Act
   // for when window has resized for example.
   template <class Window>
   void
   Resize (Window& win)
   {
      win.Act ("resize");//input::resize);
      unsigned int w = 0, h = 0;
      win.GetAttributes().GetSize (w,h);
      SetViewport (0,0,w,h);
      SetViewport();
      SetViewmatrix (projection);
      glLoadIdentity();
      SetProjection3D();
      SetViewmatrix (modelview);
      glLoadIdentity();
      win.Act ("sized");//input::sized);
   }
};

}
}
}
#endif
