/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: keyboard.h,v 1.1 2004/06/19 11:14:48 jd Exp $

   Created 06/16/04 by Jean-Dominique Frattini <zionarea@free.fr>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file libs/interactivity/keyboard.h
  \brief keyboard controler.
*/

#ifndef LEG_LIBS_INTERACTIVITY_KEYBOARD_H
#define LEG_LIBS_INTERACTIVITY_KEYBOARD_H

#include "controler.h"
#include "leg/support/timing/timing.h"
#include "leg/support/window/window.h"
#include <map>
#include <string>

namespace leg
{
namespace libs
{
namespace interactivity
{

using support::timing::Timing;

typedef support::window::Key	    Key;

//! Keyboard map.
/*!
 * A keyboard map contains Key keys and a matching boolean value telling if the
 * key is currently pressed or not. KeyboardMap is intended to be used for the
 * Event classes and interactivity only.
 */
typedef std::map<Key,bool>	    KeyboardMap;

//! KeyClickTimeMap represents a map of stored keyboard keys and GameTime.
/*!
 * This is useful to know when a key has been pressed (with GameTime precision
 * level) and this helps to detect clicks (and in the future double-clicks).
 */
typedef std::map<Key,GameTime>	    KeyClickTimeMap;

//! KeyClickMap.
/*!
 * Stores keyboard keys values with an value that is incremented each time a
 * click has been recognized, so theoretically, allowing n-clicks recognitions.
 */
typedef std::map<Key,unsigned int>  KeyClickMap;

//! Keyboard controler
/*!
 * A simple keyboard for game uses.
 *
 * status:
 *    in development.
 *
 * info:
 *    Even if polymorphism is enabled in this hierarchy, you should not use 
 *    this capability. This is because controler interfaces vary quiete well
 *    from one controler type to another (mouse have motions that keyboard does
 *    not have).
 *
 * @sa Controler.
 */
class Keyboard: private Controler
{
   Keyboard (const Keyboard&);
   const Keyboard& operator = (const Keyboard&);

   class StringManager
   {
      public:

      static void
      Begin();

      static void
      End();

      static std::string&
      Get();
   };

   protected:

   static KeyboardMap	      map;		   //!< the current keyboard map. READ ONLY .
   static KeyboardMap	      release_map;	   //!< the current released keybaord map. READ ONLY .
   static KeyboardMap	      last_map;		   //!< the previous mouse map.
   static KeyClickTimeMap     pressed_time_map;	   //!< map memoring the time when the buttons were pressed.
   static KeyClickMap	      click_map;	   //!< map remembering the clicks.
   static Timing	      timing;		   //!< The time.
   static GameTime	      click_time;	   //!< Click time duration (in seconds).
   static Key		      current_key;
   static char		      current_char;	   //!< Current corresponding ASCII char if any, or 0.
   static std::string	      current_string;	   //!< Current string memoried.

   public:
   
   Keyboard();

   ~Keyboard ();

   //! Get the current keyboard map.
   static const KeyboardMap&
   GetMap();

   //! Questions if a key is currently pressed.
   static bool
   IsKeyPressed (Key key);

   //! Test if a key has been pressed, then released.
   static bool
   IsKeyReleased (Key key);

   //! Questions if a key is clicked.
   /*!
    * Returns true if the key given in argument is just clicked. Like mouse
    * buttons, key clicks are when a key was pressed, then released in a
    * certain time delay.
    */
   static bool
   IsKeyClicked (Key key);

   //! Questions if a key is double-clicked -not working.
   static bool
   IsKeyDoubleClicked (Key key);

   //! Set the time duration between clicks and double clicks can be accepted.
   /*!
    * Be careful, high values (generally more than half a second) are quiete
    * dangerous, as they may provoke several confusions. A value too much near
    * from zero is also not advised. 0.2 up to 0.4 second seem well.
    */
   static void
   SetClickTimeDuration (GameTime duration);

   //! Returns the click time duration.
   static GameTime
   GetClickTimeDuration();

   //! Get the current char of the pressed key if any.
   /*!
    * Return the char (ASCII value) of a pressed key if any. Returns 0 if there
    * aren't. If several keys were pressed before 2 calls of this function,
    * this returns the last key pressed only. Use GetCurrentString() instead if
    * you want to have the string in order.
    */
   static char
   GetCurrentChar();

   static Key
   GetCurrentKey();

   //! Get the current string.
   /*!
    * Return the string of order pressed keys if saved or empty string instead.
    * Use SetSaveString (bool) to set if key press will be stored in a string.
    * This saves even not character keys like escape, control, F keys...
    */
   static const std::string&
   GetCurrentString();

   //! Set if key pressed will be saved in a string.
   static void
   SetSaveString (bool val);

   //! Get if key pressed are saved in a string.
   static bool
   GetSaveString();

   //! Force a key press, even if not real.
   static void
   PressKey (Key key);

   //! Force to unpress a key, even if not real.
   static void
   UnpressKey (Key key);

   protected:
   
   //! Update the status of the keys (clicks recognitions...).
   static void
   UpdateKeyStatus (Key k, bool press);

   static void
   UpdateChar (Key k, bool press);
};

}
}
}
#endif
