/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: mouse.h,v 1.2 2004/06/19 11:14:48 jd Exp $

   Created 04/29/04 by Jean-Dominique Frattini <zionarea@free.fr>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file libs/interactivity/mouse.h
  \brief mouse controler.
*/

#ifndef LEG_LIBS_INTERACTIVITY_MOUSE_H
#define LEG_LIBS_INTERACTIVITY_MOUSE_H

#include "controler.h"
#include "mousecoord.h"
#include "leg/support/window/window.h"
#include <map>

namespace leg
{
namespace libs
{
   
//! First layer of user-intended interactivity.
/*!
 * Provides first interactivity dedicated directly to game programming.
 */
namespace interactivity
{

using support::timing::Timing;

typedef support::window::MouseButton	     MouseButton;

//! Mouse buttons map.
/*!
 * Map like KeyboardMap for holding the mouse buttons presses.
 */
typedef std::map<MouseButton,bool>	     MouseMap;

typedef std::map<MouseButton,GameTime>	     MouseClickTimeMap;

typedef std::map<MouseButton,unsigned int>   MouseClickMap;

//! Mouse controler for general uses.
/*!
 * This is a mouse presented to users, that allows to get all the informations
 * transmitted by the 'mouse-user' to our program. Mouse tries to present more
 * functionalities like detecting clicks (not only presses), or double-clicks
 * and others. Mouse owns a MouseCoord member in order to get alterated
 * coordinates (like ranging them between 0 and 1). See this class for more
 * information about what it can do.
 *
 * status:
 *    in development.
 *
 * info:
 *    double-clicks are not recognized. n-clicks may be possible.
 *
 * @sa Controler, MouseCoord.
 */
class Mouse: protected Controler
{
   Mouse (const Mouse&);
   const Mouse& operator = (const Mouse&);

   protected:

   static MouseClickTimeMap   pressed_time_map;	   //!< map memoring the time when the buttons were pressed.
   static MouseClickMap	      click_map;	   //!< map remembering the clicks.
   static MouseMap	      map;		   //!< the current mouse map. READ ONLY .
   static MouseMap	      release_map;	   //!< the current mouse map. READ ONLY .
   static MouseMap	      last_map;		   //!< the previous mouse map.
   static MouseCoord	      *coord;		   //!< Coordinates of the mouse pointer.
   static Timing	      timing;		   //!< The time.
   static GameTime	      click_time;	   //!< Click time duration (in seconds).
   static MouseButton	      current_button;	   //!< Current pressed button if any.

   public:
   
   //! Only available constructor.
   Mouse();

   ~Mouse ();

   //! Set a screen size change.
   /*!
    * Do not use this function if the screen has not changed !
    */
   static void
   SetNewScreenSize (unsigned int w, unsigned int h);

   //! Update the mouse statuses.
   /*!
    * Updates the mouse pointer coordinates and the buttons status, in order
    * the questioning functions to be freely useable.
    */
   static void
   Update ();

   //! Get the current mouse coordinates (readable only).
   static MouseCoord&
   GetCoord();

   static void
   GetCurrentCoordinates (unsigned int& x, unsigned int& y);

   static MouseButton
   GetCurrentButton();

   //! Get the current mouse buttons map.
   static const MouseMap&
   GetMap();

   //! Questions if a button is currently pressed.
   /*!
    * This answers regarding the current status updated by Update(), so that
    * it's possible a button press is recognized only on a next call to this
    * function. This is because actualization and questioning behave
    * asynchronously.
    */
   static bool
   IsButtonPressed (MouseButton mb);

   //! Questions if a button is clicked.
   /*!
    * Returns true if the button given in argument is just clicked.
    */
   static bool
   IsButtonClicked (MouseButton mb);

   //! Questions if a button is double-clicked -not working.
   static bool
   IsButtonDoubleClicked (MouseButton mb);

   //! Set the time duration between clicks and double clicks can be accepted.
   /*!
    * Be careful, high values (generally more than half a second) are quiete
    * dangerous, as they may provoke several confusions. A value too much near
    * from zero is also not advised. 0.2 up to 0.4 second seem well.
    */
   static void
   SetClickTimeDuration (GameTime duration);

   //! Returns the click time duration.
   static GameTime
   GetClickTimeDuration();

   //! Set the sensibility of the mouse (in development).
   /*!
    * The sensibility of the mouse is a floating value, greater than 0, that is
    * used as a factor to get the mouse coordinates.
    */
   static void
   SetSensibility (float s);

   //! Get the sensibility.
   static float
   GetSensibility ();

   //! Force a button press, even if not real.
   static void
   PressButton (MouseButton mb);

   //! Force to unpress a button, even if not real.
   static void
   UnpressButton (MouseButton mb);

   //! Update the status of the mouse buttons (clicks recognitions...).
   static void
   UpdateButtonStatus (MouseButton n, bool press);
};

}
}
}
#endif
