/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: entity.h,v 1.3 2004/07/15 23:25:26 jd Exp $

   Created 06/23/04 by Jean-Dominique Frattini <zionarea@free.fr>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file libs/physics/entity.h
  \brief An entity for physics.

TODO:

   * Maybe we could do the collision detection in physics. This is because we
   * will need vertex data for some stuffs here, like knowing the good height
   * (y value from the default opengl system coordinates) of an object that is
   * on another (generally the relief or another object) one - we speak here of
   * scene objects.
*/

#ifndef LEG_LIBS_PHYSICS_ENTITY_H
#define LEG_LIBS_PHYSICS_ENTITY_H

#include "leg/support/maths/maths.h"
#include "leg/support/timing/timing.h"

namespace leg
{
namespace libs
{
//! Physics laws namespace.
/*!
 * All the physics compounds come here. The physics stuff is articulated around
 * a common hierarchy based on Entity. The physics classes are generally not
 * intended to be instanciated, even if they allow polymorphism: the most 
 * common way to use them is to just declare the wish type in a scene::Object.
 *
 * status:
 *    experimental.
 *
 * info:
 *    boundings seem to not behave exactly the same way each time.
 */
namespace physics
{

using leg::support::maths::real;
using leg::support::maths::Quaternion;
using leg::support::timing::Timing;
using leg::support::timing::GameTime;

typedef leg::support::maths::Vector<3> Vector;	      //!< A vector viewed by physics.
typedef leg::support::maths::Point<3>  Point;	      //!< A point viewed by physics.
typedef leg::support::maths::Point<3>  Orientation;   //!< Orientation viewed by physics.
typedef real			       Mass;	      //!< Mass quantities.
typedef real			       Energy;	      //!< Energy (not used yet).

//! Configuration of physics entity movements.
/*!
 * Each physics entity can hold one or more of the choices listed here. Just
 * simply have a look at the comments about each of them to find your best 
 * match of your wishes.
 */
enum MovingConfig
{
   no_move,
   move_follows_velocity,
   move_follows_orientation,
   move_follows_revolution,
   move_follows_field,			  // like gravity and electro-magnetism.
   move_follows_movement_equation	  // like P=(gt)/2+a0t+v0 (this directly affects locality).
};

enum GravityConfig
{
   no_gravity = 1,
   submit_gravity = 2,
   receive_gravity = 4
};

struct SystemForces: public Vector
{
   Vector W;   // Weight
   Vector R;   // support's Reaction
   Vector F1;  // rubbing Forces
   Vector F2;  // external Forces
};

class Entity
{
   protected:

   static real distance_precision;

   Entity (const Entity& e);
   
   Entity();
   
   const Entity&
   operator = (const Entity& e);

   public:
   
   virtual ~Entity();

   virtual void
   SetPosition (Vector& pos) = 0;

   virtual const Vector&
   GetPosition() const = 0;

   virtual void
   SetVelocity (Vector& vel) = 0;

   virtual const Vector&
   GetVelocity () const = 0;

   virtual void
   SetSpeed (real spd) = 0;

   virtual real
   GetSpeed() const = 0;

   virtual void
   SetAcceleration (Vector& accel) = 0;

   virtual const Vector&
   GetAcceleration() const = 0;

   virtual void
   SetAccelerationMagnitude (real magnitude) = 0;

   virtual real
   GetAccelerationMagnitude() = 0;

   virtual GameTime
   GetTime() = 0;

   virtual void
   SetTime (GameTime gt) = 0;

   virtual Orientation
   GetOrientation() const = 0;

   virtual Orientation
   GetOrientationFromVelocity() const = 0;
   
   virtual void
   SetOrientation (Orientation& orientation) = 0;

   virtual void
   SetMass (real m) = 0;

   virtual real
   GetMass() = 0;

   virtual void
   SetRevolutionCenter (Entity& entity) = 0;

   virtual const Entity*
   GetRevolutionCenter() const = 0;

   virtual void
   Update() = 0;

/*   virtual Entity&
   Clone () = 0;
*/
   virtual void
   SetMovingConfig (MovingConfig conf) = 0;

   virtual MovingConfig
   GetMovingConfig() const = 0;

   virtual void
   SetGravityConfig (GravityConfig conf) = 0;

   virtual GravityConfig
   GetGravityConfig() const = 0;
   
   virtual void
   UpdateGravity (Vector& g, real height) = 0;

   virtual void
   SetExternalForce (Vector& f) = 0;

   virtual const Vector&
   GetExternalForce() = 0;

   protected:

   virtual void
   Update (GameTime dt) = 0;

   virtual GameTime
   PreUpdate() = 0;

   virtual void
   PostUpdate() = 0;

   virtual void
   UpdatePosition (GameTime dt) = 0;

   virtual void
   UpdateVelocity (GameTime dt) = 0;

   virtual void
   UpdateAcceleration (GameTime dt) = 0;

   virtual void
   UpdateOrientation (GameTime dt) = 0;

   virtual void
   UpdateVelocityFromOrientation (GameTime dt) = 0;

   virtual void 
   UpdateAngularSpeed (GameTime dt) = 0;

   private:

   static void
   SetDistancePrecision (real val);

   friend void SetDistancePrecision (real);
};

void
SetDistancePrecision (real value);

}
}
}

#endif
