/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: $

   Created 12/08/04 by J-D Frattini <zionarea@free.fr>
   
   Copyright (c) 2004-2005 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file libs/scene/entity.h
  \brief A class representing any entity with a location.
*/

#ifndef LEG_LIBS_SCENE_ENTITY_H
#define LEG_LIBS_SCENE_ENTITY_H

#include <leg/support/maths/maths.h>
#include <leg/libs/physics/physics.h>
#include <leg/support/timing/timing.h>
#include <map>

namespace leg
{
namespace libs
{

//! Scene objects
/*!
 * Provides one hierarchy among several other objects intended to manage a 
 * scene world. You'll find here almost all classes needed to represent game
 * mode objects like cameras, lights, static objects, humanoids...
 *
 * status:
 *    in development.
 *
 * info:
 *    none.
 */
namespace scene
{

using namespace support::maths::trigo;

typedef support::maths::Vector<3>   Vector;
typedef support::maths::Vector<3>   Vec3;
typedef support::maths::Vector<4>   Vec4;

using support::timing::game_time;
using support::timing::Timing;

class WorldInterface;

//! Base polymorphic class for scene hierarchy.
/*!
 * Entity provides mainly pure virtual classes its childhood will have to
 * implement. It also provides several implemented virtual functions any child
 * class should understand in order to avoid mis-executions.
 * This class presents an interface that tries to be suffisant for managing 
 * scene objects.
 * 
 * status:
 *    undefined.
 *
 * info:
 */
class Entity
{   
   public:

   typedef unsigned int			  Propriety; // Property !!!!!!!!
   typedef std::list<std::string>	  Values;
   typedef std::map<std::string,Values>	  ExtraArgs;

   static const Propriety none		     = 0x0;
   static const Propriety moveable_object    = 0x1;
   static const Propriety articulated_object = 0x2;
   static const Propriety dynamic_object     = 0x4;
   static const Propriety light_object	     = 0x8;
   static const Propriety camera_object	     = 0x10;
   
   //
   // ! WARNING !
   //
   // You MUST redefine those two declarations for each class you will inherit
   // in this hierarchy. You also MUST assign them with the appropriate 
   // values. Otherwise, expect data corruption on loading and saving.
   // 
   
   static const std::string main_id;
   static const std::string sub_id;

   protected:

   static Timing     timer;
   static game_time  u_clock;		 //! universal clock
   static game_time  u_dt;		 //! universtal differential time (delta t)

   Vector      pos;		 //! position
   Vector      ori;		 //! orientation (should define the policy)
   Vector      size;		 //! maximum size
   std::string model_filename;	 //! filename of the corresponding model.
   std::string name;		 //! name of the object.

   ExtraArgs   args;

   Entity();

   Entity (const Entity& e);

   public:
   
   virtual ~Entity();

   const Entity&
   operator = (const Entity& e);

    //
   // Main Interface
    // 
   
   //! Initialization.
   virtual void
   Init() = 0;

   virtual const Propriety
   GetPropriety() const = 0;

   //! Update
   /*!
    * Update makes the object to live. This function should be called as much
    * often as possible. On some objects this function does nothing (almost for
    * simple static objects) or maybe update some scene info (collisions...).
    */
   virtual void
   Update() = 0;

   //! Cloning.
   /*!
    * Cloning is involved when you're not sure about the dynamic type of an 
    * object and you still want to make a copy of it (and thus all the data).
    */
   virtual Entity*
   Clone() = 0;
   
   //! Reading
   /*!
    * Loads information from a flow and fills in the object with them. This is
    * intended for the loading process only and will certainly be called from
    * any scene engine.
    */
   virtual void
   Read (std::ifstream& ifs);

   //! Writing
   /*!
    * Save the object's information into a flow, generally a file. This is
    * intended for saving object's status during a game save for example.
    */
   virtual void
   Write (std::ofstream& ofs);

    //
   // some get/set interface
    //
   
   //! Position setting.
   void
   SetPosition (const Vector& p);

   //! Getting position.
   Vector&
   GetPosition();

   //! Orientation setting.
   void
   SetOrientation (const Vector& o);

   //! Getting orientation.
   Vector&
   GetOrientation();

   //! Size setting.
   void
   SetSize (const Vector& s);

   //! Getting size.
   Vector&
   GetSize();

   //! Model filename setting.
   /*!
    * This only set the filename to the given argument, but this does not load
    * the model.
    */
   void
   SetModelFilename (const std::string& s);

   //! Getting the model filename.
   const std::string&
   GetModelFilename();

   inline void
   SetName (const std::string& n)
   {
      name = n;
   }
   
   /*
    * Those two functions must be redeclared and reimplemented in all the 
    * classes from this hierarchy, otherwise there will have type and data
    * corruption during loading and saving time.
    */
   
   //! Getting the main id.
   virtual const std::string
   GetMainId()
   {
      return main_id;
   }

   //! Getting the sub id.
   virtual const std::string
   GetSubId()
   {
      return sub_id;
   }

   ExtraArgs&
   GetArgs()
   {
      return args;
   }
   
   protected:

   //! Make a copy of the object.
   virtual void
   Copy (const Entity& e);

   void
   ReadExtra (std::ifstream& ifs);

   void
   WriteExtra (std::ofstream& ofs);
   
   private:

   friend class WorldInterface;
};

}
}
}

#endif // LEG_LIBS_SCENE_ENTITY_H
