/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: actorpolicy.h,v 1.2 2004/06/06 02:09:01 jd Exp $
 
   Created 06/02/04 by Jean-Dominique Frattini <zionarea@free.fr>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*!\file libs/threads/actorpolicy.h
  \brief state acting policy for libs::threads.
*/

#ifndef LEG_LIBS_THREADS_ACTOR_POLICY_H
#define LEG_LIBS_THREADS_ACTOR_POLICY_H

namespace leg
{
namespace libs
{
namespace threads
{

//! State actor policy.
/*!
 * This is the default state actor policy used in Leg for handling thread 
 * states. This is not advised to change this behavior.
 *
 * template arguments description:
 *    * Input is the input.
 *    * State is the state.
 *    * Sync is the threading sync (Mutex inherited class).
 *
 * status:
 *    in progress.
 *
 * info:
 *    none.
 *
 * @sa DefaultSettingPolicy, leg::support::states::StateMachine, 
 *    leg::support::states::DefaultStateActorPolicy
 */
template
<
   class Input,
   typename State,
   class Sync
>
class ActorPolicy
{
   protected:

   Sync state_mutex; //!< state thread synchronizer.
   
   public:
   
   ActorPolicy ()
   {
   }
   
   //! Act from an input.
   /*!
    * Change the states regarding the inputs (and current state).
    */
   void
   Act (const Input& input, State& state)
   {
      state_mutex.Lock();
      
      unsigned int isize = input.size();
      
      if (input.find (input::destructed) < isize){
	 state = 0;
	 state_mutex.Unlock();
	 return;
      }
      if ((input.find (input::constructed) < isize)){
	 Activate (state,state::constructed);
      }
      if ((input.find (input::configured) < isize)){
	 Activate (state,state::configured);
      }
      if ((input.find (input::created) < isize)){
	 Activate (state,state::created);
      }
      if ((input.find (input::locked) < isize)){
	 Activate (state,state::locked);
      }
      if ((input.find (input::unlocked) < isize)){
	 Deactivate (state,state::locked);
      }
      if ((input.find (input::loaded) < isize)){
	 Activate (state,state::loaded);
      }
      if ((input.find (input::cancelable) < isize)){
	 Activate (state,state::cancelable);
      }
      if ((input.find (input::not_cancelable) < isize)){
	 Deactivate (state,state::cancelable);
      }
      if ((input.find (input::in_wait) < isize)){
	 Activate (state,state::in_wait);
      }
      if ((input.find (input::ready) < isize)){
	 if (!(state & state::ready))
	    Activate (state,state::ready);
	 if (state & state::in_wait)
	    state ^= state::in_wait;
      }
      if ((input.find (input::run) < isize)){
	 Activate (state,state::running);
      }
      if ((input.find (input::finish) < isize)){
	 Activate (state,state::ending);
      }
      if ((input.find (input::ended) < isize)){
	 Activate (state,state::finished);
	 if (state & state::running)
	    state ^= state::running;
      }

      state_mutex.Unlock();
   }

   protected:
   
   inline Sync&
   GetStateMutex()
   {
      return state_mutex;
   }
   
   private:

   inline void
   Activate (State& state, const State& value)
   {
      if (!(state & value))
	 state |= value;
   }

   inline void
   Deactivate (State& state, const State& value)
   {
      if (state & value)
	 state ^= value;
   }
};

}
}
}
#endif
