/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: states.h,v 1.1 2004/06/03 23:03:23 jd Exp $

   Created 06/02/04 by Jean-Dominique Frattini <zionarea@free.fr>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*!\file libs/threads/states.h
  \brief states for libs::window.
*/

#ifndef LEG_LIBS_THREADS_STATES_H
#define LEG_LIBS_THREADS_STATES_H

namespace leg
{
namespace libs
{
namespace threads
{

//! A thread state is unsigned int.
typedef unsigned int State;

//! All valid states for libs::threads threads.
/*!
 * ThreaderCell uses those values to representate its state. Always use the
 * variable names, but not their values, that could change during the
 * development time.
 * For users, states could be well to have information about how is a thread at
 * a certain time; however this is more dedicated to the leg's programmers.
 */
namespace state
{

using libs::threads::State;

//! undefined state.
/*!
 * This state is only for starting point, and should never stay at this value.
 */
static const State undefined	 = 0x0;

//! Constructed state.
/*!
 * When a staty-thread has been succesfully constructed, its state can be set
 * into this value.
 */
static const State constructed	 = 0x1;

//! Configured state.
/*!
 * Threads configuration could depend on user's code. This bit should be set
 * when the thread has been fully configured.
 */
static const State configured	 = 0x2;

//! Created bit.
/*!
 * When the thread has been created, this bit is activated. When the thread has
 * been destructed, this bit is deactivated.
 */
static const State created	 = 0x4;

//! Loaded bit.
/*!
 * Loading a thread will create then run the thread. This bit is set to 1 when
 * it is well loaded (just before waiting).
 */
static const State loaded	 = 0x8;

//! Locked state (thread sync).
/*!
 * When the thread is locked, this bit is set to 1, otherwise, it would mean the
 * thread is unlocked. However, nothing could really tells this is real all the
 * time, but this is more intended to be as an indication. Indeed, locking and
 * unlocking are from the thread synchronization responsability, that has no
 * link with states [almost for conveniencies].
 */
static const State locked	 = 0x10;

//! Waiting bit.
/*!
 * ThreaderCell comes into a waiting condition, waiting a matching signal and
 * value(s) set(s) just before running. It activates this state until it 
 * receives the matching signal, calling ready, then deactivate this state.
 */
static const State in_wait	 = 0x20;

//! Ready bit.
/*!
 * When the thread has received the matching signal to the waiting condition,
 * it must receive just after a state activation of the ready value. This is as
 * important as threads will block if this state isn't activated just after the
 * signal has been send. When this bit is enabled, the thread will run.
 */
static const State ready	 = 0x40;

//! Running bit.
/*!
 * A thread activates this bit while it is running.
 */
static const State running	 = 0x80;

//! Cancelable bit.
/*!
 * If the thread is cancelable, this thread is activated, otherwise it is 
 * deactivated.
 */
static const State cancelable	 = 0x100;

//! Paused bit.
/*!
 * If the thread is paused (Leg's software thread pause), this bit should be
 * activated.
 *
 * status:
 *    Not implemented.
 */
static const State paused	 = 0x200;

//! Ending bit.
/*!
 * If the thread is going to end, this bit is set to 1.
 */
static const State ending	 = 0x400;

//! Finished bit.
/*!
 * When the thread has gone out of the runing code, this bit is set to 1.
 */
static const State finished	 = 0x800;

}

}
}
}
#endif
