/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: sync.h,v 1.4 2004/06/03 23:03:23 jd Exp $
 
   Created 05/21/04 by Jean-Dominique Frattini <zionarea@free.fr>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*!\file libs/threads/sync.h
  \brief thread synchronization and condition.
*/

#ifndef LEG_LIBS_THREADS_SYNC_H
#define LEG_LIBS_THREADS_SYNC_H

#include "leg/support/threads/condmutex.h"

namespace leg
{
namespace libs
{
namespace threads
{

struct Syncless{};

template
<
   class TSync = support::threads::Mutex
>
class Sync;

//! Synchronization tool.
/*!
 * Generic class for synchronization.
 * You'll surely never have to change the default template argument, or
 * provide a completly compliancy class instead, with ensuring copies
 * of your own types are well done threw copy constructors. This is really
 * important for some manipulations in this layer, like allowing several
 * instances of ThreaderCell to lock the same mutex.
 *
 * status:
 *    does not layer the mutex anymore, for best performances.
 *
 * info:
 *    none.
 *
 * @sa support::threads::Condition.
 */
template<>
class Sync<support::threads::Mutex> : public support::threads::Mutex
{
   typedef support::threads::Mutex  Parent;

   public:

   Sync(): Parent()
   {
   }

   Sync (const Sync& s): Parent (static_cast<const Parent&> (s))
   {
   }

   ~Sync()
   {
   }

   inline const Sync&
   operator = (const Sync& s)
   {
      Parent::operator = (static_cast<const Parent&> (s));
      return *this;
   }

   inline const Parent&
   GetMutex() const
   {
      return *this;
   }
};

template
<
   class TCond = support::threads::Condition
>
class Cond;

//! Generic condition template class.
/*!
 * A simple class definition for supporting Posix thread condition.
 *
 * status:
 *    does not layer the condition class anymore for best performances.
 *
 * info:
 *    none.
 *
 * @sa none.
 */
template<>
class Cond<support::threads::Condition>: public support::threads::Condition
{
   typedef support::threads::Condition Parent;
   
   public:

   Cond (): Parent()
   {
   }

   Cond (const Cond& c): Parent (static_cast<const Parent&> (c))
   {
   }

   Cond (const Parent& c): Parent (c)
   {
   }

   inline const Cond&
   operator = (const Cond& c)
   {
      Parent::operator = (static_cast<const Parent&> (c));
      return *this;
   }

   inline const Cond&
   Get() const
   {
      return *this;
   }

   private:

   using Parent::Wait;

   public:

   //! Waiting for a signal related to the mutex given in argument.
   /*!
    * Waiting for a signal involves a relation with a Sync class (generally
    * Mutex). It is given as a pure generic class here, but this has to be
    * a real synchronization class (like Mutex).
    * If you want to get rid of this argument (this is the default behavior
    * under Leg), you will need another (template) class: SyncCond or
    * CondMutex.
    *
    * @sa support::threads::Condition,SyncCond.
    */
   template <class Sync>
   inline void
   Wait (Sync& s)
   {
      Parent::Wait (s);
   }
};

// This class name should be changed...
template
<
   class TSyncCond = support::threads::CondMutex
>
class SyncCond
{
   protected:

   TSyncCond cond_mutex;
   
   public:

   SyncCond()
   {
   }

   SyncCond (const SyncCond<TSyncCond>& cm): cond_mutex (cm.cond_mutex)
   {
   }

   SyncCond (const TSyncCond& sc): cond_mutex (cond_mutex)
   {
   }

   const SyncCond<TSyncCond>&
   operator = (const SyncCond<TSyncCond>& cm)
   {
      cond_mutex = cm.cond_mutex;
   }

   inline void
   Lock()
   {
      cond_mutex.Lock();
   }

   inline void
   Unlock()
   {
      cond_mutex.Unlock();
   }

   inline void
   Signal()
   {
      cond_mutex.Signal();
   }

   inline void
   Wait()
   {
      cond_mutex.Wait();
   }

   const TSyncCond&
   GetMutex() const
   {
      return cond_mutex;
   }
};

struct NoSync
{
   void
   Lock(){}

   void
   Unlock(){}
};

struct NoCond
{
   void
   Signal(){}

   void
   Wait(){}
};

struct NoSyncNoCond: public NoSync,NoCond
{
};

}
}
}
#endif
