/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: $

   Created 05/14/04 by Jean-Dominique Frattini <zionarea@free.fr>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*!\file leg/libs/window/actorpolicy.h
  \brief state acting policy for libs::window.
*/

#ifndef LEG_LIBS_WINDOW_ACTOR_POLICY_H
#define LEG_LIBS_WINDOW_ACTOR_POLICY_H

namespace leg
{
namespace libs
{
namespace window
{

template
<
   class Input,
   typename State,
   class Sync
>
class ActorPolicy
{
   protected:

   Sync state_mutex; //!< state thread synchronizer.
   
   public:
   
   ActorPolicy ()
   {
   }
   
   //! Act from an input.
   /*!
    * Change the states regarding the inputs (and current state).
    */
   void
   Act (const Input& input, State& state)
   {
      state_mutex.Lock();
      
      unsigned int isize = input.size();
      
      if (input.find (input::destruct) < isize){
	 state = 0;
	 state_mutex.Unlock();
	 return;
      }

      if ((input.find (input::construct) < isize)){
	 Activate (state,state::constructed);
      }
      
      if ((input.find (input::configure) < isize)){
	 Activate (state,state::configured);
      }
      
      if ((input.find (input::create) < isize)){
	 Activate (state,state::created);
      }
      if ((input.find (input::destroy) < isize)){
	 Deactivate (state,state::created);
      }
      
      if ((input.find (input::lock) < isize)){
	 Activate (state,state::locked);
      }
      if ((input.find (input::unlock) < isize)){
	 Deactivate (state,state::locked);
      }
      
      if ((input.find (input::lock_draw) < isize)){
	 Activate (state,state::drawable_locked);
      }
      if ((input.find (input::unlock_draw) < isize)){
	 Deactivate (state,state::drawable_locked);
      }
      
      if ((input.find (input::drawable) < isize)){
	 Activate (state,state::drawable);
	 Deactivate (state,state::hud_drawable);
      }
      if ((input.find (input::refresh) < isize)){
	 Deactivate (state,state::drawable);
      }
      if ((input.find (input::refreshed) < isize)){
	 Activate (state,state::drawable);
      }
      
      if ((input.find (input::resize) < isize)){
	 Deactivate (state,state::sized);
      }
      if ((input.find (input::sized) < isize)){
	 Activate (state,state::sized);
      }
      
      if ((input.find (input::redraw) < isize)){
	 Deactivate (state,state::drawing);
      }
      
      if ((input.find (input::hud_drawable) < isize)){
	 Activate (state,state::hud_drawable);
	 Deactivate (state,state::drawable);
      }
      if ((input.find (input::hud_undrawable) < isize)){
	 Deactivate (state,state::hud_drawable);
      }

      //std::cout << "win state: " << state << std::endl;

      state_mutex.Unlock();
   }

   Sync&
   GetStateMutex()
   {
      return state_mutex;
   }

   private:

   void
   Activate (State& state, const State& value)
   {
      if (!(state & value))
	 state |= value;
   }

   void
   Deactivate (State& state, const State& value)
   {
      if (state & value)
	 state ^= value;
   }
};

}
}
}
#endif
