/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: $

   Created 09/10/04 by Jean-Dominique Frattini <zionarea@free.fr>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file libs/window/osd.h
  \brief generic on-screen display for gl text outputs.
*/

#ifndef LEG_LIBS_WINDOW_OSD_H
#define LEG_LIBS_WINDOW_OSD_H

#include "leg/support/window/window.h" // we have no way to find the specific file for the moment.
#include "leg/libs/threads/sync.h"

namespace leg
{
namespace libs
{
namespace window
{

template
<
   class TOsd = support::window::Osd,
   class TSync = libs::threads::Sync<>
>
class GenericOsd;

template
<
>
class GenericOsd< support::window::Osd,libs::threads::Sync<> >: public support::window::Osd
{
   protected:

   typedef support::window::Osd	 Parent;
   typedef libs::threads::Sync<> Sync;

   static Sync sync; // We'd better inherit from Sync to avoid declaring sync in a source that does not exist !!

   GenericOsd():  Parent()
   {
   }

   ~GenericOsd()
   {
   }

   public:

   //! Font iterator to get threw patterned_fonts.
   /*!
    * When CheckFontsAvailability() as been invoked and succesfully executed,
    * users will be able to manipulate the available matching found fonts
    * threw a FontIterator that will be return by a call to GetFontIterator().
    */
   typedef Parent::FontIterator FontIterator;

   //! Connect to the display.
   /*!
    * You must be connected to the display before being available to use all 
    * the other functions of this class. Simply giving a pointer to a valid
    * connection does the work.
    */
   inline static void
   Connect (support::window::internal::Display *d)
   {
      sync.Lock();
      Parent::Connect (d);
      sync.Unlock();
   }

   //! Load a font and select it as the current.
   /*!
    * Load a font from the name given in argument and select it as the current
    * font for writting into the window with Print(). Selects the font as the
    * current if the font has already been loaded. If no argument has been 
    * given, then, the "fixed" font is used as the default.
    */
   inline static void
   LoadFont (std::string font_name = "fixed")
   {
      sync.Lock();
      Parent::LoadFont (font_name);
      sync.Unlock();
   }

   //! Load the matched fonts saved into patterned_fonts.
   /*!
    * Use CheckFontsAvailability() before in order to fill the list. This will
    * load all the fonts in the patterned_fonts list, and select the last one
    * as the current font for printing.
    */
   inline static void
   LoadMatchedFonts()
   {
      sync.Lock();
      Parent::LoadMatchedFonts();
      sync.Unlock();
   }

   //! Unload a font, giving its string name.
   /*!
    * Unload the font having the same name as given in argument. This let the
    * state of the current font undefined if the font was current just before
    * calling this function.
    */
   inline static void
   UnloadFont (const std::string& font_name)
   {
      sync.Lock();
      Parent::UnloadFont (font_name);
      sync.Unlock();
   }

   //! Select an already loaded font as the current font.
   /*!
    * Does nothing if the font has not been loaded.
    */
   inline static void
   SelectFont (const std::string& font_name)
   {
      sync.Lock();
      Parent::SelectFont (font_name);
      sync.Unlock();
   }

//   inline static void
//   SelectFont (FontIterator font_iterator);
   
   //! Locate into the screen (in order to print after generally).
   /*!
    * Note: 
    *	 Generally, the origin is located on the bottom-left corner, so you may
    *	 want to put it on the upper-left corner and ordinate from up to down.
    */
   inline static void
   Locate (int x, int y, int z = 0)
   {
      sync.Lock();
      Parent::Locate (x,y,z);
      sync.Unlock();
   }

   //! Locate into the screen (in order to print) with float values.
   inline static void
   Locate (float x, float y, float z = .0)
   {
      sync.Lock();
      Parent::Locate (x,y,z);
      sync.Unlock();
   }

   //! Locate into the screen (in order to print) with double values.
   inline static void
   Locate (double x, double y, double z = .0)
   {
      sync.Lock();
      Parent::Locate (x,y,z);
      sync.Unlock();
   }

   //! Check fonts availability from a given pattern.
   /*!
    * This fills up the patterned_fonts list with available fonts that match 
    * the given pattern (you can use '*' and '?' to makes some unknowns). Not
    * all the matchings fonts will be retrived, but only max_elements at the
    * maximum. You can get and change how many with using the appropriate 
    * functions declared below.
    * 
    * Please have a look at the matching X documentation for more information.
    */
   inline static bool
   CheckFontsAvailability (const std::string& pattern)
   {
      sync.Lock();
      bool r;
      r = Parent::CheckFontsAvailability (pattern);
      sync.Unlock();
      return r;
   }

   //! Get a FontIterator at the begining of patterned_fonts list.
   /*!
    * This can be usefull to go threw the patterned_fonts list.
    */
   inline static FontIterator
   GetFontIterator()
   {
      sync.Lock();
      FontIterator r;
      r = Parent::GetFontIterator();
      sync.Unlock();
      return r;
   }

   //! Get the size of the patterned_fonts list.
   /*!
    * Return how much elements are stored into the patterned_fonts list.
    */
   inline static unsigned int
   GetPatternedFontsSize()
   {
      sync.Lock();
      unsigned int r;
      r = Parent::GetPatternedFontsSize();
      sync.Unlock();
      return r;
   }

   //! Print a string at the previous providen location with Locate().
   /*!
    * Prints out the string on the window (gl 2D projection matrix). If the
    * location hasn't been defined, then where the string will be print out on
    * the screen is undefined.
    *
    * @sa Locate().
    */
   inline static void
   Print (const std::string& str)
   {
      sync.Lock();
      Parent::Print (str);
      sync.Unlock();
   }

   //! Set how many max elements could be retrived.
   /*!
    * Set how much maximum elements could be retrieved while checking the fonts
    * availability regarding a pattern (CheckFontsAvailability()).
    *
    * @sa max_elements, CheckFontsAvailability.
    */
   inline static void
   SetMaxElementsInPatternedFonts (unsigned int max)
   {
      sync.Lock();
      Parent::SetMaxElementsInPatternedFonts (max);
      sync.Unlock();
   }

   //! Get how many max elements could be retrieved.
   /*!
    * Return how many maximum elements will be retrived while making a research
    * of fonts regarding a pattern.
    *
    * @sa GetMaxElementsInPatternedFonts().
    */
   inline static unsigned int
   GetMaxElementsInPatternedFonts()
   {
      sync.Lock();
      unsigned int r;
      r = Parent::GetMaxElementsInPatternedFonts();
      sync.Unlock();
      return r;
   }

   protected:
   
   //! Returns true if the font has already been loaded and not deleted.
   inline static bool
   FontAlreadyLoaded (const std::string& str)
   {
      sync.Lock();
      bool r;
      r = Parent::FontAlreadyLoaded (str);
      sync.Unlock();
      return r;
   }

   //! Delete all the loaded list.
   /*!
    * Delete all loaded fonts of X, Glx and free the list. There will have no 
    * more fonts available after this call, so SelectFont() and Print() will 
    * become undefined.
    */
   inline static void
   DeleteList()
   {
      sync.Lock();
      Parent::DeleteList();
      sync.Unlock();
   }
};

/*GenericOsd< support::window::Osd,libs::threads::Sync<> >::Sync 
GenericOsd< support::window::Osd,libs::threads::Sync<> >::sync;
*/
// we should find a better way to make Osd (other Osd<> types won't work with
// the leg library that uses Osd as the default type).
typedef GenericOsd<> Osd;

}
}
}
#endif
