/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: objectlevellockable.h,v 1.1 2004/05/06 07:29:54 jd Exp $
   $Log: objectlevellockable.h,v $
   Revision 1.1  2004/05/06 07:29:54  jd
   ObjectLevelLockable for Leg



   Created 03/25/04 by Jean-Dominique Frattini <zionarea@free.fr>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file support/loki/objectlevellockable.h
  \brief object level lockable sync.
*/

#ifndef LEG_SUPPORT_LOKI_OBJECT_LEVEL_LOCKABLE_H
#define LEG_SUPPORT_LOKI_OBJECT_LEVEL_LOCKABLE_H

#include "mutex.h"

namespace leg
{
namespace support
{
//! Loki's overlay for Leg.
/*!
 * This redefine some Loki's tools for leg.
 */
namespace loki
{

//! Leg's loki's ObjectLevelLockable redefinition.
/*!
 * Out object level lockable scheme. This has a compatible interface with 
 * loki's one. Use this template or inherit from it in order to have classes 
 * that are object thread safe, or to make some basic atomic operations as 
 * adding or setting. 
 * We add a new template argument for designing the type of the 
 * synchronization for threads defaulting to our Mutex. This permits to stay 
 * generic and compatible with loki. We may have could put the type hardly but
 * this won't allow other sync tools to work with.
 *
 * status:
 *    in implementation.
 *
 * info:
 *    Quiete unacurate for atomic ops because:
 *	 we cannot provide one mutex for each int values given.
 *	 we cannot propagate this mutex to other threads.
 *
 *    Object level lockable is a good scheme when we don't know how the things
 *    will work in the whole (surely W32 env ?). Under Unix (Linux), we are 
 *    the master of all the threads representing the process. So, we're able 
 *    to have a quiete different design: we may lock only when it's needed, so
 *    avoiding unnecessary cpu consumptions. Conclusion: this scheme may not 
 *    be appropriate when many locks/unlocks will be called threw several
 *    threads. We'll prefer then to higher-level decision means.
 *    
 * @sa loki::ObjectLevelLockable
 */
template
<
   class Host,
   class Sync= Mutex
>
class ObjectLevelLockable
{
   Sync mutex;	  //!< multithread synchonization tool (mutex).
   
   public:
   
   class Lock;
   friend class Lock;
   
   //! Locking capabilities.
   /*!
    * Construct an object of a type whenever you want your host object to be 
    * locked (protected from other threads). This will end up when the life 
    * of this created object takes end. So, you'll need to provoke its
    * destruction if you want to unlock.
    */
   class Lock
   {
      ObjectLevelLockable& host;
      
      Lock();
      Lock (const Lock&);
      Lock& operator= (const Lock&);
      
      public:
      
      //! only constructor needing a reference to the 'master class' host.
      /*!
       * This locks the mutex.
       */
      explicit Lock (ObjectLevelLockable& h): host (h)
      {
	 host.mutex.Lock();
      }
      
      //! destroys the locking object, so unlocking.
      ~Lock()
      {
	 host.mutex.Unlock();
      }
   };
   
   typedef volatile Host VolatileType;
   typedef int IntType;
   
   //! Atomic incrementation.
   /*!
    * Atomic operations are not well done.
    * To be good, we'll need to have a static dynamic 2D array with mutex and
    * the addresses of all int types given. Then, regarding this address we'll
    * be able to lock effectively the good mutex. However, this will require a
    * lot of memory if many int calls are used. Finally, this is, actually, the
    * only solution I found to go threw this issue.
    */
   static IntType
   AtomicIncrement (volatile IntType& lval)
   {
      volatile IntType i= lval;
      i=i+1;
      lval=i;
      
      return lval;
   }
   
   static IntType
   AtomicDecrement (volatile IntType& lval)
   { 
      volatile IntType i= lval;
      i=i-1;
      lval=i;

      return const_cast<IntType&>(lval);
   }
   
   static void
   AtomicAssign (volatile IntType& lval, IntType val)
   { 
      volatile IntType i= lval;
      lval= const_cast<volatile IntType> (val);
   }
   
   static void
   AtomicAssign (IntType& lval, volatile IntType& val)
   { 
      volatile IntType i= val;
      val= lval;
      lval= i;
   }
};

}// namespace threads
}// namespace support
}// namespace leg

#endif
