/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: polynomial.h,v 1.3 2004/03/22 21:19:02 jechk Exp $
   $Log: polynomial.h,v $
   Revision 1.3  2004/03/22 21:19:02  jechk
   Fixed compilation issue in maths; added include/Makefile.in so it is not necessary to run automake.

   Revision 1.2  2004/03/09 03:54:57  jechk
   Made support::timing::Timer::timing private, minor changes.

   Revision 1.1  2004/03/08 23:07:07  jechk
   Added polynomial files.



   Created 3/6/04 by Jeff Binder <bindej@rpi.edu>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file leg/support/maths/polynomial.h
  \brief A class representing nth-order polynomials.
*/

#include <leg/support/maths/real.h>

#include <vector>

#ifndef LEG_SUPPORT_MATHS_POLYNOMIAL_H
#define LEG_SUPPORT_MATHS_POLYNOMIAL_H

namespace leg
{

namespace support
{

namespace maths
{

//! A class to represent arbitrary-order polynomials.
class Polynomial
{
 public:
  //! Creates the polynomial `0.0'.
  Polynomial ();

  //! Creates a polynomial of the specified order.
  /*!
   * order + 1 coefficients must be supplied; they are in order from
   * the constant term to the x^order term.
   */
  Polynomial (unsigned int order, ...); // TODO: Is this a good idea?

  //! Creates a polynomial with the specified coefficents.
  /*!
   * Coefficients are in order from the constant term to the x^order
   * term.
   */
  Polynomial (std::vector<real> coeff);

  //! Copy constructor.
  Polynomial (const Polynomial &p);

  ~Polynomial ();

  //! Returns the coefficient of the x^n term of the polynomial.
  /*!
   * It is safe to call this with any value of n regardless of the
   * polynomial's order.
   */
  real
  operator [] (unsigned int n) const
  {
    return n > order? 0.0: coeff[n];
  }

  //! Evaluates the polynomial at x.
  real operator () (real x) const;

  //! Adds the specified amount to the constant term.
  Polynomial operator + (real x) const
  {
    Polynomial n (*this);

    n.SetCoefficient (0, n[0] + x);

    return n;
  }

  //! Adds two polynomials.
  Polynomial operator + (const Polynomial &x) const
  {
    Polynomial n (*this);

    for (unsigned int i = 0; i < order; ++i)
      {
	n.SetCoefficient (i, n[i] + x[i]);
      }

    return n;
  }

  //! Returns -1 times the polynomial.
  Polynomial operator - () const
  {
    Polynomial n (*this);

    n.SetCoefficient (0, -n[0]);

    return n;
  }

  //! Subtracts the specified amount from the constant term.
  Polynomial operator - (real x) const
  {
    Polynomial n (*this);

    n.SetCoefficient (0, n[0] - x);

    return n;
  }

  //! Subtracts two polynomials.
  Polynomial operator - (const Polynomial &x) const
  {
    Polynomial n (*this);

    for (unsigned int i = 0; i < order; ++i)
      {
	n.SetCoefficient (i, n[i] - x[i]);
      }

    return n;
  }

  //! Multiplies a polynomial by a constant.
  Polynomial operator * (real x) const
  {
    Polynomial n (*this);

    for (unsigned int i = 0; i < order; ++i)
      {
	n.SetCoefficient (i, n[i] * x);
      }

    return n;
  }

  //! Multiplies two polynomials.
  Polynomial operator * (const Polynomial &x) const
  {
    Polynomial n;

    for (unsigned int i = 0; i < order; ++i)
      {
	for (unsigned int j = 0; j < x.order; ++j)
	  {
	    n.SetCoefficient (i+j, n[i+j] + coeff[i] * x[j]);
	  }
      }

    return n;
  }

  //! Divides a polynomial by a constant.
  Polynomial operator / (real x) const
  {
    Polynomial n (*this);

    for (unsigned int i = 0; i < order; ++i)
      {
	n.SetCoefficient (i, n[i] * x);
      }

    return n;
  }

  // TODO:  Quotient & modulus?

  //! Sets the coefficient for the specified term.
  /*!
   * The order of the polynomial is adjusted if need be.
   */
  void
  SetCoefficient (unsigned int term, real coefficient)
  {
    if (coefficient)
      {
	while (term > order)
	  {
	    coeff.push_back (0.0);
	    ++order;
	  }
	coeff[term] = coefficient;
      }
  }

  //! Returns the order of the polynomial.
  unsigned int
  Order () const
  {
    return order;
  }

  //! Returns the nth order derivative of the polynomial.
  Polynomial Derivative (unsigned int n = 1) const;

  //! Calculates the nth antiderivative (indefinite integral).
  /*!
   * The constants of integration are all assumed to be 0.
   */
  Polynomial Antiderivative (unsigned int n = 1) const;

  //! Calculates the definite integral over the range [a, b].
  real Integral (real a, real b) const;

  //! Calculates the roots of the polynomial to within epsilon.
  /*!
   * Not yet implemented---what should we do about complex roots?
   */
  std::vector<real> Roots (real epsilon = real_epsilon);

 private:
  void CalcOrder ();

  std::vector<real> coeff;
  unsigned int order;
};

std::ostream &operator << (std::ostream &s, Polynomial p);

}

}

}

#endif // LEG_SUPPORT_MATHS_POLYNOMIAL_H
