/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: $

   Created 07/28/04 by Jean-Dominique Frattini <zionarea@free.fr>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file support/model/entity.h
  \brief entity class for models.
*/

#ifndef LEG_SUPPORT_MODEL_ENTITY_H
#define LEG_SUPPORT_MODEL_ENTITY_H

#include "leg/support/maths/vector.h"
#include <string>

namespace leg
{
namespace support
{

//! 3D model and map area.
/*!
 * Provides tools to manipulate 3D model files and later, game maps too. This
 * namespace defines several means to represent a 3D model, a 3D map, and to
 * be able to load them threw several supported format (for the moment, only 
 * 3D studio files). The entity's model hierarchy is intended to be transformed
 * before being able to be used for rendering.
 *
 * status:
 *    experimenting.
 *
 * info:
 *    Needs approval.
 */
namespace model
{

//! Entity based class for model representation.
/*!
 * Never use polymorphism with this hierarchy (at least for the moment).
 *
 * status:
 *    experimenting.
 *
 * info:
 *    none.
 *
 * @sa Model,Map.
 */
class Entity
{
   public:
   
   //! Kind of model.
   /*!
    * Any model must have this type and an associated value.
    */
   enum Kind
   {
      //! Entity will represent a model.
      /*!
       * A model is only a single data structure to represent a complete 3D 
       * object as seenable during the game (or at least modeling).
       */
      model,

      //! Entity will represent a map.
      /*!
       * A map is a model owning other models in order to represent the game 
       * map (or world). Only 'living' objects like players don't enter a map.
       */
      map
   };

   // for what purpose it is intended to be used for ?
   enum MeshType
   {
      static_mesh,
      semistatic_mesh,
      dynamic_mesh,
      animated_mesh
   };
   
   // deprecated
   
   typedef unsigned int Type;

   static const Type undefined	       = 0x0;
   static const Type normal	       = 0x1;
   static const Type ground	       = 0x2;
   static const Type cast_shadow       = 0x4;
   static const Type receive_shadow    = 0x8;
   static const Type cast_reflexion    = 0x10;
   static const Type receive_reflexion = 0x20;
   static const Type solid	       = 0x40; // solid if interact with collisions
   static const Type camera	       = 0x80; // some game styles may require a predefined camera
   static const Type light	       = 0x100;

   enum VisibilityTest
   {
      do_not_test,
      do_test
   };

   enum Bounding
   {
      none,
      box,
      sphere
   };
   
   protected:

   std::string		      name;	  //! Name of the entity.
   support::maths::Vector<3>  pos;	  //! Position.
   support::maths::Point<3>   ori;	  //! Orientation.
   support::maths::Vector<3>  size;	  //! Size.
   Kind			      kind;	  //! Kind of model.
   Type			      type;	  //! type of model regarding the world
   MeshType		      mesh_type;  //! type of mesh (static data or dynamic data).
   VisibilityTest	      v_test;
   Bounding		      bounding;
   support::maths::Vector<3>  min_v;
   support::maths::Vector<3>  max_v;
   
   //! Default constructor.
   /*!
    * This defaults the name to "undefined".
    */
   Entity();

   //! Copy constructor.
   /*!
    * status:
    *	 not useable yet.
    */
   Entity (const Entity& me);

   public:
   
   virtual ~Entity();

   const Entity&
   operator = (const Entity& me);

   //! Set the name.
   void
   SetName (const std::string& n);

   inline Type
   GetType() const
   {
      return type;
   }

   inline MeshType
   GetMeshType() const
   {
      return mesh_type;
   }

   //! Make a display list useable within OpenGL.
   /*!
    * A model or a map will have the ability to make its own OpenGL display list
    * from the data it contains. Even if display lists are useable inside any
    * games, it's not advised to try to do so: they are here in order to help
    * the development of supporting 3D format and to display easily their
    * appearence (i.e threw a modeler).
    * You'd better use the Maja-Lucida renderer engine instead.
    */
   virtual unsigned int
   MakeDisplayList() = 0;

   inline const support::maths::Vector<3>&
   GetPos() const
   {
      return pos;
   }

   inline const support::maths::Vector<3>&
   GetOrientation() const
   {
      return ori;
   }

   inline const support::maths::Vector<3>&
   GetSize() const
   {
      return size;
   }

   inline void
   SetSize (support::maths::Vector<3>& s)
   {
      size = s;
   }

   inline void
   GetMinMax (support::maths::Vector<3>& min, support::maths::Vector<3>& max) const
   {
      min = min_v;
      max = max_v;
   }

   inline void
   SetMinMax (const support::maths::Vector<3>& min, const support::maths::Vector<3>& max)
   {
      min_v = min;
      max_v = max;
   }

   inline VisibilityTest
   GetVisibilityTest() const
   {
      return v_test;
   }

   inline Bounding
   GetBounding() const
   {
      return bounding;
   }

   protected:

   void
   Copy (const Entity& me);
};

}
}
}

#endif
