/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: statemachine.h,v 1.5 2004/06/09 20:16:24 jd Exp $
   
   Created 03/16/04 by Jean-Dominique Frattini <zionarea@free.fr>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file support/states/statemachine.h
  \brief StateMachine hoster.
*/

#ifndef LEG_SUPPORT_STATES_STATE_MACHINE_H
#define LEG_SUPPORT_STATES_STATE_MACHINE_H

#include "stateactor.h"
#include "statereactor.h"

namespace leg
{
namespace support
{

//! Sub namespace intended for state manipulations.
/*!
 * States are intended to provide a fair mean to present and manipulate states.
 * It is generic and provide ways for classes to go along with it.
 *
 * status:
 *    in development.
 *
 * info:
 *    states are used in libs::threads and in libs::window only for the moment.
 *    They are intended to be widely used (particularilyy in scene).
 */
namespace states
{

//! Generic, independant state machine.
/*!
 * A state machine is an instanciable class that manipulates a state regarding
 * inputs, and that manipulates a host regarding the state.
 * The host is not a part from this class, so this allows technically to use
 * it for different hosts. However, this should not.
 * A state machine could be interresting for almost all objects that will 
 * 'live' during the progam runs.
 *
 * template arguments description:
 *
 *    Host is the host that will be hanged by states. This could be any class
 *    that requires the use of states for several reasons: inter-object
 *    communication (very basic), best multithread behaving...
 *    
 *    Input is the input type used for 'disturbing' states. States will change
 *    only regarding the received inputs.
 *    
 *    State is the type of the state (this will be generally unsigned int).
 *    
 *    Sync is the thread synchronization type used (that must allow copies).
 *    
 *    ActingPolicy is an implementation of state managing regarding inputs. 
 *    Users will have to provide their own implementation of this policy, so 
 *    states changes are the user's responsability. Leg will provide several
 *    implementations of this policy in several locations (as in libs::threads).
 *    
 *    ReactingPolicy is an implementation of host managing regarding state.
 *    Users will provide their own implementation too.
 *    
 *    StateActor is the state acting interface. This should not be redefined.
 *
 *    StateReactor is the interface of state reacting. This should not be
 *    redefined.
 *
 * status:
 *    experimental.
 *
 * info:
 *    Prefers HostInheritedStateMachine whenever it's possible.
 *
 * @sa StateActor, StateReactor.
 */
template
<
   class Host,
   class Input,
   typename State,
   class Sync,
   template <class,typename,class> class ActingPolicy,
   template <class,typename,class> class ReactingPolicy,
   template <template<class,typename,class> class,class,class,class> class Actor = StateActor,
   template <template<class,typename,class> class,class,class,class> class Reactor = StateReactor
>
class StateMachine:  public Actor<ActingPolicy,Input,State,Sync>,
		     public Reactor<ReactingPolicy,Host,State,Sync>
{
   protected:

   typedef Actor<ActingPolicy,Input,State,Sync>	   ParentActor;
   typedef Reactor<ReactingPolicy,Host,State,Sync> ParentReactor;

   using ParentActor::state;
   using ParentActor::state_mutex;
   
   public:

   //! Default constructor.
   /*!
    * Each state must be initialized with a starting state.
    */
   explicit StateMachine (State s): ParentActor (s),
				    ParentReactor ()
   {
      // this has to be the same mutex than action.
      ParentReactor::SetStateMutex (ParentActor::GetStateMutex());
   }

   ~StateMachine()
   {
   }

   //! React to an eventual state change.
   /*!
    * This state machine design obliged to give the matching host in argument.
    * So, this could be perturbating, because we could pass several different
    * hosts with the same instanciation of this state machine. We should only
    * pass the same host each time.
    * TODO: maybe we compare the host with the first called and refuse to
    * execute the matching code if it's not the same.
    */
   inline void
   React (Host& host)
   {
      ParentReactor::React (host,state);
   }
};

//! Generic, dependant state machine that inherits from the host.
/*!
 * This machine inherits from the host that needs its functionalities. The host
 * class must provides a default constructor (with no arguments) for this to
 * compile, and it's not really advised to redefine this one, because it will
 * then surely be needed to redefine almost all other state classes.
 * This class should be prefered because it knows how to React() without giving
 * any argument to the function. It is also a bit faster and quiete more safe
 * for the state reaction.
 *
 * See StateMachine for more details.
 *
 * status:
 *    experimental.
 *
 * info:
 *    Prefer this one whenever it's possible.
 *
 * @sa StateMachine.
 */
template
<
   class Host,
   class Input,
   typename State,
   class Sync,
   template <class,typename,class> class ActingPolicy,
   template <class,typename,class> class ReactingPolicy,
   template <template<class,typename,class> class,class,class,class> class Actor = StateActor,
   template <template<class,typename,class> class,class,class,class> class Reactor = StateReactor
>
class HostInheritedStateMachine: public Host,
				 public StateMachine<Host,Input,State,Sync,ActingPolicy,ReactingPolicy,Actor,Reactor>
{
   protected:

   typedef StateMachine<Host,Input,State,Sync,ActingPolicy,ReactingPolicy,Actor,Reactor>  ParentMachine;
   typedef Host										  ParentHost;
   typedef typename ParentMachine::ParentActor						  ParentActor;
//   typedef typename ParentMachine::ParentReactor					  ParentReactor;

//   using ParentActor::state;
   
   public:

   explicit HostInheritedStateMachine (State s):ParentHost(),
						ParentMachine (s)
   {
   }

   ~HostInheritedStateMachine()
   {
   }

   //! React to an eventual state change.
   void
   React()
   {
      //ParentReactor::React (*this,state);
      ParentMachine::React (*this);
   }

   private:

//   using ParentMachine::React;
};

}
}
}
#endif
