/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: common.h,v 1.2 2004/03/03 03:50:02 jechk Exp $

   Created 01/26/04 by Jean-Dominique Frattini <zionarea@free.fr>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file support/threads/common.h
  \brief First common needs for support of Posix threads under Leg.
*/

#ifndef LEG_SUPPORT_THREADS_COMMON_H
#define LEG_SUPPORT_THREADS_COMMON_H

#ifndef GNU_SOURCE
   #define GNU_SOURCE
#endif
#include <unistd.h>
#ifndef _POSIX_PRIORITY_SCHEDULING
   #warning "Leg: No Posix real-time scheduling support."
#else
   #include <sched.h>
   #ifdef _POSIX_THREAD_PRIORITY_SCHEDULING
      #include <pthread.h>
      #ifndef LEG_IS_THREADED
	 #define LEG_IS_THREADED 1
      #endif
      #ifndef _REENTRANT
	 #error "Leg: Posix multithread is activated without _REENTRANT was defined !"
      #endif
   #else
      #warning "Leg: No Posix real-time multithread support."
   #endif
#endif

namespace leg
{

namespace support
{

//! First layer for supporting threads.
/*!
 * This provides the Posix threads embeded into classes. Only the linux port of
 * pthreads is supported yet, but pthreads exists under Windows too.
 * All classes, other types and functions (classes or not) tend to have the 
 * same names as in pthreads library in order to be familiar with C pthread
 * programmers. There are only few naming changes like ScheduleAlgorithm.
 *
 * status:
 *    Default behavior is safe.
 *
 * info:
 *    lacking some documentation.
 *    Need to test non-default behaviors.
 *
 * TODO:
 *    New pthread support for Linux 2.6.x kernel and above.
 *    
 *    Windows version of pthreads::cancel_push/pop functions needs to be 
 *    compiled for the matchings C or C++ compilers. Have a look for how the
 *    things work under Linux.
 *
 * @sa leg::libs::threads.
 */
namespace threads
{

typedef pthread_t	   ThreadType;		//!< Type of the thread.

typedef pthread_attr_t	   ThreadAttrType;	//!< Type of the thread attribute.

//! State of cancel of a thread.
/*!
 * Threads can set their cancel state to be enabled or disabled. Under Leg, the
 * default behavior is to set them to uncancellable, and it's not advised to 
 * change this behavior.
 *
 * But threads could more easily stay blocked when cancel is disabled (only the
 * thread will be able to leave itself).
 */
typedef int		   ThreadCancelState;

typedef int		   ThreadCancelType;	//!< Type of cancel.

typedef struct sched_param ScheduleParam;	//!< Type for scheduling parameter.

//! Algorithm of scheduling with the OS kernel.
/*!
 * Please, have a look at the Posix thread and at the Linux scheduling for more
 * information about how threads behave in a system.
 */
enum ScheduleAlgorithm
{
   //! Under this mode, threads run at normal speed.
   /*!
    * This is the default algorithm used for almost all processes and threads.
    * Actually, the X (XFree) server runs out only on this configuration, so
    * it is really dangerous to choose another scheduling algorithm for all
    * X interacting programs.
    */
   normal=		   SCHED_OTHER,

   //! first-in first-out real-time mode (quiete fast).
   /*!
    * This is the first 'real-time' scheduling mode supported by the Posix
    * threads. Under those conditions, the scheduler will often choose threads
    * of this nature before those with normal speed (SCHED_OTHER). And fifo
    * threads obey the law of FIFO regarding the scheduler.
    */
   fifo=		   SCHED_FIFO,

   //! Round Robin is the fastest algorithm for scheduling.
   /*!
    * The OS scheduler choose very often a real-time thread, and prefers the
    * ones with round robin algorithm.
    */
   robin=		   SCHED_RR
};

static const ThreadCancelState	enable	     = PTHREAD_CANCEL_ENABLE;	      //!< enable the cancel state.
static const ThreadCancelState	disable	     = PTHREAD_CANCEL_DISABLE;	      //!< disable the cancel state.

static const ThreadCancelType	deferred     = PTHREAD_CANCEL_DEFERRED;	      //!< cancel type is deferred.
static const ThreadCancelType	asynchronous = PTHREAD_CANCEL_ASYNCHRONOUS;   //!< cancel type is asychronous.

}// namespace threads
}// namespace support
}// namespace leg

#endif
