/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: thread.h,v 1.2 2004/03/03 03:50:02 jechk Exp $
   $Log: thread.h,v $
   Revision 1.2  2004/03/03 03:50:02  jechk
   Changed some names, comments and other things for consistency.

   Revision 1.1  2004/03/03 02:05:22  jechk
   Merged many changes.  See ChangeLog for details.


   Created 01/23/04 by Jean-Dominique Frattini <zionarea@free.fr>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file support/threads/thread.h
  \brief thread support
*/

#ifndef LEG_SUPPORT_THREADS_THREAD_H
#define LEG_SUPPORT_THREADS_THREAD_H

#include "attribute.h"

namespace leg
{

namespace support
{

namespace threads
{

//! Simple interface for pthread support.
/*!
 * This is a simple interface for pthread support without managing any synchronization.
 * This class is designed to be on the top of the threads abstraction hierarchy, so providing
 * polymorphism for all its, yet, unknown childhood.
 *
 * status: 
 *    default configuration is useable.
 */
class Thread
{
   protected:
   
   static unsigned int	count;		  //!< simple thread counter.
   
   ThreadType		thread;		  //!< the thread struct itself.
   
   Attribute		attributes;	  //!< attributes of the thread.
   
   ScheduleParam	sched_params;	  //!< parameterization of the scheduling.

   void			*parameters;	  //!< parameters that could
   
   struct
   {
      int  priority;
      int  min;
      int  max;
   }			priority;	  //!< priority.
   
   ScheduleAlgorithm	schedule_algo;	  //!< own algorithm used for scheduling.

   ThreadCancelState	cancel_state;	  //!< current cancel state.

   ThreadCancelType	cancel_type;	  //!< current cancel type.

   public:
		
   Thread ();

   protected:

   Thread (const Thread&);

   public:

   virtual ~Thread ();

   protected:

   //! It's really not advised to copy threads.
   Thread& operator= (const Thread&);

   public:

   //! Set the thread loop arguments.
   /*!
    * Set the arguments that will be passed to the thread when it will be created.
    * You'll generally give what will be directly needed by this thread, ensuring,
    * if the data are shared between threads, that you correctly synchronize them.
    */
   inline void
   SetParameters (void *p)
   {
      parameters= p;
   }

   //! Get the thread loop arguments.
   /*!
    * Get the parameters that are stored for passing when creating thread.
    */
   inline void*
   GetParameters ()
   {
      return parameters;
   }

   //! running loop.
   /*!
    * When a thread starts, it loops what's implemented in this function.
    * You will generally derived from Thread and overload this pure virtual function.
    * @param arg is anything you want to pass to the function (what you could need).
    */
   virtual void* 
   Go (void* arg)=0;

   //! create and run the thread.
   /*!
    * Create the thread with running it.
    */
   virtual void
   Create ();

   //! get the returing value of the thread, when finished.
   /*!
    * This won't be used generally because of the default behavior of threads that tends 
    * to finish thereselves in good conditions.
    */
   void*
   Join () const;

   protected:

   //! return the thread id.
   /*!
    * return the thread stored value of the created thread.
    */
   inline const ThreadType&
   GetThreadId () const
   {
      return thread;
   }
   
   private:
   
   //! set the thread value.
   /*!
    * Set the thread value, change the thread.
    * It's dangerous to call this if you already run the thread, because you will loose 
    * the thread ownership, and probably could never have it anymore. 
    * That's why it's private.
    */
   inline void
   SetThreadId (ThreadType t)
   {
      thread= t;
   }

   protected:
   
   //! get the thread attributes.
   /*! return a reference to the attributes.
    */
   inline Attribute&
   GetAttributes ()
   {
      return attributes;
   }

   //! set the attributes.
   /*! change the attributes to the new one specified.
    * This is not implemented yet.
    */
   inline void
   SetAttributes (const Attribute& attr)
   {
      //attributes= attr;
   }

   public:

   //! return the current scheduling parameters.
   /*!
    * return the current scheduling parameters.
    * If the thread is not started, nothing really tells you it would use this one.
    */
   inline const ScheduleParam&
   GetSchedParams ()
   {
      attributes.GetScheduleParam (&sched_params);
      return sched_params;
   }

   //! Set the scheduling parameters.
   /*!
    * Set the parameters of the schedule.
    */
   inline void
   SetSchedParams (ScheduleParam& p)
   {
      sched_params= p;
      attributes.SetScheduleParam (&sched_params);
   }

   //! Set the thread priority
   /*! Set the priority of the thread, with
    * checking prior is a suitable value.
    * This function must be called after setScheduleConfig (..)
    */
   void
   SetPriority (int prior);

   //! Return the priority range values.
   /*!
    * Return in min and max arguments the stored priority range
    * last calculated (if calculated).
    */
   inline void
   GetPriorityRange (int& min, int& max)
   {
      min= priority.min;
      max= priority.max;
   }

   //! Get the available priority range values.
   /*!
    * Get the min and max priority values available for the current thread state.
    * This depends mainly on the schedule algorithm used for the thread.
    */
   void
   CalculatePriorityRange ();

   //! Get the current schedule algorithm.
   /*!
    * Get the current algorithm used for scheduling between threads and/or processed.
    */
   inline const ScheduleAlgorithm
   GetScheduleAlgorithm () const
   {
      return schedule_algo;
   }

   //! Set the current schedule algorithm.
   /*!
    * Set the current algorithm to be used for scheduling.
    * The default value is normal for simple no real-time capabilities thread.
    * Please, only use the default value for now, as you could encounter hangs
    * in other ways due to their default high priorities. It's as dangerous as
    * you are running on X.
    */
   virtual void
   SetScheduleConfig (const ScheduleAlgorithm sched_algo= normal);

   protected:

   //! return true if we are currently running in this thread.
   /*!
    * return true if we are currently running on this thread.
    */
   bool
   OwnThread() const;
   
   //! set the cancel state and switch to it
   bool
   SetCancelState (const ThreadCancelState st);

   public:

   //! get the cancel state
   inline const ThreadCancelState
   GetCancelState () const
   {
      return cancel_state;
   }

   private:

   //! cancel a specific thread
   inline int
   CancelThread (const ThreadType th) const
   {
      return pthread_cancel (th);
   }

   /*! cancel our thread
   inline int
   cancelThread () const
   {
      return pthread_cancel (thread);
   }*/

   public:

   //! set the cancel type of the thread. return true if succeed, false otherwise.
   bool
   SetCancelType (const ThreadCancelType ty);

   //! get the cancel type
   inline const ThreadCancelType
   GetCancelType () const
   {
      return cancel_type;
   }

   protected:

   //! test if some thread is (are) waiting for our thread to cancel (get end).
   void
   TestCancel () const;
};

}// namespace threads
}// namespace support
}// namespace leg

#endif
