/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: timing-class.h,v 1.3 2004/05/13 04:16:02 jechk Exp $
   $Log: timing-class.h,v $
   Revision 1.3  2004/05/13 04:16:02  jechk
   Split timing into two parts; moved message to libs.

   Revision 1.2  2004/03/03 03:50:02  jechk
   Changed some names, comments and other things for consistency.

   Revision 1.1  2004/03/03 02:05:22  jechk
   Merged many changes.  See ChangeLog for details.


   Created 2/21/04 by Jeff Binder <bindej@rpi.edu>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file support/timing/timing-class.h
  \brief The Timing class, which manages timers and time units.
*/

#ifndef LEG_SUPPORT_TIMING_TIMINGCLASS_H
#define LEG_SUPPORT_TIMING_TIMINGCLASS_H

namespace leg
{

namespace support
{

namespace timing
{

class Timing
{
public:
  /// Create a new timing system.  units_per_sec is the number of game
  //! time units per second of real time.  current_time is the current
  //! game time in arbitrary units.  
  Timing (double units_per_sec = 1., game_time current_time = 0.);

  //! Set the conversion factor for seconds to game time units.
  void SetUnitsPerSec (double units_per_sec)
  {
    units_per_usecond = units_per_sec * 1000000.;
    useconds_per_unit = 1. / units_per_usecond;
  }

  //! Converts a timeval representing a time interval to game time units.
  game_time TimevalToGameTimeInterval (struct timeval time)
  {
    return (time.tv_sec * 1000000. + time.tv_usec) * units_per_usecond;
  }

  //! Converts a time interval in game time units to timeval.
  timeval GameTimeToTimevalInterval (game_time time)
  {
    struct timeval tv;

    tv.tv_sec = static_cast<time_t> (time * useconds_per_unit / 1000000);
    tv.tv_usec = static_cast<suseconds_t> (fmod (time * useconds_per_unit, 1000000));

    return tv;
  }

  //! Converts a timeval representing an absolute time to game time units.
  game_time TimevalToGameTime (struct timeval time)
  {
    return TimevalToGameTimeInterval(time - epoch_time);
  }

  //! Converts an absolute time in game time units to timeval.
  timeval GameTimeToTimeval (game_time time)
  {
    return GameTimeToTimevalInterval (time) + epoch_time;
  }

  //! Sets the current game time.
  void SetCurrentTime (game_time time);

  //! Returns the current game time.
  game_time GetCurrentTime ()
  {
    struct timeval tv;

    gettimeofday (&tv, NULL);

    return TimevalToGameTime (tv);
  }

private:

  // Conversion factor from microseconds to game time units.
  double units_per_usecond;

  // Conversion factor from game time units to microseconds.
  double useconds_per_unit;

  // The game time at the epoch.
  struct timeval epoch_time;
};

}

}

}

#endif // LEG_SUPPORT_TIMING_TIMINGCLASS_H
