/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: attributes.h,v 1.5 2004/05/09 23:33:35 jd Exp $

   $Log: attributes.h,v $
   Revision 1.5  2004/05/09 23:33:35  jd
   directory support for all the project, best windowing support, best interactivity support...

   Revision 1.4  2004/04/30 01:11:28  jd
   new revision

   Revision 1.3  2004/03/15 23:41:04  jd
   glx 1.2 compatibility first release

   Revision 1.2  2004/03/13 03:02:48  jechk
   Incorrect prototype in attributes.h.

   Revision 1.1  2004/03/12 21:58:28  jd
   Initial revision


 
   Created 03/08/04 by Jean-Dominique Frattini <zionarea@free.fr>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file support/window/attributes.h
  \brief Attributes for gl window rendering support.
*/

#ifndef LEG_SUPPORT_WINDOW_ATTRIBUTES_H
#define LEG_SUPPORT_WINDOW_ATTRIBUTES_H

#include "firsttypes.h"
#include <string>

namespace leg
{
namespace support
{
namespace window
{

//! Windowing attributes support.
/*!
 * This is a generic attributes support for windowing.
 * This class is not intended to be instanciated, use one of its childhood 
 * instead.
 *
 * status: 
 *    in progress.
 *
 * info: 
 *    Please, be careful with int and unsigned int. That could have 
 *    influences on glx 1.2 and 1.3.
 */
class Attributes
{
   protected:

   int		  xpos;		       //!< Window y position regarding the desktop.
   int		  ypos;		       //!< Window y position regarding the desktop.
   unsigned int	  width;	       //!< width of the window.
   unsigned int	  height;	       //!< height of the window.
   int		  color;	       //!< Color buffer in bits of the window.
   int		  depth;	       //!< depth buffer in bits of the window.
   int		  stencil;	       //!< Stencil buffer size in bits.
   int		  red;		       //!< red color buffers sizes.
   int		  green;	       //!< green color buffers sizes.
   int		  blue;		       //!< blue color buffers sizes.
   int		  alpha;	       //!< alpha color buffers sizes.
   std::string	  name;		       //!< Window's name.
   bool		  fullscreen;	       //!< Fullscreen bit.
   bool		  decoration;	       //!< Window window manager's decoration bit.
   bool		  keyboard_autorepeat; //!< is true if keyboard autorepeat is on (default is off on games).

   public:

   //! Default constructor.
   /*!
    * Create and initialize the object.
    * This stores all members to default values.
    */
   Attributes( unsigned int w=800, unsigned int h=600);

   virtual ~Attributes();

   Attributes (const Attributes&);
   
   Attributes&
   operator = (const Attributes&);

   //! Set the attributes for the window.
   /*!
    * Set the attributes for the window as defined by the
    * user's calls to the other functions, or to default.
    * This function must be overriden by the class's childhood
    * in order to effectively store the attributes.
    * @sa XAttributes
    */
   virtual void
   SetAttributes()=0;

   virtual int* const
   GetGLWMAttributes (int &nb)=0;

   //! Set the position of the window.
   /*!
    * Set the window position to the matching members.
    * This does not affect the attributes. You must call
    * SetAttributes() to take effects.
    * This is the same thing for all other functions of this class.
    * @sa SetAttributes
    */
   void
   SetPosition (int x, int y);

   void
   GetPosition (const int& x, const int& y) const;

   //! Set the size of the window.
   /*!
    * Set the window size to the matching members.
    * This does not affect the attributes. You must call
    * SetAttributes() to take effects.
    */
   void
   SetSize (unsigned int w, unsigned int h);

   void
   GetSize (unsigned int& w, unsigned int& h);

   //! Set the color buffer size of the window.
   /*!
    * Set the window color buffer size to the matching members.
    * This does not affect the attributes. You must call
    * SetAttributes() to take effects.
    */
   void
   SetColorBufferSize (int c);

   //! Set the color sizes for the color buffer of the window.
   /*!
    * Set each red, green, blue and alpha size to the matching members.
    * This does not affect the attributes. You must call
    * SetAttributes() to take effects.
    */
   void
   SetColorsSize (int r, int g, int b, int a);

   int
   GetColorBufferSize () const;

   //! Set the depth buffer size of the window.
   /*!
    * Set the window depth buffer size to the matching members.
    * This does not affect the attributes. You must call
    * SetAttributes() to take effects.
    */
   void
   SetDepthBufferSize (int d);

   int
   GetDepthBufferSize () const;

   //! Set the stencil buffer size of the window.
   /*!
    * Set the window size to the matching members.
    * This does not affect the attributes. You must call
    * SetAttributes() to take effects.
    */
   void
   SetStencilBufferSize (int s);

   int
   GetStencilBufferSize () const;

   //! Set the name of the window.
   /*!
    * Set the window's name to the matching members.
    * This does not affect the attributes. You must call
    * SetAttributes() to take effects.
    */
   void
   SetName (std::string& n);

   const std::string&
   GetName () const;

   //! Set it the window will be fullscreen or not.
   /*!
    * Set the window fullscreen attribute to the matching members.
    * This does not affect the attributes. You must call
    * SetAttributes() to take effects.
    */
   void
   SetFullScreenUse (bool f);

   //! Get if we use fullscreen mode.
   bool
   GetFullScreenUse () const;

   //! Set the use of window decoration TODO
   /*!
    * This could be useful when displaying a simple starting 'window-image'.
    */
   void
   SetDecorationUse (bool d);

   //! Get the use of decoration.
   bool
   GetDecorationUse() const;

   //! Set the keyboard auto repeat mode.
   /*!
    * @sa XAttributes::SetKeyboardAutoRepeatMode
    */
   virtual void
   SetKeyboardAutoRepeatMode (internal::Display *disp, bool mode) = 0;

   //! Get the keyboard auto repeat mode.
   bool
   GetKeyboardAutoRepeatMode ()
   {
      return keyboard_autorepeat;
   }
};

}
}
}

#endif
