/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: drawable.h,v 1.8 2004/06/14 15:14:47 jd Exp $

   Created 02/18/04 by Jean-Dominique Frattini <zionarea@free.fr>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file support/window/drawable.h
  \brief base drawable support
*/

#ifndef LEG_SUPPORT_WINDOW_DRAWABLE_H
#define LEG_SUPPORT_WINDOW_DRAWABLE_H

#include "firsttypes.h"

namespace leg
{
namespace support
{
namespace window
{

//! Virtual base class for all window's drawables.
/*!
 * Pure virtual base class for drawables. This class has been created because 
 * the way drawables are made does not allow them to be used as a generic type.
 *
 * status:
 *    works.
 *
 * info:
 *    none.
 *
 * @sa GLXDrawable, Window.
 */
class BaseDrawable
{
   protected:

   BaseDrawable()
   {
   }

   BaseDrawable (const BaseDrawable&);
   
   public:

   virtual ~BaseDrawable()
   {
   }

   //! Set the display connection.
   /*!
    * Set an existed display connection.
    */
   virtual void
   SetDisplay (internal::Display *disp)= 0;
   
   virtual internal::Display* const
   GetDisplay() = 0;

   //! Create a gl wm context.
   /*!
    * Create an OpenGL window manager's context.
    * Please look at the implementation doc about this function for more 
    * details about how it is created and if you could parameterize that.
    */
   virtual internal::Context
   CreateContext(internal::Context shared=0) = 0;

   //! Destroy a gl wm context.
   /*!
    * Destroy an OpenGL window manager's context.
    */
   virtual void
   DestroyContext(internal::Context c) = 0;

   //! Enter the context.
   /*!
    * Make the context active.
    */
   virtual void
   EnterContext (internal::Context c) = 0;

   //! Leave the current context.
   //! status: 
   //!	 not implemented because there's nothing letting us doing this job.
   virtual void
   LeaveContext() = 0;

   //! Get the current context.
   /*!
    * Return the current context or 0 if not.
    */
   virtual internal::Context
   GetCurrentContext() = 0;

   //! Create the gl wm window.
   /*! 
    * Create the OpenGL window manager's window.
    */
   virtual void
   CreateDrawable (int *glx_attr) = 0;

   //! Destroy the glx drawable.
   /*!
    * Destroy the glx drawable.
    */
   virtual void
   DestroyDrawable() = 0;

   //! Return the firsttype window.
   /*!
    * Get the internal window.
    * Users are not invited to use this function.
    */
   virtual internal::Window&
   GetWindow() = 0;

   //! Swap the gl wm buffers.
   /*!
    * Swap the gl buffers.
    */
   virtual void
   SwapBuffers() = 0;

   //! Wait for the window manager.
   /*!
    * Wait for the window manager to finish.
    */
   virtual void
   WaitWM() = 0;

   //! Wait for OpenGL.
   /*!
    * Wait for OpenGL to finish its work.
    */
   virtual void
   WaitGL() = 0;

   //! Return the drawable.
   /*!
    * Users are not invited to use this function.
    */
   virtual const internal::Drawable&
   GetDrawable() = 0;

   //! Get a pointer to the visual.
   /*!
    * Return a pointer to the visual. This function is not intended to user's 
    * purposes.
    *
    * info:
    *	 the arguments are only needed for glx 1.2.
    */
   virtual internal::VisualInfo* const
   GetVisual (int scr, int *glx_attr) = 0;

   //! Checks if direct rendering is supported threw the created window.
   /*!
    * Returns true if direct rendering is supported threw the created
    * window or false if not.
    */
   virtual bool
   CheckDirectRendering (internal::Display *d, internal::Context c) = 0;

   private:
   
   //! Operator has a disabled polymorphism behavior and is not implemented, so
   //! don't use it, neither don't try to overload it. Overriden is also
   //! forbidden because the common use of drawable is now polymorphism.
   BaseDrawable&
   operator= (const BaseDrawable&);
};

//! Rendering drawable.
/*!
 * Need to be used in a Window. Allow then gl renderings into it. Even if it's 
 * designed to be generic and portable, this does not ensure that any specific 
 * windowing could be supported. This is why there's only this specialization 
 * available.
 *
 * status:
 *    testing.
 *
 * info: 
 *    This class does not allow instance to be copied or assigned.
 * 
 * @sa internal::DrawableImpl for implementation details.
 */
template
<
   class Impl
>
class Drawable: public BaseDrawable
{
   Impl *const pimpl;

   public:
   
   //! Default constructor.
   /*!
    * Construct and open a connection to the display.
    */
   Drawable(): BaseDrawable(),
	       pimpl (new Impl)
   {
   }
   
   private:

   Drawable (const Drawable&);

   public:

   ~Drawable()
   {
      delete pimpl;
   }

   //! Set the display connection.
   /*!
    * Set an existed display connection.
    */
   inline void
   SetDisplay (internal::Display *disp)
   {
      pimpl->SetDisplay (disp);
   }
   
   inline internal::Display* const
   GetDisplay()
   {
      return pimpl->GetDisplay();
   }

   //! Create a gl wm context.
   /*!
    * Create an OpenGL window manager's context.
    * Please look at the implementation doc about this function for more 
    * details about how it is created and if you could parameterize that.
    */
   inline internal::Context
   CreateContext(internal::Context shared=0)
   {
      return pimpl->CreateContext(shared);
   }

   //! Destroy a gl wm context.
   /*!
    * Destroy an OpenGL window manager's context.
    */
   inline void
   DestroyContext(internal::Context c)
   {
      pimpl->DestroyContext(c);
   }

   //! Enter the context.
   /*!
    * Make the context active.
    */
   inline void
   EnterContext (internal::Context c)
   {
      pimpl->EnterContext(c);
   }

   //! Leave the current context.
   //! status: 
   //!	 not implemented because there's nothing letting us doing this job.
   inline void
   LeaveContext()
   {
      pimpl->LeaveContext();
   }

   //! Get the current context.
   /*!
    * Return the current context or 0 if not.
    */
   inline internal::Context
   GetCurrentContext()
   {
      return pimpl->GetCurrentContext();
   }

   //! Create the gl wm window.
   /*! 
    * Create the OpenGL window manager's window.
    */
   inline void
   CreateDrawable (int *glx_attr)
   {
      pimpl->CreateDrawable(glx_attr);
   }

   //! Destroy the glx drawable.
   /*!
    * Destroy the glx drawable.
    */
   inline void
   DestroyDrawable()
   {
      pimpl->DestroyDrawable();
   }

   //! Return the firsttype window.
   /*!
    * Get the internal window.
    * Users are not invited to use this function.
    */
   inline internal::Window&
   GetWindow()
   {
      return pimpl->GetWindow();
   }

   //! Swap the gl wm buffers.
   /*!
    * Swap the gl buffers.
    */
   inline void
   SwapBuffers()
   {
      pimpl->SwapBuffers();
   }

   //! Wait for the window manager.
   /*!
    * Wait for the window manager to finish.
    */
   inline void
   WaitWM()
   {
      pimpl->WaitWM();
   }

   //! Wait for OpenGL.
   /*!
    * Wait for OpenGL to finish its work.
    */
   inline void
   WaitGL()
   {
      pimpl->WaitGL();
   }

   //! Return the drawable.
   /*!
    * Users are not invited to use this function.
    */
   inline const internal::Drawable&
   GetDrawable()
   {
      return pimpl->GetDrawable();
   }

   //! Get a pointer to the visual.
   /*!
    * Return a pointer to the visual.
    * This function is not intended to user's purposes.
    * info:
    *	 the arguments are only needed for glx 1.2.
    */
   inline internal::VisualInfo* const
   GetVisual (int scr, int *glx_attr)
   {
      return pimpl->GetVisual (scr, glx_attr);
   }

   //! Checks if direct rendering is supported threw the created window.
   /*!
    * Returns true if direct rendering is supported threw the created
    * window or false if not.
    */
   inline bool
   CheckDirectRendering (internal::Display *d, internal::Context c)
   {
      return pimpl->CheckDirectRendering (d,c);
   }

   private:
   
   Drawable&
   operator= (const Drawable&);
};

}
}
}
#endif
