/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: event.h,v 1.6 2004/06/19 11:13:02 jd Exp $

   Created 04/15/04 by Jean-Dominique Frattini <zionarea@free.fr>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file support/window/event.h
  \brief base for event handlings.
*/

#ifndef LEG_SUPPORT_WINDOW_EVENT_H
#define LEG_SUPPORT_WINDOW_EVENT_H

#include "firsttypes.h"
#include <string>
#include <map>
#include <queue>

namespace leg
{
namespace support
{
namespace window
{

//! Window-focused keyboard keys.
typedef internal::Key		 Key;

//! Mouse button type.
typedef internal::MouseButton	 MouseButton;

struct UserEvent
{
   enum
   {
      none,
      key_press,
      key_release,
      button_press,
      button_release,
      mouse_motion,
      button_motion
   } type;

   union
   {
      Key key;
      
      union
      {
	 typedef struct
	 {
	    unsigned int x;
	    unsigned int y;
	 } Coord;
	 
	 MouseButton button;
	 Coord	     coord;
      } mouse;
   } event;
};

enum MouseOrigin
{
   upper_left_corner,
   lower_left_corner
};

enum MouseAreaConfig
{
   empty = 0,
   full,
   partial
};

//! Base class for handlings user events.
/*!
 * Base polymorphic class for user events.
 *
 * status:
 *    second version: fastest.
 *
 * infos:
 *    all interface has changed !
 *
 * @sa Window, Xevent
 */
class BaseEvent
{ 
   protected:

   typedef std::queue<UserEvent> EventQueue;

   MouseAreaConfig mouse_area_config;

   struct
   {
      unsigned int start_x;
      unsigned int start_y;
      unsigned int size_x;
      unsigned int size_y;
   }		  mouse_area;

   public:
      
   typedef internal::Event UEvent;

   protected:

   EventQueue	  event_queue;
   UEvent	  event;	       //!< User useable event.
   Key		  current_key;
   std::string	  current_string_key;  //!< Current string entered with the keyboard.
   unsigned int	  current_mouse_button;//!< Current mouse button.
   MouseOrigin	  mouse_origin;	       //!< Where mouse coordinates start (often upper_left_corner).
   MouseOrigin	  gl_mouse_origin;     //!< Where OpenGL set the mouse origin (often lower_left_corner).
   unsigned int	  win_width;	       //!< Window width.
   unsigned int	  win_height;	       //!< window height.
   bool		  save_string_key;     //!< True if we save current_string_key, or false if not updated.
   
   BaseEvent();
   
   public:
   
   virtual ~BaseEvent();

   //! Do the events receive from the event manager (generally the window).
   /*!
    * Actualize the events management of the current window. If no events have
    * occured between two calls, the function immediately returns (0).
    */
   virtual void
   Update() = 0;

   UserEvent *const
   GetLastEvent()
   {
      if (event_queue.empty())
	 return 0;
      return &event_queue.back();
   }

   void
   SetMouseAreaConfig (MouseAreaConfig cfg)
   {
      mouse_area_config = cfg;
   }

   void
   SetMouseArea (unsigned int sx, unsigned int sy, unsigned int w, unsigned int h)
   {
      mouse_area.start_x = sx;
      mouse_area.start_y = sy;
      mouse_area.size_x = w;
      mouse_area.size_y = h;
   }
};

}
}
}

#endif
