/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: glxdrawableimpl.h,v 1.1 2004/04/30 01:28:51 jd Exp $

   Created 02/18/04 by Jean-Dominique Frattini <zionarea@free.fr>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file support/window/glx/glxdrawableimpl.h
  \brief glx virtual base class drawable support
*/

#ifndef LEG_SUPPORT_WINDOW_GLX_GLX_DRAWABLE_IMPL_H
#define LEG_SUPPORT_WINDOW_GLX_GLX_DRAWABLE_IMPL_H

#include "../firsttypes.h"

namespace leg
{
namespace support
{
namespace window
{

namespace internal
{

//! rendering drawable impl.
/*!
 * os-dependant implementation of drawable.
 * This is the default for GNU systems.
 *
 * @sa Drawable, GLX12DrawableImpl, GLX13DrawableImpl
 *
 * status: 
 *    working.
 *
 * info: 
 *    The functions tries to be named seemly like for X and glX.
 *    There are some exceptions designed in their comments.
 *    Please, refer to glX 1.3 and XFree 4 lib documentation for more details.
 */
class GLXDrawableImpl
{
   protected:

   internal::Display	*display;	  //!< X display.
   
   internal::Window	window;		  //!< glx window.
   
   internal::Drawable	drawable;	  //!< glx drawable.
   
   //internal::Drawable	back_drawable;	  //!< glx back drawable (needed for pbuffers).

   //internal::Context	context;	  //!< glx context.

   internal::VisualInfo		*visual_info;
   
   public:

   GLXDrawableImpl();

   protected:

   GLXDrawableImpl (const GLXDrawableImpl&);

   public:
   
   virtual ~GLXDrawableImpl();

   inline void
   SetDisplay (internal::Display *disp)
   {
      display=disp;
   }
   
   inline internal::Display* const
   GetDisplay()
   {
      return display;
   }

   //! Creates a context.
   /*!
    * Create a context and allow shared to be a shared
    * context for the contextx memory.
    * Please, refer to glXCreateContextNew() for more details.
    */
   virtual internal::Context
   CreateContext (internal::Context shared)=0;

   //! Destroy the context given in argument.
   virtual void
   DestroyContext (internal::Context c)=0;

   //! Enter the context.
   /*!
    * Enter the context for enabling rendering possibilities.
    * This calls glXMakeContextCurrent().
    */
   virtual void
   EnterContext (internal::Context c)= 0;

   virtual void
   LeaveContext (internal::Context c=0)=0;

   //! Return the current context.
   virtual internal::Context
   GetCurrentContext ()=0;

   //! Create the glx drawable.
   /*!
    * Create the glx drawable that could have rendering
    * commands into it (as gl functions).
    */
   virtual void
   CreateDrawable (int *glx_attr)=0;

   //! Destroys the glx drawable.
   virtual void
   DestroyDrawable ()=0;

   //! Swap the rendering buffers.
   virtual void
   SwapBuffers()=0;
 
   //! Wait for X.
   inline void
   WaitWM()
   {
      glXWaitX();
   }

   //! Wait for GL.
   inline void
   WaitGL()
   {
      glXWaitGL();
   }

   inline const internal::Drawable&
   GetDrawable()
   {
      return drawable;
   }

   //! get the visual info
   /*!
    * @arg int scr, int *glx_attr are needed only for glx 1.2
    * backward compatibility. So, if your system is glx 1.3 or above,
    * you could passe anything to them, they won't be cared.
    */
   virtual internal::VisualInfo* const
   GetVisual (int scr, int *glx_attr)=0;
   
   //! R/W window returns.
   inline internal::Window& 
   GetWindow()
   {
      return window;
   }

   bool
   CheckDirectRendering (internal::Display *d, internal::Context c);

   protected:
   
   GLXDrawableImpl& operator= (const GLXDrawableImpl&);
};

}// namespace internal
}
}
}
#endif
