/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: $

   Created 07/19/04 by Jean-Dominique Frattini <zionarea@free.fr>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file support/window/glx/glxtext.h
  \brief X glx base class for gl text outputs.
*/

#ifndef LEG_SUPPORT_WINDOW_SDL_SDL_TEXT_H
#define LEG_SUPPORT_WINDOW_SDL_SDL_TEXT_H

#include "../firsttypes.h"
#include <SDL/SDL.h>
#include <GL/gl.h>
#include <string>
#include <list>

#ifndef WIN32
#include <GL/glx.h>
#endif

namespace leg
{
namespace support
{
namespace window
{

//! On-Screen Display text management output on a Hud.
/*!
 * Text management and output on the window (On-Screen Display on a Hud). This
 * class allows you to load a font, to display strings on a specific location
 * at the screen and some other manipulations to make things easier.
 * 
 * status:
 *    useable.
 *
 * info:
 *    This class should never be instanciated.
 *    Nothing tells other (future) ports of this class will provide the same
 *    functionalities; but they are greatly encouraged to follow the same main
 *    interface as much as possible.
 */
#ifndef WIN32
class SdlText
{
   protected:

   //! font name and X font pair.
   typedef std::pair<std::string,XFontStruct*>	FP1;

   //! OpenGL font marquers.
   typedef std::pair<GLuint,GLuint>		FP2;

   //! Font-display list equivalence.
   /*!
    * FontPair is used to store all the needed remaining informations about a
    * loaded font. This stores the font name, the X font structure and the 
    * OpenGL information for it.
    */
   typedef std::pair<FP1*,FP2*>			FontPair;
   
   static internal::Display	    *disp;		 //!< XFree display connection.
   static GLuint		    list_start_id;	 //!< id start list from the actual loaded font.
   static std::list<FontPair*>	    loaded_fonts;	 //!< list of loaded fonts and their matching id.
   static std::list<std::string*>   patterned_fonts;	 //!< available patterned fonts.
   static unsigned int		    max_elements;	 //!< maximum matched fonts retrieved and saved into patterned_fonts.

   SdlText();

   ~SdlText();

   public:

   //! Font iterator to get threw patterned_fonts.
   /*!
    * When CheckFontsAvailability() as been invoked and succesfully executed,
    * users will be able to manipulate the available matching found fonts
    * threw a FontIterator that will be return by a call to GetFontIterator().
    */
   typedef std::list<std::string*>::const_iterator FontIterator;

   //! Connect to the display.
   /*!
    * You must be connected to the display before being available to use all 
    * the other functions of this class. Simply giving a pointer to a valid
    * connection does the work.
    */
   static void
   Connect (internal::Display *d = 0);

   //! Load a font and select it as the current.
   /*!
    * Load a font from the name given in argument and select it as the current
    * font for writting into the window with Print(). Selects the font as the
    * current if the font has already been loaded. If no argument has been 
    * given, then, the "fixed" font is used as the default.
    */
   static void
   LoadFont (std::string font_name = "fixed");

   //! Load the matched fonts saved into patterned_fonts.
   /*!
    * Use CheckFontsAvailability() before in order to fill the list. This will
    * load all the fonts in the patterned_fonts list, and select the last one
    * as the current font for printing.
    */
   static void
   LoadMatchedFonts();

   //! Unload a font, giving its string name.
   /*!
    * Unload the font having the same name as given in argument. This let the
    * state of the current font undefined if the font was current just before
    * calling this function.
    */
   static void
   UnloadFont (const std::string& font_name);

   //! Select an already loaded font as the current font.
   /*!
    * Does nothing if the font has not been loaded.
    */
   static void
   SelectFont (const std::string& font_name);

//   static void
//   SelectFont (FontIterator font_iterator);
   
   //! Locate into the screen (in order to print after generally).
   /*!
    * Note: 
    *	 Generally, the origin is located on the bottom-left corner, so you may
    *	 want to put it on the upper-left corner and ordinate from up to down.
    */
   static void
   Locate (int x, int y, int z = 0);

   //! Locate into the screen (in order to print) with float values.
   static void
   Locate (float x, float y, float z = .0);

   //! Locate into the screen (in order to print) with double values.
   static void
   Locate (double x, double y, double z = .0);

   //! Check fonts availability from a given pattern.
   /*!
    * This fills up the patterned_fonts list with available fonts that match 
    * the given pattern (you can use '*' and '?' to makes some unknowns). Not
    * all the matchings fonts will be retrived, but only max_elements at the
    * maximum. You can get and change how many with using the appropriate 
    * functions declared below.
    * 
    * Please have a look at the matching X documentation for more information.
    */
   static bool
   CheckFontsAvailability (const std::string& pattern);

   //! Get a FontIterator at the begining of patterned_fonts list.
   /*!
    * This can be usefull to go threw the patterned_fonts list.
    */
   static FontIterator
   GetFontIterator();

   //! Get the size of the patterned_fonts list.
   /*!
    * Return how much elements are stored into the patterned_fonts list.
    */
   static unsigned int
   GetPatternedFontsSize();

   //! Print a string at the previous providen location with Locate().
   /*!
    * Prints out the string on the window (gl 2D projection matrix). If the
    * location hasn't been defined, then where the string will be print out on
    * the screen is undefined.
    *
    * @sa Locate().
    */
   static void
   Print (const std::string& str);

   //! Set how many max elements could be retrived.
   /*!
    * Set how much maximum elements could be retrieved while checking the fonts
    * availability regarding a pattern (CheckFontsAvailability()).
    *
    * @sa max_elements, CheckFontsAvailability.
    */
   static void
   SetMaxElementsInPatternedFonts (unsigned int max);

   //! Get how many max elements could be retrieved.
   /*!
    * Return how many maximum elements will be retrived while making a research
    * of fonts regarding a pattern.
    *
    * @sa GetMaxElementsInPatternedFonts().
    */
   static unsigned int
   GetMaxElementsInPatternedFonts();

   protected:
   
   //! Returns true if the font has already been loaded and not deleted.
   static bool
   FontAlreadyLoaded (const std::string& str);

   //! Delete all the loaded list.
   /*!
    * Delete all loaded fonts of X, Glx and free the list. There will have no 
    * more fonts available after this call, so SelectFont() and Print() will 
    * become undefined.
    */
   static void
   DeleteList();
};

//! Osd is the type available for users, and any ports should typedef into this
//! type too. Osd may be a temporary solution to print out some info until a
//! console class arises.
typedef SdlText Osd;

#endif

}
}
}
#endif
