/* This file is part of GNU Libraries and Engines for Games  -*- c++ -*-

   $Id: win.h,v 1.8 2004/06/15 12:56:13 jd Exp $

   $Log: win.h,v $
   Revision 1.8  2004/06/15 12:56:13  jd
   events separation

   Revision 1.7  2004/06/14 15:14:47  jd
   no more drawing neither resize in windowing

   Revision 1.6  2004/05/28 14:08:38  jd
   correct some little errors

   Revision 1.5  2004/05/21 23:15:01  jd
   correction for libs/window

   Revision 1.4  2004/05/01 00:04:38  jd
   initial support for fullscreen

   Revision 1.3  2004/04/30 01:13:02  jd
   new revision

   Revision 1.2  2004/03/22 03:22:54  jd
   glx 1.2 support.

   Revision 1.1  2004/03/12 22:02:48  jd
   Initial revision


 
   Created 03/08/04 by Jean-Dominique Frattini <zionarea@free.fr>
   
   Copyright (c) 2004 Free Software Foundation
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/
/*! \file support/window/win.h
  \brief Pure virtual base class for windowing support.
*/

#ifndef LEG_SUPPORT_WINDOW_WIN_H
#define LEG_SUPPORT_WINDOW_WIN_H

#include "firsttypes.h"
#include "displayconnect.h"
#include "attributes.h"
#include "event.h"
#include "drawable.h"
#include <string>

namespace leg
{
namespace support
{
namespace window
{

//! Main class of the windowing hierarchy.
/*!
 * This is the first base class of the window classes hierarchy.
 * At this stage, window is 'windowless' as it has no hardware connection.
 * This will generally be the main pointer type of a dynamically allocated
 * object, so using polymorphism. However, this layer does not provide extra
 * tools for managing window (like thread synchronization or state machine), so
 * final users will probably prefer using the above layer (libs::window) that
 * provides many orientations for handling window in a game.
 * Window doesn't manage functions like resizing, drawing, or doing the events
 * which are all left to specific classes, whether in the same namespace (that
 * is the case of events) or in graphics (drawing) or elsewhere.
 *
 * Please have a look at the window type matching your configuration in order
 * to know what is implemented yet, and how.
 *
 * status:
 *    Interface fits well with Glx.
 *    This class is almost pure virtual, but should be pure virtual !
 *
 * info:
 *    none.
 *
 * @sa AbstractWindow, GLXWindow, libs::window.
 */
class Window
{
   protected:
   
   bool		     direct_rendered;  //!< direct render enabled/disabled.
   internal::Context context;	       //!< initial context of the window.
   DisplayConnect&   display;	       //!< Display connection.
   void	 *	     drawable_ptr;     //!< Generic auto-allocated and auto-typed drawable.
   bool		     is_created;
   
   //! Only constructor for Window and its childhood.
   /*!
    * At least under X (and XFree), a window has to be connected to an opened
    * display in order to be valid. However, under Windows, this is not obliged
    * just because Windows in single user oriented (and doesn't allow displays
    * to be connected threw a network). In this case, Windows window child 
    * class will have to fake a DisplayConnect (generally with a doing-nothing
    * class).
    */
   Window (DisplayConnect& disp);

   public:

   virtual ~Window();

   //! Create the window.
   /*!
    * Creating the window is the step that will create the window on the screen,
    * and allows it to be drawable by OpenGL commands. Mostly important, several
    * OpenGL commands won't be able to be called without making an error while
    * the window is not created; other commands only require that the display
    * connection is made (only under X).
    */
   virtual void 
   Create()= 0;

   //! Destroy the window.
   virtual void 
   Destroy() = 0;

   virtual BaseDrawable&
   GetDrawable() = 0;

   // this function should be pure virtual (=0)
   virtual internal::Display*
   GetDisplay()
   {
      return display.Get();
   }

   virtual Attributes&
   GetAttributes() = 0;

   virtual void
   ShowCursor (bool) = 0;

   inline bool
   IsCreated()
   {
      return is_created;
   }

   protected:

   //! Change to fullscreen mode.
   virtual void 
   SwitchToFullScreen()= 0;

   //! Change to window mode (from fullscreen).
   virtual void 
   SwitchToWindow()= 0;
};

}
}
}
#endif
